from typing import Any, Dict, List, Optional, Union
from abc import ABC, abstractmethod
import logging
import os
from pathlib import Path
import pandas as pd
import sqlalchemy
from sqlalchemy import create_engine, text
from urllib.parse import quote_plus
from utils import *

# 로깅 설정
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

class UserToolBase(ABC):
    """사용자 정의 도구의 기본 클래스
    
    모든 사용자 정의 도구는 이 클래스를 상속받아야 합니다.
    """
    
    def __init__(self, name: str, description: str):
        """
        Args:
            name: 도구의 이름
            description: 도구의 설명
        """
        self.name = name
        self.description = description
        
    @abstractmethod
    def execute(self, *args, **kwargs) -> Any:
        """도구의 실행 로직을 구현하는 메서드
        
        Returns:
            실행 결과
        """
        pass
    
    def validate_params(self, params: Dict[str, Any]) -> bool:
        """파라미터 유효성 검사
        
        Args:
            params: 검사할 파라미터 딕셔너리
            
        Returns:
            유효성 검사 통과 여부
        """
        return True
    
    def log_execution(self, params: Dict[str, Any], result: Any) -> None:
        """도구 실행 로깅
        
        Args:
            params: 실행 파라미터
            result: 실행 결과
        """
        logger.info(f"도구 '{self.name}' 실행")
        logger.info(f"파라미터: {params}")
        logger.info(f"결과: {result}")

# 예시 도구 클래스
class ExampleTool(UserToolBase):
    """예시 도구 클래스
    
    이 클래스는 사용자 정의 도구 구현의 예시를 보여줍니다.
    """
    
    def __init__(self):
        super().__init__(
            name="example_tool",
            description="예시 도구입니다. 이것은 사용자 정의 도구 구현 방법을 보여주기 위한 것입니다."
        )
        
    def execute(self, text: str) -> str:
        """예시 도구 실행
        
        Args:
            text: 처리할 텍스트
            
        Returns:
            처리된 텍스트
        """
        # 파라미터 검증
        params = {"text": text}
        if not self.validate_params(params):
            raise ValueError("잘못된 파라미터입니다.")
            
        try:
            # 실행 로직
            result = text.upper()
            
            # 실행 로깅
            self.log_execution(params, result)
            
            return result
            
        except Exception as e:
            logger.error(f"도구 실행 중 오류 발생: {str(e)}")
            raise

class DBQueryTool(UserToolBase):
    """데이터베이스 쿼리 실행 및 결과 저장 도구
    
    이 도구는 데이터베이스에 쿼리를 실행하고 결과를 CSV 파일로 저장합니다.
    """
    
    def __init__(self):
        super().__init__(
            name="db_query_tool",
            description="데이터베이스 쿼리를 실행하고 결과를 CSV 파일로 저장하는 도구"
        )
        self.engine = None
        
    def connect(self, db_type: str, host: str, port: int, database: str, 
               username: str, password: str) -> bool:
        """데이터베이스 연결
        
        Args:
            db_type: 데이터베이스 종류 (mysql, postgresql, oracle 등)
            host: 데이터베이스 호스트
            port: 데이터베이스 포트
            database: 데이터베이스 이름
            username: 사용자 이름
            password: 비밀번호
            
        Returns:
            연결 성공 여부
        """
        try:
            # URL 인코딩된 비밀번호 생성
            encoded_password = quote_plus(password)
            
            # 데이터베이스 URL 생성
            if db_type == 'mysql':
                url = f'mysql+pymysql://{username}:{encoded_password}@{host}:{port}/{database}'
            elif db_type == 'postgresql':
                url = f'postgresql://{username}:{encoded_password}@{host}:{port}/{database}'
            elif db_type == 'oracle':
                url = f'oracle+cx_oracle://{username}:{encoded_password}@{host}:{port}/{database}'
            else:
                raise ValueError(f"지원하지 않는 데이터베이스 종류: {db_type}")
            
            # 엔진 생성 및 연결 테스트
            self.engine = create_engine(url)
            self.engine.connect()
            logger.info(f"데이터베이스 연결 성공: {host}:{port}/{database}")
            return True
            
        except Exception as e:
            logger.error(f"데이터베이스 연결 실패: {str(e)}")
            self.engine = None
            return False
    
    def execute(self, query: str, output_path: str, params: Optional[Dict[str, Any]] = None) -> bool:
        """쿼리 실행 및 결과 저장
        
        Args:
            query: SQL 쿼리문
            output_path: 결과를 저장할 CSV 파일 경로
            params: 쿼리 파라미터 (선택사항)
            
        Returns:
            실행 성공 여부
        """
        if not self.engine:
            raise RuntimeError("데이터베이스 연결이 필요합니다. connect() 메서드를 먼저 호출하세요.")
            
        try:
            # 쿼리 실행
            with self.engine.connect() as conn:
                if params:
                    result = conn.execute(text(query), params)
                else:
                    result = conn.execute(text(query))
                
                # 결과를 DataFrame으로 변환
                df = pd.DataFrame(result.fetchall(), columns=result.keys())
                
                # CSV 파일로 저장
                output_path = normalize_path(output_path)
                os.makedirs(os.path.dirname(output_path), exist_ok=True)
                df.to_csv(output_path, index=False, encoding='utf-8-sig')
                
                logger.info(f"쿼리 결과 저장 완료: {output_path} (총 {len(df)}행)")
                return True
                
        except Exception as e:
            logger.error(f"쿼리 실행 또는 저장 실패: {str(e)}")
            return False
            
    def close(self) -> None:
        """데이터베이스 연결 종료"""
        if self.engine:
            self.engine.dispose()
            self.engine = None
            logger.info("데이터베이스 연결 종료")
