#!/bin/bash

# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# 스크립트가 위치한 디렉토리로 변경
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR"

# Docker Compose가 설치되어 있는지 확인
if ! command -v docker-compose &> /dev/null; then
    echo -e "${RED}Docker Compose가 설치되어 있지 않습니다.${NC}"
    echo -e "${YELLOW}먼저 Docker Compose를 설치하세요:${NC}"
    echo "apt-get update && apt-get install -y docker-compose"
    exit 1
fi

# 사용법 출력
usage() {
    echo -e "${GREEN}PostgreSQL Docker 컨테이너 관리 스크립트${NC}"
    echo "사용법: $0 [명령]"
    echo ""
    echo "명령:"
    echo "  start     - PostgreSQL 컨테이너 시작"
    echo "  stop      - PostgreSQL 컨테이너 중지"
    echo "  restart   - PostgreSQL 컨테이너 재시작"
    echo "  status    - PostgreSQL 컨테이너 상태 확인"
    echo "  logs      - PostgreSQL 컨테이너 로그 확인"
    echo "  shell     - PostgreSQL 컨테이너 쉘 접속"
    echo "  psql      - PostgreSQL 데이터베이스 접속"
    echo "  backup    - 데이터베이스 백업 생성"
    echo "  restore   - 데이터베이스 복원"
    echo "  reset     - 컨테이너 제거 후 데이터와 함께 재시작"
    echo ""
}

# 환경 변수 파일 로드
load_env() {
    if [ -f .env ]; then
        export $(grep -v '^#' .env | xargs)
    else
        echo -e "${RED}.env 파일을 찾을 수 없습니다.${NC}"
        exit 1
    fi
}

# PostgreSQL 컨테이너 시작
start_postgres() {
    echo -e "${YELLOW}PostgreSQL 컨테이너를 시작합니다...${NC}"
    docker-compose up -d
    echo -e "${GREEN}PostgreSQL 컨테이너가 시작되었습니다.${NC}"
}

# PostgreSQL 컨테이너 중지
stop_postgres() {
    echo -e "${YELLOW}PostgreSQL 컨테이너를 중지합니다...${NC}"
    docker-compose stop
    echo -e "${GREEN}PostgreSQL 컨테이너가 중지되었습니다.${NC}"
}

# PostgreSQL 컨테이너 상태 확인
check_status() {
    echo -e "${YELLOW}PostgreSQL 컨테이너 상태:${NC}"
    docker-compose ps
    
    # 포트 리스닝 확인
    if netstat -tuln | grep -q ":5432 "; then
        echo -e "${GREEN}PostgreSQL 서버가 포트 5432에서 실행 중입니다.${NC}"
    else
        echo -e "${RED}PostgreSQL 서버가 포트 5432에서 실행되고 있지 않습니다.${NC}"
    fi
    
    # 컨테이너 내부 PostgreSQL 상태 확인
    if docker-compose exec postgres pg_isready -U $PG_USER -d $PG_DATABASE > /dev/null 2>&1; then
        echo -e "${GREEN}PostgreSQL 서버에 접속 가능합니다.${NC}"
    else
        echo -e "${RED}PostgreSQL 서버에 접속할 수 없습니다.${NC}"
    fi
}

# PostgreSQL 컨테이너 로그 확인
view_logs() {
    echo -e "${YELLOW}PostgreSQL 컨테이너 로그:${NC}"
    docker-compose logs -f postgres
}

# PostgreSQL 컨테이너 쉘 접속
enter_shell() {
    echo -e "${YELLOW}PostgreSQL 컨테이너 쉘에 접속합니다...${NC}"
    docker-compose exec postgres bash
}

# PostgreSQL 데이터베이스 접속
connect_psql() {
    echo -e "${YELLOW}PostgreSQL 데이터베이스에 접속합니다...${NC}"
    docker-compose exec postgres psql -U $PG_USER -d $PG_DATABASE
}

# 데이터베이스 백업
backup_database() {
    BACKUP_DIR="./backups"
    mkdir -p $BACKUP_DIR
    TIMESTAMP=$(date +%Y%m%d_%H%M%S)
    BACKUP_FILE="$BACKUP_DIR/backup_$PG_DATABASE-$TIMESTAMP.sql"
    
    echo -e "${YELLOW}데이터베이스 $PG_DATABASE를 백업합니다...${NC}"
    docker-compose exec -T postgres pg_dump -U $PG_USER -d $PG_DATABASE > $BACKUP_FILE
    
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}백업이 성공적으로 생성되었습니다: $BACKUP_FILE${NC}"
    else
        echo -e "${RED}백업 생성에 실패했습니다.${NC}"
        exit 1
    fi
}

# 데이터베이스 복원
restore_database() {
    if [ -z "$1" ]; then
        echo -e "${RED}복원할 백업 파일을 지정해야 합니다.${NC}"
        echo -e "${YELLOW}사용법: $0 restore <백업 파일명>${NC}"
        exit 1
    fi
    
    BACKUP_FILE="$1"
    
    if [ ! -f "$BACKUP_FILE" ]; then
        echo -e "${RED}백업 파일 $BACKUP_FILE을 찾을 수 없습니다.${NC}"
        exit 1
    fi
    
    echo -e "${YELLOW}데이터베이스 $PG_DATABASE를 복원합니다...${NC}"
    cat $BACKUP_FILE | docker-compose exec -T postgres psql -U $PG_USER -d $PG_DATABASE
    
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}복원이 성공적으로 완료되었습니다.${NC}"
    else
        echo -e "${RED}복원에 실패했습니다.${NC}"
        exit 1
    fi
}

# 컨테이너 리셋
reset_postgres() {
    echo -e "${RED}경고: 이 작업은 컨테이너를 재시작하지만 데이터는 보존됩니다.${NC}"
    echo -e "${YELLOW}계속하시겠습니까? (y/n)${NC}"
    read -r answer
    
    if [[ "$answer" =~ ^[Yy]$ ]]; then
        echo -e "${YELLOW}PostgreSQL 컨테이너를 중지하고 제거합니다...${NC}"
        docker-compose down
        echo -e "${YELLOW}PostgreSQL 컨테이너를 다시 시작합니다...${NC}"
        docker-compose up -d
        echo -e "${GREEN}PostgreSQL 컨테이너가 재설정되었습니다.${NC}"
    else
        echo -e "${YELLOW}작업이 취소되었습니다.${NC}"
    fi
}

# 메인 함수
main() {
    # 명령행 인수 처리
    if [ $# -eq 0 ]; then
        usage
        exit 0
    fi
    
    load_env
    
    case "$1" in
        start)
            start_postgres
            ;;
        stop)
            stop_postgres
            ;;
        restart)
            stop_postgres
            start_postgres
            ;;
        status)
            check_status
            ;;
        logs)
            view_logs
            ;;
        shell)
            enter_shell
            ;;
        psql)
            connect_psql
            ;;
        backup)
            backup_database
            ;;
        restore)
            restore_database "$2"
            ;;
        reset)
            reset_postgres
            ;;
        *)
            echo -e "${RED}알 수 없는 명령: $1${NC}"
            usage
            exit 1
            ;;
    esac
}

# 스크립트 실행
main "$@" 