import jwt from 'jsonwebtoken';
import bcrypt from 'bcrypt';
import { cookies } from 'next/headers';
import { NextRequest, NextResponse } from 'next/server';
import { pg } from '../db';
import { getApiServerUrl } from '@/config/serverConfig';

// 환경 변수, 실제로는 .env 파일에서 로드해야 함
const JWT_SECRET = process.env.JWT_SECRET || 'your_jwt_secret_key';
const JWT_EXPIRES_IN = '7d'; // 7일
const COOKIE_NAME = 'auth_token';

// API 서버 URL을 동적으로 가져오기 (통일된 환경변수 사용)
function getDebugApiServerUrl(): string {
  const url = getApiServerUrl();
  // console.log('[AUTH] API 서버 URL:', {
  //   NODE_ENV: process.env.NODE_ENV,
  //   NEXT_PUBLIC_API_SERVER_URL: process.env.NEXT_PUBLIC_API_SERVER_URL,
  //   result: url
  // });
  return url;
}

const API_SERVER = getDebugApiServerUrl();

export interface UserData {
  id: string;
  username: string;
  email: string;
  role: string;
  apiKey?: string; // 선택적 필드로 변경
}

/**
 * 비밀번호 해싱
 */
export async function hashPassword(password: string): Promise<string> {
  const saltRounds = 10;
  return bcrypt.hash(password, saltRounds);
}

/**
 * 비밀번호 검증
 */
export async function comparePasswords(password: string, hash: string): Promise<boolean> {
  return bcrypt.compare(password, hash);
}

/**
 * JWT 토큰 생성
 */
export function generateToken(user: UserData): string {
  return jwt.sign(
    {
      id: user.id,
      username: user.username,
      email: user.email,
      role: user.role,
    },
    JWT_SECRET,
    { expiresIn: JWT_EXPIRES_IN }
  );
}

/**
 * JWT 토큰 검증
 */
export function verifyToken(token: string): UserData | null {
  try {
    return jwt.verify(token, JWT_SECRET) as UserData;
  } catch (error) {
    return null;
  }
}

/**
 * 쿠키에서 토큰 추출 (서버 컴포넌트용)
 */
export async function getTokenFromCookies(): Promise<string | undefined> {
  try {
    const cookieStore = await cookies();
    const token = cookieStore.get(COOKIE_NAME);
    return token?.value;
  } catch (error) {
    console.error('쿠키를 가져오는 중 오류 발생:', error);
    return undefined;
  }
}

/**
 * 현재 요청의 프로토콜이 HTTPS인지 확인
 */
function isHttpsRequest(req?: NextRequest): boolean {
  try {
    // 환경 변수로 강제 설정 가능
    if (process.env.USE_HTTPS === 'true') return true;
    if (process.env.USE_HTTPS === 'false') return false;
    
    // 요청 객체가 있는 경우 헤더에서 프로토콜 확인
    if (req) {
      // x-forwarded-proto 헤더 확인 (프록시/로드밸런서 환경)
      const forwardedProto = req.headers.get('x-forwarded-proto');
      if (forwardedProto) {
        return forwardedProto === 'https';
      }
      
      // host 헤더에서 프로토콜 추론
      const host = req.headers.get('host');
      if (host) {
        // localhost:3000 같은 경우는 HTTP, 도메인은 HTTPS로 추론
        return !host.includes('localhost') && !host.includes('127.0.0.1') && !host.match(/:\d+$/);
      }
    }
    
    // 환경 변수에서 프로토콜 확인
    const forwardedProto = process.env.X_FORWARDED_PROTO;
    if (forwardedProto) {
      return forwardedProto === 'https';
    }
    
    // Next.js 환경에서 기본 프로토콜 확인
    const protocol = process.env.NODE_ENV === 'production' 
      ? (process.env.VERCEL_URL ? 'https' : 'http') // Vercel 환경에서는 기본적으로 HTTPS
      : 'http'; // 개발 환경에서는 기본적으로 HTTP
    
    return protocol === 'https';
  } catch (error) {
    console.warn('프로토콜 확인 중 오류:', error);
    return false; // 오류 시 안전하게 false 반환
  }
}

/**
 * 토큰을 쿠키에 저장
 */
export function setTokenCookie(res: NextResponse, token: string, req?: NextRequest): void {
  const isProduction = process.env.NODE_ENV === 'production';
  
  // 현재 요청의 프로토콜에 따라 secure 설정 자동 결정
  const useHttps = isHttpsRequest(req);
  
  // 운영 환경에서 도메인 설정
  const cookieOptions: any = {
    name: COOKIE_NAME,
    value: token,
    httpOnly: true,
    secure: useHttps, // HTTPS 요청시에만 secure 설정
    sameSite: isProduction ? 'lax' : 'strict',
    maxAge: 60 * 60 * 24 * 7, // 7일
    path: '/',
  };
  
  // 환경 변수에서 쿠키 도메인 설정 읽기
  const cookieDomain = process.env.COOKIE_DOMAIN;
  if (cookieDomain && cookieDomain.trim() !== '') {
    cookieOptions.domain = cookieDomain;
  }
  
  console.log('쿠키 설정:', { 
    isProduction, 
    useHttps,
    secure: cookieOptions.secure, 
    sameSite: cookieOptions.sameSite,
    domain: cookieOptions.domain,
    protocol: useHttps ? 'https' : 'http'
  });
  
  res.cookies.set(cookieOptions);
}

/**
 * 쿠키 삭제
 */
export function clearTokenCookie(res: NextResponse, req?: NextRequest): void {
  const isProduction = process.env.NODE_ENV === 'production';
  
  // 현재 요청의 프로토콜에 따라 secure 설정 자동 결정
  const useHttps = isHttpsRequest(req);
  
  const cookieOptions: any = {
    name: COOKIE_NAME,
    value: '',
    httpOnly: true,
    secure: useHttps, // HTTPS 요청시에만 secure 설정
    sameSite: isProduction ? 'lax' : 'strict',
    maxAge: 0,
    path: '/',
  };
  
  // 환경 변수에서 쿠키 도메인 설정 읽기
  const cookieDomain = process.env.COOKIE_DOMAIN;
  if (cookieDomain && cookieDomain.trim() !== '') {
    cookieOptions.domain = cookieDomain;
  }
  
  res.cookies.set(cookieOptions);
}

/**
 * 요청에서 토큰 추출
 */
export function getTokenFromRequest(req: NextRequest): string | undefined {
  // 1. Authorization 헤더에서 토큰 확인 (Bearer 토큰)
  const authHeader = req.headers.get('Authorization');
  if (authHeader && authHeader.startsWith('Bearer ')) {
    return authHeader.substring(7); // "Bearer " 제거
  }
  
  // 2. 쿠키에서 토큰 확인 (fallback)
  const cookie = req.cookies.get(COOKIE_NAME);
  return cookie?.value;
}

/**
 * 요청에서 사용자 정보 가져오기
 */
export async function getUserFromRequest(req: NextRequest): Promise<UserData | null> {
  console.log('=== getUserFromRequest 시작 ===');
  
  const token = getTokenFromRequest(req);
  console.log('추출된 토큰:', token ? '있음' : '없음');
  
  if (!token) {
    console.log('토큰이 없어서 null 반환');
    return null;
  }

  try {
    console.log(`API 서버로 사용자 정보 요청: ${API_SERVER}/auth/me`);
    console.log('사용할 토큰:', token.substring(0, 20) + '...');
    
    // API 서버에서 현재 사용자 정보 조회
    const response = await fetch(`${API_SERVER}/auth/me`, {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${token}`
      }
    });

    console.log('API 서버 응답 상태:', response.status, response.statusText);
    
    if (!response.ok) {
      const errorText = await response.text();
      console.log('API 서버 오류 응답:', errorText);
      return null;
    }
    
    const data = await response.json();
    console.log('API 서버 응답 데이터:', data);
    
    if (!data.success) {
      console.log('API 서버에서 success: false 반환');
      return null;
    }
    
    console.log('사용자 정보 조회 성공:', data.data);
    return data.data;
  } catch (error) {
    console.error('사용자 정보 조회 중 오류:', error);
    return null;
  }
}

/**
 * 세션 저장
 */
export async function saveSession(userId: string, token: string, ipAddress?: string, userAgent?: string): Promise<boolean> {
  try {
    // 만료일 계산 (7일)
    const expiresAt = new Date();
    expiresAt.setDate(expiresAt.getDate() + 7);

    // 기존 세션 삭제
    await pg.query('DELETE FROM sessions WHERE user_id = $1', [userId]);

    // 새 세션 생성
    await pg.query(
      `INSERT INTO sessions (id, user_id, token, expires_at, ip_address, user_agent)
       VALUES (uuid_generate_v4(), $1, $2, $3, $4, $5)`,
      [userId, token, expiresAt.toISOString(), ipAddress || null, userAgent || null]
    );
    
    return true;
  } catch (error) {
    console.error('Error saving session:', error);
    return false;
  }
}

/**
 * API 서버에 로그인 요청
 */
export async function loginToApiServer(username: string, password: string): Promise<{ token: string; user: UserData; apiKey: string } | null> {
  try {
    // console.log(`API 서버에 로그인 요청: 사용자 ${username}`);
    // console.log(`API 서버 URL: ${API_SERVER}`);

    const response = await fetch(`${API_SERVER}/auth/login`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ username, password }),
    });

    if (!response.ok) {
      console.error(`API 서버 응답 오류: ${response.status} ${response.statusText}`);
      return null;
    }
    
    const data = await response.json();
    if (!data.success) {
      console.error('API 서버 로그인 실패:', data.error);
      return null;
    }

    // API 키 추출
    let apiKey = '';
    
    // apiKeys 배열이 있는 경우 처리
    if (data.data.apiKeys && Array.isArray(data.data.apiKeys) && data.data.apiKeys.length > 0) {
      // API 키 객체 타입 정의
      interface ApiKeyData {
        id: number;
        name: string;
        key: string;
        status: string;
        created_at: string;
        last_used: string | null;
        permissions: any;
      }

      // 활성 상태의 API 키 필터링
      const activeKeys = data.data.apiKeys.filter((k: ApiKeyData) => k.status === 'active');
      
      if (activeKeys.length > 0) {
        // 가장 최근에 사용된 API 키 우선, 없으면 첫 번째 키 사용
        let selectedKey = activeKeys.find((k: ApiKeyData) => k.last_used);
        if (!selectedKey) {
          selectedKey = activeKeys[0];
        }
        
        apiKey = selectedKey.key;
        // console.log(`🔑 API 키 선택: ${selectedKey.name} (${apiKey.substring(0, 10)}...)`);
      }
    } 
    // 단일 apiKey 필드가 있는 경우 처리 (이전 버전 호환성)
    else if (data.data.apiKey) {
      apiKey = data.data.apiKey;
      // console.log(`🔑 단일 API 키 사용: ${apiKey.substring(0, 10)}...`);
    }
    
    const hasValidApiKey = apiKey && typeof apiKey === 'string' && apiKey.trim() !== '';
    
    // console.log('API 서버 로그인 응답:', {
    //   token: data.data.token ? '토큰 있음' : '토큰 없음',
    //   apiKey: hasValidApiKey ? `API 키 있음 (${apiKey.substring(0, 5)}...)` : 'API 키 없음',
    //   apiKeysCount: data.data.apiKeys?.length || 0,
    //   user: data.data.user ? '사용자 정보 있음' : '사용자 정보 없음'
    // });
    
    // API 키가 없거나 유효하지 않은 경우 기본값 생성
    if (!hasValidApiKey) {
      // 실제 API 키와 유사한 형식으로 생성된 기본 API 키
      apiKey = `airun_default_${Date.now().toString(36)}`;
      console.warn(`⚠️ 유효한 API 키가 없어 기본값 생성: ${apiKey.substring(0, 10)}...`);
    } else {
      console.log(`✅ 유효한 API 키 확인: ${apiKey.substring(0, 5)}...`);
    }
    
    return {
      token: data.data.token,
      user: data.data.user,
      apiKey: apiKey
    };
  } catch (error) {
    console.error('로그인 중 오류 발생:', error);
    return null;
  }
}

/**
 * API 키 확인
 */
export async function validateApiKey(apiKey: string): Promise<UserData | null> {
  try {
    console.log('API 키 검증 시작:', apiKey.substring(0, 4) + '...');
    
    const response = await fetch(`${API_SERVER}/auth/validate-key`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ apiKey }),
    });

    if (!response.ok) {
      console.error('API 키 검증 실패:', {
        status: response.status,
        statusText: response.statusText
      });
      return null;
    }
    
    const data = await response.json();
    if (!data.success) {
      console.error('API 키 검증 실패:', data.error);
      return null;
    }
    
    console.log('API 키 검증 성공');
    return data.data.user;
  } catch (error) {
    console.error('API 키 검증 오류:', error);
    return null;
  }
} 