import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../middlewares';
import { getApiServerUrl } from '@/config/serverConfig';

const API_SERVER = getApiServerUrl();

// 기본 도구들 정의
const tools = [
  {
    type: "function",
    function: {
      name: "get_current_time",
      description: "현재 시간을 가져옵니다",
      parameters: {
        type: "object",
        properties: {
          timezone: {
            type: "string", 
            description: "시간대 (기본값: Asia/Seoul)"
          }
        }
      }
    }
  },
  {
    type: "function",
    function: {
      name: "calculate",
      description: "수학 계산을 수행합니다",
      parameters: {
        type: "object",
        properties: {
          expression: {
            type: "string",
            description: "계산할 수식 (예: 2+3*4)"
          }
        },
        required: ["expression"]
      }
    }
  },
  {
    type: "function", 
    function: {
      name: "get_random_number",
      description: "랜덤 숫자를 생성합니다",
      parameters: {
        type: "object",
        properties: {
          min: {
            type: "number",
            description: "최소값 (기본값: 1)"
          },
          max: {
            type: "number", 
            description: "최대값 (기본값: 100)"
          }
        }
      }
    }
  }
];

// 도구 관련 오류인지 확인하는 함수
function isToolRelatedError(errorText: string): boolean {
  return errorText.includes('tools') || 
         errorText.includes('tool_choice') || 
         errorText.includes('function calling') || 
         errorText.includes('does not support tools');
}

// 도구 실행 함수
async function executeTool(toolCall: any) {
  const { name, arguments: args } = toolCall.function;
  let parsedArgs: any = {};
  
  try {
    parsedArgs = typeof args === 'string' ? JSON.parse(args) : args;
  } catch (error) {
    return { error: `인수 파싱 실패: ${error}` };
  }

  switch (name) {
    case 'get_current_time':
      const timezone = parsedArgs.timezone || 'Asia/Seoul';
      const now = new Date().toLocaleString('ko-KR', { 
        timeZone: timezone,
        year: 'numeric',
        month: '2-digit', 
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
      });
      return { result: `현재 시간 (${timezone}): ${now}` };

    case 'calculate':
      try {
        const expression = parsedArgs.expression;
        // 안전한 계산을 위해 eval 대신 간단한 계산기 구현
        const result = Function(`"use strict"; return (${expression})`)();
        return { result: `${expression} = ${result}` };
      } catch (error) {
        return { error: `계산 오류: ${error}` };
      }

    case 'get_random_number':
      const min = parsedArgs.min || 1;
      const max = parsedArgs.max || 100;
      const randomNum = Math.floor(Math.random() * (max - min + 1)) + min;
      return { result: `${min}과 ${max} 사이의 랜덤 숫자: ${randomNum}` };

    default:
      return { error: `알 수 없는 도구: ${name}` };
  }
}

// GET 메서드: 모든 세션 반환 (type 필터링 제거)
export async function GET(request: NextRequest) {
  try {
    const userToken = request.headers.get('Authorization')?.split(' ')[1];
    const apiKey = request.headers.get('X-API-Key');
    
    if (!userToken || !apiKey) {
      return NextResponse.json({ 
        error: '인증 정보가 누락되었습니다. API 키와 토큰이 모두 필요합니다.' 
      }, { status: 401 });
    }

    // API 서버 URL 구성
          const apiServerUrl = getApiServerUrl().replace('/api/v1', '');
    const apiUrl = `${apiServerUrl}/api/v1/sessions`;
    
    // API 서버에서 세션 목록 조회
    const response = await fetch(apiUrl, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${userToken}`,
        'X-API-Key': apiKey,
        'accept': 'application/json'
      },
      cache: 'no-store'
    });

    if (!response.ok) {
      const errorText = await response.text();
      return NextResponse.json(
        { error: `세션 목록을 가져오는데 실패했습니다. (${response.status} ${response.statusText})` },
        { status: response.status }
      );
    }

    const apiData = await response.json();
    let sessions: any[] = [];
    if (Array.isArray(apiData.data)) {
      sessions = apiData.data.map((session: any) => ({
        id: session.id,
        title: session.title || '새 대화',
        lastMessage: session.lastMessage || session.title || '새 대화',
        updatedAt: session.updatedAt || session.createdAt || new Date().toISOString(),
        messageCount: (session.history || []).length,
        provider: session.provider,
        model: session.model,
        type: session.type,
        history: session.history || []
      }));
    } else if (typeof apiData.data === 'object' && apiData.data !== null) {
      sessions = Object.entries(apiData.data)
        .map(([id, session]: [string, any]) => ({
          id,
          title: session.title || '새 대화',
          lastMessage: session.lastMessage || session.title || '새 대화',
          updatedAt: session.updatedAt || session.createdAt || new Date().toISOString(),
          messageCount: (session.history || []).length,
          provider: session.provider,
          model: session.model,
          type: session.type,
          history: session.history || []
        }));
    }

    // 최신 순으로 정렬
    sessions.sort((a, b) => new Date(b.updatedAt).getTime() - new Date(a.updatedAt).getTime());

    return NextResponse.json({
      success: true,
      data: sessions
    });

  } catch (error) {
    return NextResponse.json(
      { error: '세션 목록을 가져오는 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const { model, message, messages, files, enableTools } = await request.json();

    if (!model || !message) {
      return NextResponse.json(
        { error: '모델과 메시지가 필요합니다.' },
        { status: 400 }
      );
    }

    // 메시지 히스토리 구성
    const chatHistory = messages.map((msg: any) => ({
      role: msg.role,
      content: msg.content
    }));

    // 파일이 있는 경우 처리
    if (files && files.length > 0) {
      const imageFiles = files.filter((file: any) => file.type.startsWith('image/'));
      const textFiles = files.filter((file: any) => file.type.startsWith('text/'));
      const pdfFiles = files.filter((file: any) => file.type === 'application/pdf');
      
      // 텍스트 파일 내용 추출
      let additionalText = '';
      if (textFiles.length > 0) {
        for (const file of textFiles) {
          try {
            // base64 디코딩하여 텍스트 내용 추출
            const base64Content = file.data.split(',')[1];
            const textContent = Buffer.from(base64Content, 'base64').toString('utf-8');
            additionalText += `\n\n[파일: ${file.name}]\n${textContent}`;
          } catch (error) {
            console.error(`텍스트 파일 처리 오류: ${file.name}`, error);
            additionalText += `\n\n[파일: ${file.name}] - 텍스트 파일 처리 중 오류가 발생했습니다.`;
          }
        }
      }
      
      // PDF 파일 처리 (텍스트 내용 포함)
      if (pdfFiles.length > 0) {
        for (const file of pdfFiles) {
          if (file.extractedText) {
            additionalText += `\n\n[PDF 파일: ${file.name}]\n${file.extractedText}`;
          } else {
            additionalText += `\n\n[PDF 파일: ${file.name}] - PDF 텍스트 추출에 실패했습니다.`;
          }
        }
      }
      
      // 최종 메시지에 텍스트 내용 포함
      const finalMessage = message + additionalText;
      
      if (imageFiles.length > 0) {
        // 이미지와 텍스트 모두 포함
        const content: any[] = [
          {
            type: "text",
            text: finalMessage
          }
        ];
        
        // 이미지 추가
        imageFiles.forEach((file: any) => {
          content.push({
            type: "image_url",
            image_url: {
              url: file.data
            }
          });
        });
        
        chatHistory.push({
          role: 'user',
          content: content
        });
      } else {
        // 텍스트/PDF 파일만 있는 경우
        chatHistory.push({
          role: 'user',
          content: finalMessage
        });
      }
    } else {
      // 파일이 없는 경우
      chatHistory.push({
        role: 'user',
        content: message
      });
    }

    // Ollama용 메시지 히스토리 (images 필드 사용)
    const ollamaChatHistory = chatHistory.map((msg: any) => {
      if (Array.isArray(msg.content)) {
        // OpenAI 형식을 Ollama 형식으로 변환
        const textContent = msg.content.find((c: any) => c.type === 'text')?.text || '';
        const imageUrls = msg.content.filter((c: any) => c.type === 'image_url').map((c: any) => c.image_url.url);
        
        const ollamaMsg: any = {
          role: msg.role,
          content: textContent
        };
        
        if (imageUrls.length > 0) {
          // Ollama API는 순수한 base64 문자열을 요구하므로 data: 접두사 제거
          ollamaMsg.images = imageUrls.map((url: string) => {
            if (url.startsWith('data:image/')) {
              return url.split(',')[1]; // data:image/jpeg;base64, 부분 제거
            }
            return url;
          });
        }
        
        return ollamaMsg;
      }
      return msg;
    });

    // 환경변수에서 서버 주소 가져오기
    const ollamaUrl = process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || process.env.OLLAMA_URL || 'http://localhost:11434';
    const vllmUrl = process.env.NEXT_PUBLIC_VLLM_SERVER || 'http://localhost:8000';

    // 도구 사용이 활성화된 경우 요청에 도구 정보 추가
    const ollamaRequestBody: any = {
      model: model,
      messages: ollamaChatHistory,
      stream: false,
      options: {
        temperature: 0.7,
        top_p: 0.9,
        top_k: 40,
      }
    };

    if (enableTools) {
      ollamaRequestBody.tools = tools;
    }

    console.log('Ollama 요청 시도:', { 
      url: `${ollamaUrl}/api/chat`, 
      model, 
      hasImages: ollamaChatHistory.some((m: any) => m.images),
      imageCount: ollamaChatHistory.filter((m: any) => m.images).length,
      messageStructure: ollamaChatHistory.map((m: any) => ({
        role: m.role,
        hasImages: !!m.images,
        imageCount: m.images ? m.images.length : 0,
        contentLength: m.content ? m.content.length : 0
      }))
    });
    
    // 모델 타입 확인 (ollama 또는 vllm)
    const response = await fetch(`${ollamaUrl}/api/chat`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(ollamaRequestBody),
    });

    console.log('Ollama 응답 상태:', response.status, response.statusText);

    if (!response.ok) {
      const ollamaErrorText = await response.text();
      console.log('Ollama 오류 응답:', ollamaErrorText);
      
      // 도구 지원하지 않는 모델인 경우 도구 없이 재시도
      if (enableTools && isToolRelatedError(ollamaErrorText)) {
        console.log('Ollama: 도구를 지원하지 않는 모델로 판단, 도구 없이 재시도');
        
        const withoutToolsRequest: any = {
          model: model,
          messages: ollamaChatHistory,
          stream: false,
          options: {
            temperature: 0.7,
            top_p: 0.9,
            top_k: 40,
          }
        };
        // tools 필드 제거
        
        const withoutToolsResponse = await fetch(`${ollamaUrl}/api/chat`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(withoutToolsRequest),
        });
        
        if (withoutToolsResponse.ok) {
          console.log('Ollama 도구 없는 요청 성공');
          const data = await withoutToolsResponse.json();
          const assistantResponse = data.message?.content || '응답을 받을 수 없습니다.';
          
          // 도구 관련 오류가 발생한 경우에만 안내 메시지 표시
          const responseText = '⚠️ 이 모델은 도구 기능을 지원하지 않습니다. 일반 채팅으로 처리합니다.\n\n' + assistantResponse;
          
          return NextResponse.json({
            response: responseText,
          });
        }
      }
      
      // 이미지가 포함된 경우 특정 오류에서만 fallback 처리
      if (files && files.length > 0 && (
        ollamaErrorText.includes('does not support images') ||
        ollamaErrorText.includes('vision') ||
        ollamaErrorText.includes('multimodal') ||
        response.status === 400
      )) {
        console.log('Ollama: 이미지 지원하지 않는 모델로 판단, 텍스트만으로 재시도');
        
        const textOnlyOllamaRequest: any = {
          model: model,
          messages: [{ role: 'user', content: message }],
          stream: false,
          options: {
            temperature: 0.7,
            num_predict: 2048,
          }
        };
        
        if (enableTools) {
          textOnlyOllamaRequest.tools = tools;
        }
        
        const textOnlyOllamaResponse = await fetch(`${ollamaUrl}/api/chat`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify(textOnlyOllamaRequest),
        });
        
        if (textOnlyOllamaResponse.ok) {
          console.log('Ollama 텍스트 전용 요청 성공');
          const textOnlyData = await textOnlyOllamaResponse.json();
          const assistantResponse = textOnlyData.message?.content || '응답을 받을 수 없습니다.';
          
          return NextResponse.json({
            response: '⚠️ 이 모델은 이미지 처리를 지원하지 않습니다. 텍스트만 처리합니다.\n\n' + assistantResponse,
          });
        }
      }
      
      // VLLM API 시도
      const vllmRequestBody: any = {
        model: model,
        messages: chatHistory,
        max_tokens: 2048,
        temperature: 0.7,
      };

      if (enableTools) {
        vllmRequestBody.tools = tools;
        vllmRequestBody.tool_choice = "auto";
      }

      console.log('VLLM 요청 시도:', { url: `${vllmUrl}/v1/chat/completions`, model, hasImages: chatHistory.some((m: any) => Array.isArray(m.content)) });

      const vllmResponse = await fetch(`${vllmUrl}/v1/chat/completions`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(vllmRequestBody),
      });

      console.log('VLLM 응답 상태:', vllmResponse.status, vllmResponse.statusText);

      if (!vllmResponse.ok) {
        const vllmErrorText = await vllmResponse.text();
        console.log('VLLM 오류 응답:', vllmErrorText);
        
        // 도구 지원하지 않는 모델인 경우 도구 없이 재시도
        if (enableTools && isToolRelatedError(vllmErrorText)) {
          console.log('VLLM: 도구를 지원하지 않는 모델로 판단, 도구 없이 재시도');
          
          const withoutToolsRequest = {
            model: model,
            messages: chatHistory,
            max_tokens: 2048,
            temperature: 0.7,
          };
          // tools와 tool_choice 필드 제거
          
          const withoutToolsResponse = await fetch(`${vllmUrl}/v1/chat/completions`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(withoutToolsRequest),
          });
          
          if (withoutToolsResponse.ok) {
            console.log('VLLM 도구 없는 요청 성공');
            const data = await withoutToolsResponse.json();
            const assistantResponse = data.choices[0].message.content || '응답을 받을 수 없습니다.';
            
            // 도구 관련 오류가 발생한 경우에만 안내 메시지 표시
            const responseText = '⚠️ 이 모델은 도구 기능을 지원하지 않습니다. 일반 채팅으로 처리합니다.\n\n' + assistantResponse;
            
            return NextResponse.json({
              response: responseText,
            });
          }
        }
        
        // 이미지가 포함된 경우 특정 오류에서만 fallback 처리
        if (files && files.length > 0 && (
          vllmErrorText.includes('does not support images') ||
          vllmErrorText.includes('vision') ||
          vllmErrorText.includes('multimodal') ||
          vllmErrorText.includes('content_type') ||
          vllmResponse.status === 400 || vllmResponse.status === 422
        )) {
          console.log('VLLM: 이미지 지원하지 않는 모델로 판단, 텍스트만으로 재시도');
          
          // 텍스트만 포함된 간단한 요청
          const textOnlyRequest = {
            model: model,
            messages: [{ role: 'user', content: message }],
            max_tokens: 2048,
            temperature: 0.7,
          };
          
          const textOnlyResponse = await fetch(`${vllmUrl}/v1/chat/completions`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(textOnlyRequest),
          });
          
          if (textOnlyResponse.ok) {
            console.log('VLLM 텍스트 전용 요청 성공');
            const textOnlyData = await textOnlyResponse.json();
            const assistantResponse = textOnlyData.choices[0].message.content || '응답을 받을 수 없습니다.';
            
            return NextResponse.json({
              response: '⚠️ 이 모델은 이미지 처리를 지원하지 않습니다. 텍스트만 처리합니다.\n\n' + assistantResponse,
            });
          }
        }
        
        throw new Error(`모델 응답을 받을 수 없습니다. Ollama: ${response.status} ${ollamaErrorText.substring(0, 100)}, VLLM: ${vllmResponse.status} ${vllmErrorText.substring(0, 100)}`);
      }

      const vllmData = await vllmResponse.json();
      const choice = vllmData.choices[0];
      
      // 도구 호출이 있는지 확인
      if (enableTools && choice.message.tool_calls && choice.message.tool_calls.length > 0) {
        let finalResponse = choice.message.content || '';
        
        // 각 도구 호출 실행
        for (const toolCall of choice.message.tool_calls) {
          const toolResult = await executeTool(toolCall);
          if (toolResult.result) {
            finalResponse += `\n\n🔧 도구 실행 결과: ${toolResult.result}`;
          } else if (toolResult.error) {
            finalResponse += `\n\n❌ 도구 실행 오류: ${toolResult.error}`;
          }
        }
        
        return NextResponse.json({
          response: finalResponse,
          model: model,
          type: 'vllm',
          tool_calls: choice.message.tool_calls
        });
      }

      return NextResponse.json({
        response: choice.message.content,
        model: model,
        type: 'vllm'
      });
    }

    const data = await response.json();
    let finalResponse = data.message.content;
    let toolCalls = null;

    // Ollama 도구 호출 처리
    if (enableTools && data.message.tool_calls && data.message.tool_calls.length > 0) {
      toolCalls = data.message.tool_calls;
      
      // 각 도구 호출 실행
      for (const toolCall of data.message.tool_calls) {
        const toolResult = await executeTool(toolCall);
        if (toolResult.result) {
          finalResponse += `\n\n🔧 도구 실행 결과: ${toolResult.result}`;
        } else if (toolResult.error) {
          finalResponse += `\n\n❌ 도구 실행 오류: ${toolResult.error}`;
        }
      }
    }

    return NextResponse.json({
      response: finalResponse,
      model: model,
      type: 'ollama',
      tool_calls: toolCalls
    });

  } catch (error) {
    console.error('채팅 API 오류:', error);
    return NextResponse.json(
      { error: '채팅 처리 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
} 