import { NextRequest, NextResponse } from 'next/server';
import path from 'path';
import fs from 'fs/promises';
import { existsSync } from 'fs';

const LOGO_DIR = path.join(process.cwd(), 'public/images/logos');

// 로고 디렉토리 생성
async function ensureLogoDir() {
  if (!existsSync(LOGO_DIR)) {
    await fs.mkdir(LOGO_DIR, { recursive: true });
  }
}

/**
 * 로고 업로드
 */
export async function POST(req: NextRequest) {
  try {
    await ensureLogoDir();

    // FormData에서 데이터 추출
    const formData = await req.formData();
    const file = formData.get('logo') as File;
    
    if (!file) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '로고 파일이 제공되지 않았습니다.'
          }
        }, 
        { status: 400 }
      );
    }
    
    // 이미지 파일 타입 확인
    if (!file.type.startsWith('image/')) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '올바른 이미지 파일이 아닙니다.'
          }
        }, 
        { status: 400 }
      );
    }

    // 파일 크기 제한 (5MB)
    const maxSize = 5 * 1024 * 1024;
    if (file.size > maxSize) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '로고 파일은 5MB 이하여야 합니다.'
          }
        }, 
        { status: 400 }
      );
    }
    
    // 파일 확장자 추출
    const ext = path.extname(file.name) || '.png';
    const fileName = `logo${ext}`;
    const filePath = path.join(LOGO_DIR, fileName);
    
    // 기존 로고 삭제 (다른 확장자 파일들도 삭제)
    const existingFiles = await fs.readdir(LOGO_DIR).catch(() => []);
    for (const existingFile of existingFiles) {
      if (existingFile.startsWith('logo.')) {
        await fs.unlink(path.join(LOGO_DIR, existingFile)).catch(() => {});
      }
    }
    
    // 파일 데이터를 바이너리 형태로 준비
    const bytes = await file.arrayBuffer();
    const buffer = Buffer.from(bytes);
    
    // 파일 저장
    await fs.writeFile(filePath, buffer);
    
    // 파일 권한 설정 (읽기 권한 추가)
    await fs.chmod(filePath, 0o644);
    
    const logoUrl = `/images/logos/${fileName}`;
    
    console.log('로고 업로드 완료:', {
      fileName,
      logoUrl,
      fileSize: file.size
    });
    
    return NextResponse.json({
      success: true,
      data: {
        logoUrl,
        message: '로고가 성공적으로 업로드되었습니다.'
      }
    });
    
  } catch (error) {
    console.error('로고 업로드 중 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '로고 업로드 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}

/**
 * 로고 조회
 */
export async function GET(req: NextRequest) {
  try {
    await ensureLogoDir();
    
    // 로고 파일 찾기
    const files = await fs.readdir(LOGO_DIR).catch(() => []);
    const logoFile = files.find(file => file.startsWith('logo.'));
    
    if (logoFile) {
      const logoUrl = `/images/logos/${logoFile}`;
      return NextResponse.json({
        success: true,
        data: {
          logoUrl,
          hasLogo: true
        }
      });
    } else {
      return NextResponse.json({
        success: true,
        data: {
          logoUrl: null,
          hasLogo: false
        }
      });
    }
    
  } catch (error) {
    console.error('로고 조회 중 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '로고 조회 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}

/**
 * 로고 삭제
 */
export async function DELETE(req: NextRequest) {
  try {
    await ensureLogoDir();
    
    // 로고 파일 찾기 및 삭제
    const files = await fs.readdir(LOGO_DIR).catch(() => []);
    let deleted = false;
    
    for (const file of files) {
      if (file.startsWith('logo.')) {
        await fs.unlink(path.join(LOGO_DIR, file));
        deleted = true;
      }
    }
    
    if (deleted) {
      return NextResponse.json({
        success: true,
        data: {
          message: '로고가 삭제되었습니다.'
        }
      });
    } else {
      return NextResponse.json({
        success: false,
        error: {
          message: '삭제할 로고가 없습니다.'
        }
      }, { status: 404 });
    }
    
  } catch (error) {
    console.error('로고 삭제 중 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '로고 삭제 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
} 