import { NextRequest, NextResponse } from 'next/server';
import { createReactAgent } from '@langchain/langgraph/prebuilt';
import { ChatOpenAI } from '@langchain/openai';
import { ChatAnthropic } from '@langchain/anthropic';
import { tool } from '@langchain/core/tools';
import { MemorySaver } from '@langchain/langgraph';
import { RunnableConfig } from '@langchain/core/runnables';
import fs from 'fs';
import path from 'path';
import os from 'os';
import { spawn, ChildProcess } from 'child_process';

// app.py 패턴: 정적 파일 디렉토리 설정
const STATIC_DIR = path.join(process.cwd(), 'public', 'static');
if (!fs.existsSync(STATIC_DIR)) {
  fs.mkdirSync(STATIC_DIR, { recursive: true });
}

// app.py 패턴: 이미지 저장 함수
function saveImageToStatic(imgPath: string, filename?: string): string | null {
  try {
    if (!filename) {
      filename = path.basename(imgPath);
    }
    
    // static 디렉토리에 파일 복사
    const staticPath = path.join(STATIC_DIR, filename);
    fs.copyFileSync(imgPath, staticPath);
    
    // 상대 URL 반환
    return `/static/${filename}`;
  } catch (error) {
    console.warn(`이미지 저장 중 오류 발생: ${error}`);
    return null;
  }
}

// app.py 패턴: 텍스트에서 이미지 처리
function processImageInText(text: string): string {
  try {
    // 이미지 경로 패턴 찾기
    const imagePatterns = [
      /\/[^\s]+\.(?:png|jpg|jpeg|gif|svg|webp)/gi,
      /file:\/\/[^\s]+\.(?:png|jpg|jpeg|gif|svg|webp)/gi,
      /\.\/[^\s]+\.(?:png|jpg|jpeg|gif|svg|webp)/gi
    ];
    
    let processedText = text;
    
    for (const pattern of imagePatterns) {
      const matches = text.match(pattern);
      if (matches) {
        for (const imagePath of matches) {
          try {
            // 절대 경로로 변환
            let fullPath = imagePath;
            if (imagePath.startsWith('./')) {
              fullPath = path.resolve(process.cwd(), imagePath);
            } else if (imagePath.startsWith('file://')) {
              fullPath = imagePath.replace('file://', '');
            }
            
            // 파일이 존재하는지 확인
            if (fs.existsSync(fullPath)) {
              const filename = `generated_${Date.now()}_${path.basename(fullPath)}`;
              const staticUrl = saveImageToStatic(fullPath, filename);
              
              if (staticUrl) {
                // 텍스트에서 이미지 경로를 마크다운 이미지 문법으로 변경
                processedText = processedText.replace(
                  imagePath,
                  `![생성된 이미지](${staticUrl})`
                );
              }
            }
          } catch (error) {
            console.warn(`이미지 처리 실패: ${imagePath}`, error);
          }
        }
      }
    }
    
    return processedText;
  } catch (error) {
    console.warn('이미지 처리 중 오류:', error);
    return text;
  }
}

// 설정 파일 경로
const getAirunConfigPath = () => {
  const homeDir = os.homedir();
  return path.join(homeDir, '.airun', 'airun.conf');
};

const getMCPConfigPath = () => {
  return path.join(process.cwd(), 'mcp_config.json');
};

// airun.conf에서 설정 읽기
const getAirunConfig = () => {
  const configPath = getAirunConfigPath();
  const config: any = {};
  
  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      for (const line of content.split('\n')) {
        const trimmed = line.trim();
        if (trimmed.startsWith('export ')) {
          const match = trimmed.match(/export\s+([A-Z_]+)=["']?([^"']+)["']?/);
          if (match) {
            config[match[1]] = match[2];
          }
        }
      }
    } catch (error) {
      console.error('airun.conf 읽기 실패:', error);
    }
  }
  
  return config;
};

// MCP 설정 로드
const loadMCPConfig = () => {
  const configPath = getMCPConfigPath();
  const defaultConfig = {
    "hamonize": {
      "command": "node",
      "args": ["./mcp_tools/hamonize/dist/index.js"],
      "transport": "stdio"
    },
    "postgres": {
      "transport": "stdio",
      "command": "node",
      "args": ["./mcp_tools/mcp-postgres-server/build/index.js"]
    }
  };

  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      return JSON.parse(content);
    } catch (error) {
      console.error('MCP 설정 로드 실패:', error);
      return defaultConfig;
    }
  }
  
  return defaultConfig;
};

// app.py 패턴을 적용한 개선된 MCP 클라이언트 클래스
class ImprovedMCPClient {
  private processes: Map<string, ChildProcess> = new Map();
  private messageId = 1;

  async initialize(mcpConfig: any): Promise<boolean> {
    console.log('🔄 MCP 클라이언트 초기화 시작 (app.py pattern)...');
    
    // app.py 패턴: 병렬 초기화 및 타임아웃 적용
    const initPromises = [];
    for (const [serverName, config] of Object.entries(mcpConfig)) {
      initPromises.push(this.startServer(serverName, config as any));
    }
    
    try {
      // app.py 패턴: 60초 타임아웃으로 모든 서버 초기화
      await Promise.race([
        Promise.allSettled(initPromises),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error('MCP 서버 초기화 시간 초과')), 60000)
        )
      ]);
      
      // app.py 패턴: 초기화 완료를 위한 대기 시간
      await new Promise(resolve => setTimeout(resolve, 3000));
      
      console.log(`✅ MCP 클라이언트 초기화 완료. 활성 서버: ${this.processes.size}개`);
      return this.processes.size > 0;
    } catch (error) {
      console.error('❌ MCP 클라이언트 초기화 실패:', error);
      return this.processes.size > 0; // 일부라도 성공하면 true
    }
  }

  private async startServer(serverName: string, config: any): Promise<void> {
    const { command, args, transport, env } = config;
    
    if (transport !== 'stdio') {
      console.warn(`${serverName}: stdio transport만 지원됩니다.`);
      return;
    }

    try {
      const spawnEnv = { ...process.env };
      if (env && typeof env === 'object') {
        Object.assign(spawnEnv, env);
      }

      const childProcess = spawn(command, args, {
        stdio: ['pipe', 'pipe', 'pipe'],
        cwd: process.cwd(),
        env: spawnEnv
      });

      childProcess.stderr?.on('data', (data: Buffer) => {
        console.debug(`${serverName} stderr:`, data.toString());
      });

      this.processes.set(serverName, childProcess);
      
      // app.py 패턴: 1초 대기 후 초기화
      await new Promise(resolve => setTimeout(resolve, 1000));
      
      // 서버 초기화 (타임아웃 적용)
      await Promise.race([
        this.sendRequest(serverName, 'initialize', {
          protocolVersion: '2024-11-05',
          capabilities: {
            roots: { listChanged: true },
            sampling: {}
          },
          clientInfo: {
            name: 'airun-langgraph-client',
            version: '1.0.0'
          }
        }),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error(`${serverName} 초기화 시간 초과`)), 30000)
        )
      ]);

      console.log(`✅ ${serverName} MCP 서버 초기화 완료`);
    } catch (error) {
      console.error(`❌ ${serverName} MCP 서버 시작 실패:`, error);
      // 실패한 프로세스 정리
      const failedProcess = this.processes.get(serverName);
      if (failedProcess) {
        failedProcess.kill();
        this.processes.delete(serverName);
      }
    }
  }

  private async sendRequest(serverName: string, method: string, params: any = {}) {
    const childProcess = this.processes.get(serverName);
    if (!childProcess || !childProcess.stdin) {
      throw new Error(`${serverName} 서버가 실행 중이지 않습니다.`);
    }

    const request: any = {
      jsonrpc: '2.0',
      id: this.messageId++,
      method
    };

    if (params !== null && params !== undefined) {
      request.params = params;
    }

    return new Promise((resolve, reject) => {
      // app.py 패턴: 30초 타임아웃
      const timeout = setTimeout(() => {
        reject(new Error(`${serverName} 요청 시간 초과`));
      }, 30000);

      let buffer = '';
      const onData = (data: Buffer) => {
        buffer += data.toString();
        
        while (true) {
          const newlineIndex = buffer.indexOf('\n');
          if (newlineIndex === -1) break;
          
          const line = buffer.slice(0, newlineIndex);
          buffer = buffer.slice(newlineIndex + 1);
          
          if (line.trim()) {
            try {
              const response = JSON.parse(line);
              if (response.id === request.id) {
                clearTimeout(timeout);
                childProcess.stdout?.off('data', onData);
                
                if (response.error) {
                  reject(new Error(`${serverName}: ${response.error.message}`));
                } else {
                  resolve(response.result);
                }
                return;
              }
            } catch (e) {
              // JSON 파싱 오류는 무시하고 계속 진행
            }
          }
        }
      };

      childProcess.stdout?.on('data', onData);
      childProcess.stdin?.write(JSON.stringify(request) + '\n');
    });
  }

  async callTool(serverName: string, toolName: string, arguments_: any) {
    try {
      // app.py 패턴: 도구 호출에도 타임아웃 적용
      const result = await Promise.race([
        this.sendRequest(serverName, 'tools/call', {
          name: toolName,
          arguments: arguments_
        }),
        new Promise((_, reject) => 
          setTimeout(() => reject(new Error(`${serverName}/${toolName} 도구 호출 시간 초과`)), 60000)
        )
      ]);
      
      return result;
    } catch (error) {
      console.error(`${serverName} 도구 호출 실패 (${toolName}):`, error);
      
      // app.py 패턴: 도구 실패 시 graceful fallback
      return {
        content: [{
          type: "text",
          text: `${toolName} 도구 호출에 실패했습니다: ${(error as Error).message}`
        }]
      };
    }
  }

  cleanup() {
    console.log('🧹 MCP 클라이언트 정리 시작...');
    for (const [serverName, process] of this.processes) {
      try {
        process.kill('SIGTERM');
        
        // app.py 패턴: 강제 종료를 위한 타이머
        setTimeout(() => {
          if (!process.killed) {
            process.kill('SIGKILL');
          }
        }, 5000);
        
        console.log(`🧹 ${serverName} 프로세스 종료됨`);
      } catch (error) {
        console.error(`${serverName} 정리 중 오류:`, error);
      }
    }
    this.processes.clear();
    console.log('🧹 MCP 클라이언트 정리 완료');
  }
}

// app.py 패턴: 전역 메모리 저장소 및 스레드 관리
const memorySaver = new MemorySaver();
const threadIds = new Map<string, string>();

// app.py 패턴: 사용자별 스레드 ID 생성 또는 가져오기
function getOrCreateThreadId(sessionId?: string): string {
  const key = sessionId || 'default';
  if (!threadIds.has(key)) {
    threadIds.set(key, `thread_${Date.now()}_${Math.random().toString(36).substring(7)}`);
  }
  return threadIds.get(key)!;
}

// 글로벌 MCP 클라이언트
let globalMCPClient: ImprovedMCPClient | null = null;
let isInitializing = false;
let initializationComplete = false;

// MCP 클라이언트 초기화
const initializeMCPClient = async () => {
  if (globalMCPClient && initializationComplete) {
    return globalMCPClient;
  }

  if (isInitializing) {
    while (isInitializing) {
      await new Promise(resolve => setTimeout(resolve, 100));
    }
    return globalMCPClient;
  }

  isInitializing = true;
  
  try {
    const mcpConfig = loadMCPConfig();
    globalMCPClient = new ImprovedMCPClient();
    
    const success = await globalMCPClient.initialize(mcpConfig);
    if (!success) {
      console.warn('⚠️ MCP 클라이언트 초기화 부분 실패, 기본 도구로 계속 진행');
    }
    
    initializationComplete = true;
    console.log('✅ 글로벌 MCP 클라이언트 초기화 완료');
  } catch (error) {
    console.error('❌ MCP 클라이언트 초기화 실패:', error);
    globalMCPClient = null;
    initializationComplete = false;
  } finally {
    isInitializing = false;
  }

  return globalMCPClient;
};

// LangGraph.js tools 정의 (JSON Schema 사용)
const webSearchTool = tool(
  async (input: any) => {
    console.log(`🔍 웹 검색 도구 호출됨: "${input.query}"`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'web_search', { query: input.query }) as any;
      console.log(`✅ 웹 검색 완료: ${input.query}`);
      return result.content?.[0]?.text || '검색 결과가 없습니다.';
    } catch (error) {
      console.error('웹 검색 실패:', error);
      throw new Error(`웹 검색 실패: ${(error as Error).message}`);
    }
  },
  {
    name: "web_search",
    description: "웹에서 정보를 검색합니다. 최신 뉴스, 기술 동향, 일반적인 정보 등을 찾을 때 사용합니다.",
    schema: {
      type: "object",
      properties: {
        query: {
          type: "string",
          description: "검색할 키워드나 질문"
        }
      },
      required: ["query"]
    }
  }
);

const translateTool = tool(
  async (input: any) => {
    console.log(`🌍 번역 도구 호출됨: "${input.text}" -> ${input.target_lang || 'KO'}`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'translate_text', { 
        text: input.text, 
        target_lang: input.target_lang || 'KO' 
      }) as any;
      console.log(`✅ 번역 완료: ${input.text} -> ${input.target_lang || 'KO'}`);
      return result.content?.[0]?.text || '번역 결과가 없습니다.';
    } catch (error) {
      console.error('번역 실패:', error);
      throw new Error(`번역 실패: ${(error as Error).message}`);
    }
  },
  {
    name: "translate_text",
    description: "텍스트를 다른 언어로 번역합니다. 한국어, 영어, 일본어, 중국어 등 다양한 언어를 지원합니다.",
    schema: {
      type: "object",
      properties: {
        text: {
          type: "string",
          description: "번역할 텍스트"
        },
        target_lang: {
          type: "string",
          description: "번역할 언어 (KO, EN, JA, ZH 등)",
          default: "KO"
        }
      },
      required: ["text"]
    }
  }
);

const generateCodeTool = tool(
  async (input: any) => {
    console.log(`⚡ 코드 생성 도구 호출됨: "${input.prompt}" (${input.language || 'python'})`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'generate_code', { 
        prompt: input.prompt, 
        language: input.language || 'python' 
      }) as any;
      console.log(`✅ 코드 생성 완료: ${input.prompt}`);
      return result.content?.[0]?.text || '코드 생성 결과가 없습니다.';
    } catch (error) {
      console.error('코드 생성 실패:', error);
      throw new Error(`코드 생성 실패: ${(error as Error).message}`);
    }
  },
  {
    name: "generate_code",
    description: "지정된 프로그래밍 언어로 코드를 생성합니다. 함수, 클래스, 스크립트 등을 생성할 수 있습니다.",
    schema: {
      type: "object",
      properties: {
        prompt: {
          type: "string",
          description: "코드 생성에 사용할 프롬프트나 요구사항"
        },
        language: {
          type: "string",
          description: "생성할 코드의 프로그래밍 언어 (python, javascript, java, c++ 등)",
          default: "python"
        }
      },
      required: ["prompt"]
    }
  }
);

const ragSearchTool = tool(
  async (input: any) => {
    console.log(`📚 RAG 검색 도구 호출됨: "${input.query}"`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'rag_search', { query: input.query }) as any;
      console.log(`✅ RAG 검색 완료: ${input.query}`);
      return result.content?.[0]?.text || 'RAG 검색 결과가 없습니다.';
    } catch (error) {
      console.error('RAG 검색 실패:', error);
      throw new Error(`RAG 검색 실패: ${(error as Error).message}`);
    }
  },
  {
    name: "rag_search",
    description: "문서 데이터베이스에서 관련 정보를 검색합니다. 업로드된 문서나 지식 베이스에서 정보를 찾을 때 사용합니다.",
    schema: {
      type: "object",
      properties: {
        query: {
          type: "string",
          description: "검색할 키워드나 질문"
        }
      },
      required: ["query"]
    }
  }
);

const createReportTool = tool(
  async (input: any) => {
    console.log(`📄 리포트 생성 도구 호출됨: ${input.format || 'pdf'} 형식`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'create_report', { 
        content: input.content, 
        format: input.format || 'pdf', 
        template: input.template || 'simple' 
      }) as any;
      console.log(`✅ 리포트 생성 완료: ${input.format || 'pdf'} 형식`);
      return result.content?.[0]?.text || '리포트 생성 결과가 없습니다.';
    } catch (error) {
      console.error('리포트 생성 실패:', error);
      throw new Error(`리포트 생성 실패: ${(error as Error).message}`);
    }
  },
  {
    name: "create_report",
    description: "지정된 내용으로 리포트를 생성합니다. PDF, Word, HTML 등 다양한 형식을 지원합니다.",
    schema: {
      type: "object",
      properties: {
        content: {
          type: "string",
          description: "리포트에 포함할 내용"
        },
        format: {
          type: "string",
          description: "리포트 형식 (pdf, docx, html 등)",
          default: "pdf"
        },
        template: {
          type: "string",
          description: "리포트 템플릿 (simple, professional, academic 등)",
          default: "simple"
        }
      },
      required: ["content"]
    }
  }
);

// 모든 도구를 배열로 정의
const allTools = [
  webSearchTool,
  translateTool,
  generateCodeTool,
  ragSearchTool,
  createReportTool,
];

// LLM 모델 생성 함수
function createLLMModel(provider: string, model: string, config: any, temperature?: number) {
  switch (provider) {
    case 'openai':
      return new ChatOpenAI({
        apiKey: config.OPENAI_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    case 'anthropic':
      return new ChatAnthropic({
        apiKey: config.ANTHROPIC_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    default:
      throw new Error(`지원하지 않는 프로바이더: ${provider}`);
  }
}

// 스트리밍 응답 생성 함수
const createStreamingResponse = (generator: () => AsyncGenerator<string>) => {
  return new Response(
    new ReadableStream({
      async start(controller) {
        try {
          for await (const chunk of generator()) {
            controller.enqueue(new TextEncoder().encode(chunk));
          }
        } catch (error) {
          console.error('스트리밍 오류:', error);
          controller.enqueue(new TextEncoder().encode(`\n\n오류: ${(error as Error).message}`));
        } finally {
          controller.close();
        }
      },
    }),
    {
      headers: {
        'Content-Type': 'text/plain; charset=utf-8',
        'Cache-Control': 'no-cache',
        'Connection': 'keep-alive',
      },
    }
  );
};

// LangGraph 에이전트로 처리
async function* processWithLangGraph(
  message: string,
  files: any[],
  provider: string,
  model: string,
  temperature?: number,
  history?: Array<{role: 'user' | 'assistant', content: string}>
) {
  console.log(`🚀 LangGraph 완전 에이전트 처리 시작: ${provider}/${model}`);
  
  try {
    // 설정 로드
    const config = getAirunConfig();
    
    // LLM 모델 생성
    const llm = createLLMModel(provider, model, config, temperature);
    
    // app.py 패턴: MemorySaver와 설정을 포함한 LangGraph 에이전트 생성
    console.log('🔧 LangGraph 완전 에이전트 생성 중 (with MemorySaver)...');
    const agent = createReactAgent({
      llm,
      tools: allTools,
      checkpointer: memorySaver,  // app.py 패턴: 대화 기록 관리
    });
    
    console.log('✅ LangGraph 완전 에이전트 생성 완료 (with MemorySaver)');
    
    // 메시지 준비
    let finalMessage = message;
    if (files && files.length > 0) {
      const fileDescriptions = files.map(file => {
        let desc = `파일: "${file.name}" (${file.type})`;
        if (file.extractedText) {
          desc += `\n텍스트 내용:\n${file.extractedText}`;
        }
        if (file.isImage && file.base64) {
          desc += `\n이미지가 첨부되었습니다.`;
        }
        return desc;
      }).join('\n\n');
      
      finalMessage = `${fileDescriptions}\n\n질문: ${message}`;
    }
    
    // 히스토리를 메시지 형식으로 변환
    const messages = [];
    if (history && history.length > 0) {
      for (const msg of history) {
        messages.push({
          role: msg.role === 'user' ? 'human' : 'assistant',
          content: msg.content,
        });
      }
    }
    
    // 현재 메시지 추가
    messages.push({
      role: 'human' as const,
      content: finalMessage,
    });
    
    console.log('📤 LangGraph 완전 에이전트 실행 중...');
    
    // 에이전트 실행
    const result = await agent.invoke({
      messages: messages,
    });
    
    console.log('✅ LangGraph 완전 에이전트 처리 완료');
    
    // 결과 처리
    if (result && result.messages && result.messages.length > 0) {
      const lastMessage = result.messages[result.messages.length - 1];
      
      // 응답 텍스트 추출
      let responseText = '';
      if (typeof lastMessage.content === 'string') {
        responseText = lastMessage.content;
      } else if (Array.isArray(lastMessage.content)) {
        // 복합 메시지 처리
        responseText = lastMessage.content.map((part: any) => 
          typeof part === 'string' ? part : part.text || ''
        ).join(' ');
      } else {
        responseText = '응답을 생성했습니다.';
      }
      
      // 스트리밍 방식으로 응답 전송
      const words = responseText.split(' ');
      for (let i = 0; i < words.length; i++) {
        yield words[i] + (i < words.length - 1 ? ' ' : '');
        await new Promise(resolve => setTimeout(resolve, 50)); // 50ms 지연
      }
      
    } else {
      yield '응답을 생성할 수 없습니다.';
    }
    
  } catch (error) {
    console.error('❌ LangGraph 완전 에이전트 처리 실패:', error);
    yield `처리 중 오류가 발생했습니다: ${(error as Error).message}`;
  }
}

// POST: 채팅 메시지 처리 (LangGraph + MCP 도구 사용)
export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const message = formData.get('message') as string;
    const provider = formData.get('provider') as string;
    const model = formData.get('model') as string;
    const temperature = parseFloat(formData.get('temperature') as string) || undefined;
    const historyString = formData.get('history') as string;
    
    let history: Array<{role: 'user' | 'assistant', content: string}> = [];
    if (historyString) {
      try {
        history = JSON.parse(historyString);
      } catch (error) {
        console.error('히스토리 파싱 실패:', error);
      }
    }
    
    if (!message || message.trim() === '') {
      return NextResponse.json(
        { error: '메시지가 필요합니다.' },
        { status: 400 }
      );
    }
    
    // 지원하는 프로바이더 확인
    if (!['openai', 'anthropic'].includes(provider)) {
      return NextResponse.json(
        { error: `현재 LangGraph 모드에서는 OpenAI와 Anthropic만 지원됩니다. (요청: ${provider})` },
        { status: 400 }
      );
    }
    
    // 다중 파일 처리
    const files: any[] = [];
    let fileIndex = 0;
    
    while (true) {
      const file = formData.get(`file_${fileIndex}`) as File | null;
      if (!file) break;
      
      const fileInfo: any = {
        name: file.name,
        type: file.type,
        size: file.size,
        file: file
      };
      
      // 텍스트 추출 데이터
      const extractedText = formData.get(`file_${fileIndex}_text`) as string | null;
      if (extractedText) {
        fileInfo.extractedText = extractedText;
      }
      
      // Base64 데이터 (이미지)
      const base64Data = formData.get(`file_${fileIndex}_base64`) as string | null;
      if (base64Data) {
        fileInfo.base64 = base64Data;
        fileInfo.isImage = true;
      }
      
      files.push(fileInfo);
      fileIndex++;
    }
    
    console.log(`🦜 LangGraph 완전 채팅 요청: ${message}`);
    if (files.length > 0) {
      console.log(`📎 첨부 파일 ${files.length}개:`, files.map(f => `${f.name} (${f.type})`).join(', '));
    }

    // 스트리밍 응답 생성
    return createStreamingResponse(async function* () {
      yield* processWithLangGraph(message, files, provider, model, temperature, history);
    });
    
  } catch (error) {
    console.error('LangGraph 완전 채팅 처리 실패:', error);
    return NextResponse.json(
      { error: '채팅 처리 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
}

// GET: 채팅 상태 확인
export async function GET() {
  try {
    const config = getAirunConfig();
    
    return NextResponse.json({
      available: true,
      mode: 'langgraph-full',
      provider: config.USE_LLM || 'anthropic',
      model: config.ANTHROPIC_MODEL || config.OPENAI_MODEL || 'default',
      supportedProviders: ['openai', 'anthropic'],
      toolsCount: allTools.length,
      tools: allTools.map(tool => tool.name),
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error('LangGraph 완전 채팅 상태 확인 실패:', error);
    return NextResponse.json(
      { error: '채팅 상태 확인에 실패했습니다.' },
      { status: 500 }
    );
  }
}

// 서버 종료 시 정리
process.on('SIGINT', async () => {
  if (globalMCPClient) {
    globalMCPClient.cleanup();
  }
});

process.on('SIGTERM', async () => {
  if (globalMCPClient) {
    globalMCPClient.cleanup();
  }
}); 