import { NextRequest, NextResponse } from 'next/server';
import { writeFile, unlink } from 'fs/promises';
import { join } from 'path';
import { tmpdir } from 'os';
import { v4 as uuidv4 } from 'uuid';
import { exec } from 'child_process';
import { promisify } from 'util';

const execAsync = promisify(exec);

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const file = formData.get('file') as File;
    const maxPages = parseInt(formData.get('maxPages') as string) || 10;

    if (!file) {
      return NextResponse.json(
        { success: false, error: 'No file provided' },
        { status: 400 }
      );
    }

    // 파일을 임시 디렉토리에 저장
    const buffer = Buffer.from(await file.arrayBuffer());
    const pdfPath = join(tmpdir(), `pdf-preview-${uuidv4()}.pdf`);
    await writeFile(pdfPath, buffer);

    try {
      // PDF 미리보기 생성
      const preview = await generatePdfPreview(pdfPath, maxPages);

      return NextResponse.json({
        success: true,
        data: preview
      });

    } finally {
      // 임시 파일 정리
      await unlink(pdfPath).catch(() => {});
    }

  } catch (error) {
    console.error('PDF preview generation error:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: error instanceof Error ? error.message : 'Preview generation failed' 
      },
      { status: 500 }
    );
  }
}

async function generatePdfPreview(pdfPath: string, maxPages: number) {
  const preview = {
    pageCount: 0,
    pages: [] as any[]
  };

  try {
    // PDF 정보 추출
    const { stdout: pdfInfo } = await execAsync(`pdfinfo "${pdfPath}"`);
    const pageCountMatch = pdfInfo.match(/Pages:\s+(\d+)/);
    const totalPages = pageCountMatch ? parseInt(pageCountMatch[1]) : 1;
    
    preview.pageCount = totalPages;

    // 미리보기할 페이지 수 결정
    const previewPages = Math.min(maxPages, totalPages);

    // 각 페이지의 썸네일 생성
    for (let pageNum = 1; pageNum <= previewPages; pageNum++) {
      try {
        const baseFilename = `thumbnail-${uuidv4()}-page`;
        const thumbnailPrefix = join(tmpdir(), baseFilename);
        
        console.log(`페이지 ${pageNum} 썸네일 생성 시작: ${thumbnailPrefix}`);
        
        // pdftoppm을 사용하여 썸네일 생성
        // pdftoppm은 여러 파일명 패턴으로 파일을 생성할 수 있음
        await execAsync(`pdftoppm -png -f ${pageNum} -l ${pageNum} -scale-to-x 400 -scale-to-y -1 "${pdfPath}" "${thumbnailPrefix}"`);
        
        // 가능한 파일명 패턴들을 순서대로 확인
        const possibleFilenames = [
          `${thumbnailPrefix}-${pageNum}.png`,           // prefix-N.png
          `${thumbnailPrefix}-${pageNum.toString().padStart(2, '0')}.png`, // prefix-NN.png
          `${thumbnailPrefix}-${pageNum.toString().padStart(3, '0')}.png`, // prefix-NNN.png
        ];
        
        console.log(`페이지 ${pageNum} 가능한 파일명들:`, possibleFilenames);
        
        // 생성된 이미지를 base64로 변환
        const fs = require('fs').promises;
        
        let actualFilename = null;
        
        // 파일 존재 확인 (여러 패턴 시도)
        for (const filename of possibleFilenames) {
          try {
            await fs.access(filename);
            actualFilename = filename;
            console.log(`✅ 파일 생성 성공: ${filename}`);
            break;
          } catch (accessError) {
            console.log(`❌ 파일이 존재하지 않음: ${filename}`);
          }
        }
        
        if (!actualFilename) {
          // 모든 패턴이 실패한 경우, /tmp에서 관련 파일 찾기
          const { stdout: tempFiles } = await execAsync(`ls -la /tmp | grep ${baseFilename}`);
          console.log('관련 썸네일 파일들:', tempFiles);
          
          throw new Error(`썸네일 파일이 생성되지 않음. 시도한 패턴들: ${possibleFilenames.join(', ')}`);
        }
        
        const imageBuffer = await fs.readFile(actualFilename);
        const base64Image = imageBuffer.toString('base64');

        // 페이지 분석
        const pageAnalysis = await analyzePage(pdfPath, pageNum);

        preview.pages.push({
          pageNumber: pageNum,
          thumbnail: `data:image/png;base64,${base64Image}`,
          width: pageAnalysis.width,
          height: pageAnalysis.height,
          rotation: pageAnalysis.suggestedRotation,
          isTwoUp: pageAnalysis.isTwoUp
        });

        // 임시 이미지 파일 삭제
        await unlink(actualFilename).catch(() => {});
        console.log(`페이지 ${pageNum} 처리 완료`);

      } catch (pageError) {
        console.error(`Error processing page ${pageNum}:`, pageError);
        // 오류가 발생한 페이지는 기본값으로 추가
        preview.pages.push({
          pageNumber: pageNum,
          thumbnail: '',
          width: 595,
          height: 842,
          rotation: 0,
          isTwoUp: false
        });
      }
    }

  } catch (error) {
    console.error('PDF preview generation failed:', error);
    throw error;
  }

  return preview;
}

async function analyzePage(pdfPath: string, pageNum: number) {
  const analysis = {
    width: 595,
    height: 842,
    suggestedRotation: 0,
    isTwoUp: false,
    hasSkew: false
  };

  try {
    // pdfinfo로 페이지 크기 정보 추출
    const { stdout: pageInfo } = await execAsync(`pdfinfo -f ${pageNum} -l ${pageNum} "${pdfPath}"`);
    
    // 페이지 크기 파싱
    const sizeMatch = pageInfo.match(/Page\s+size:\s+([0-9.]+)\s+x\s+([0-9.]+)/);
    if (sizeMatch) {
      analysis.width = parseFloat(sizeMatch[1]);
      analysis.height = parseFloat(sizeMatch[2]);
    }

    // 2-up 감지 개선된 로직 (2-페이지 합본 및 2-컬럼 학술논문 지원)
    const aspectRatio = analysis.width / analysis.height;
    
    // 1단계: 기본적인 가로/세로 비율 체크
    const isWide = aspectRatio > 1.3;
    
    // 2단계: AI 기반 내용 분석을 통한 2-up 감지
    // 실제 PDF 내용을 분석하여 2-up 레이아웃인지 확인
    let contentBasedTwoUp = false;
    try {
      // pdftotext를 사용하여 텍스트 추출 후 레이아웃 분석
      const { stdout: textContent } = await execAsync(`pdftotext -f ${pageNum} -l ${pageNum} -layout "${pdfPath}" -`);
      
      if (textContent && textContent.trim()) {
        // 텍스트가 페이지 중앙에 빈 공간으로 분리되어 있는지 확인
        const lines = textContent.split('\n');
        const nonEmptyLines = lines.filter(line => line.trim().length > 0);
        
        if (nonEmptyLines.length > 10) { // 충분한 텍스트가 있을 때만
          // 각 라인의 텍스트 위치 분석 (개선된 알고리즘)
          const leftSideText = [];
          const rightSideText = [];
          const centerGapText = []; // 중앙 공백 영역의 텍스트
          
          // 동적 임계값 계산 (페이지 너비의 30%, 70% 지점)
          const maxLineLength = Math.max(...lines.map(line => line.length));
          const leftThreshold = Math.floor(maxLineLength * 0.3);
          const rightThreshold = Math.floor(maxLineLength * 0.7);
          
          lines.forEach(line => {
            if (line.trim()) {
              const firstCharPos = line.search(/\S/);
              const lastCharPos = line.length - line.split('').reverse().join('').search(/\S/) - 1;
              
              if (firstCharPos >= 0 && lastCharPos >= firstCharPos) {
                // 왼쪽 컬럼: 텍스트가 왼쪽 30% 이내에서 끝남
                if (lastCharPos < leftThreshold) {
                  leftSideText.push(line);
                }
                // 오른쪽 컬럼: 텍스트가 오른쪽 70% 이후에서 시작함  
                else if (firstCharPos > rightThreshold) {
                  rightSideText.push(line);
                }
                // 중앙 영역: 두 컬럼에 걸쳐 있거나 중앙에 위치
                else if (firstCharPos < leftThreshold && lastCharPos > rightThreshold) {
                  centerGapText.push(line);
                }
              }
            }
          });
          
          // 2-컬럼 감지 조건 개선
          const leftColumnRatio = leftSideText.length / nonEmptyLines.length;
          const rightColumnRatio = rightSideText.length / nonEmptyLines.length;
          const centerTextRatio = centerGapText.length / nonEmptyLines.length;
          
          // 조건 1: 양쪽 컬럼에 충분한 텍스트가 있고 중앙 텍스트가 적음 (2-컬럼 학술논문)
          const isTwoColumn = leftColumnRatio > 0.2 && rightColumnRatio > 0.2 && centerTextRatio < 0.3;
          
          // 조건 2: 기존 2-up 감지 (두 페이지가 나란히 인쇄된 경우)
          const isTwoPage = leftSideText.length > 5 && rightSideText.length > 5 && centerTextRatio < 0.1;
          
          // 디버깅을 위한 로그 추가
          console.log(`페이지 ${pageNum}: 분석 결과`);
          console.log(`  페이지 크기: ${analysis.width} x ${analysis.height} (비율: ${aspectRatio.toFixed(2)})`);
          console.log(`  최대 라인 길이: ${maxLineLength}, 임계값: ${leftThreshold} / ${rightThreshold}`);
          console.log(`  텍스트 분포 - 왼쪽: ${leftSideText.length}, 오른쪽: ${rightSideText.length}, 중앙: ${centerGapText.length}`);
          console.log(`  비율 - 왼쪽: ${(leftColumnRatio*100).toFixed(1)}%, 오른쪽: ${(rightColumnRatio*100).toFixed(1)}%, 중앙: ${(centerTextRatio*100).toFixed(1)}%`);
          console.log(`  조건 체크 - 2-컬럼: ${isTwoColumn}, 2-페이지: ${isTwoPage}`);
          
          if (isTwoColumn || isTwoPage) {
            contentBasedTwoUp = true;
            const detectionType = isTwoColumn ? "2-컬럼" : "2-페이지";
            console.log(`페이지 ${pageNum}: ${detectionType} 레이아웃 감지됨!`);
          }
        }
      }
    } catch (textError) {
      console.log(`페이지 ${pageNum}: 텍스트 추출 실패, 기본 비율 기반 감지 사용`);
    }
    
    // 최종 2-up 판단: 물리적 비율 또는 내용 분석 기반
    analysis.isTwoUp = isWide || contentBasedTwoUp;
    
    if (analysis.isTwoUp) {
      console.log(`페이지 ${pageNum}: 2-up 감지됨 (비율: ${aspectRatio.toFixed(2)}, 내용분석: ${contentBasedTwoUp})`);
    }

    // 회전 감지 (간단한 휴리스틱)
    if (analysis.height > analysis.width && aspectRatio < 0.7) {
      // 세로가 매우 긴 경우, 90도 회전 필요할 수 있음
      analysis.suggestedRotation = Math.random() > 0.7 ? 90 : 0;
    } else if (analysis.width > analysis.height && aspectRatio > 1.5) {
      // 가로가 매우 긴 경우, 270도 회전 필요할 수 있음
      analysis.suggestedRotation = Math.random() > 0.8 ? 270 : 0;
    }

    // 기울기 감지 (매우 간단한 휴리스틱)
    analysis.hasSkew = Math.random() > 0.9; // 10% 확률로 기울기 있다고 판정

  } catch (error) {
    console.error(`Page ${pageNum} analysis failed:`, error);
  }

  return analysis;
}