import { NextRequest, NextResponse } from 'next/server';
import path from 'path';
import fs from 'fs/promises';
import { existsSync } from 'fs';

const PROFILE_IMAGES_DIR = path.join(process.cwd(), 'public/images/profiles');

// 프로필 이미지 디렉토리 생성
async function ensureProfileImagesDir() {
  if (!existsSync(PROFILE_IMAGES_DIR)) {
    await fs.mkdir(PROFILE_IMAGES_DIR, { recursive: true });
  }
}

/**
 * 프로필 이미지 업로드
 */
export async function POST(req: NextRequest) {
  try {
    await ensureProfileImagesDir();

    // FormData에서 데이터 추출
    const formData = await req.formData();
    const file = formData.get('image') as File;
    const userId = formData.get('userId') as string;
    
    if (!userId) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '사용자 ID가 필요합니다.'
          }
        }, 
        { status: 400 }
      );
    }
    
    if (!file) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '이미지 파일이 제공되지 않았습니다.'
          }
        }, 
        { status: 400 }
      );
    }
    
    // 이미지 파일 타입 확인
    if (!file.type.startsWith('image/')) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '올바른 이미지 파일이 아닙니다.'
          }
        }, 
        { status: 400 }
      );
    }

    // 파일 크기 제한 (5MB)
    const maxSize = 5 * 1024 * 1024;
    if (file.size > maxSize) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '이미지 파일은 5MB 이하여야 합니다.'
          }
        }, 
        { status: 400 }
      );
    }
    
    // 파일 확장자 추출
    const ext = path.extname(file.name) || '.jpg';
    const fileName = `${userId}${ext}`;
    const filePath = path.join(PROFILE_IMAGES_DIR, fileName);
    
    // 기존 프로필 이미지 삭제 (다른 확장자 파일들도 삭제)
    const existingFiles = await fs.readdir(PROFILE_IMAGES_DIR).catch(() => []);
    for (const existingFile of existingFiles) {
      if (existingFile.startsWith(`${userId}.`)) {
        await fs.unlink(path.join(PROFILE_IMAGES_DIR, existingFile)).catch(() => {});
      }
    }
    
    // 파일 데이터를 바이너리 형태로 준비
    const bytes = await file.arrayBuffer();
    const buffer = Buffer.from(bytes);
    
    // 파일 저장
    await fs.writeFile(filePath, buffer);
    
    // 파일 권한 설정 (읽기 권한 추가)
    await fs.chmod(filePath, 0o644);
    
    const imageUrl = `/images/profiles/${fileName}`;
    
    console.log('프로필 이미지 업로드 완료:', {
      userId,
      fileName,
      imageUrl,
      fileSize: file.size
    });
    
    return NextResponse.json({
      success: true,
      data: {
        imageUrl,
        message: '프로필 이미지가 성공적으로 업로드되었습니다.'
      }
    });
    
  } catch (error) {
    console.error('프로필 이미지 업로드 중 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '프로필 이미지 업로드 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}

/**
 * 프로필 이미지 조회
 */
export async function GET(req: NextRequest) {
  try {
    await ensureProfileImagesDir();
    
    // URL 파라미터에서 userId 추출
    const { searchParams } = new URL(req.url);
    const userId = searchParams.get('userId');
    
    if (!userId) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '사용자 ID가 필요합니다.'
          }
        }, 
        { status: 400 }
      );
    }
    
    // 사용자의 프로필 이미지 파일 찾기
    const files = await fs.readdir(PROFILE_IMAGES_DIR).catch(() => []);
    const userImageFile = files.find(file => file.startsWith(`${userId}.`));
    
    if (userImageFile) {
      const imageUrl = `/images/profiles/${userImageFile}`;
      return NextResponse.json({
        success: true,
        data: {
          imageUrl,
          hasImage: true
        }
      });
    } else {
      return NextResponse.json({
        success: true,
        data: {
          imageUrl: '/images/user.png', // 기본 이미지
          hasImage: false
        }
      });
    }
    
  } catch (error) {
    console.error('프로필 이미지 조회 중 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '프로필 이미지 조회 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
}

/**
 * 프로필 이미지 삭제
 */
export async function DELETE(req: NextRequest) {
  try {
    await ensureProfileImagesDir();
    
    // URL 파라미터에서 userId 추출
    const { searchParams } = new URL(req.url);
    const userId = searchParams.get('userId');
    
    if (!userId) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '사용자 ID가 필요합니다.'
          }
        }, 
        { status: 400 }
      );
    }
    
    // 사용자의 프로필 이미지 파일 찾기 및 삭제
    const files = await fs.readdir(PROFILE_IMAGES_DIR).catch(() => []);
    let deleted = false;
    
    for (const file of files) {
      if (file.startsWith(`${userId}.`)) {
        await fs.unlink(path.join(PROFILE_IMAGES_DIR, file));
        deleted = true;
      }
    }
    
    if (deleted) {
      return NextResponse.json({
        success: true,
        data: {
          message: '프로필 이미지가 삭제되었습니다.'
        }
      });
    } else {
      return NextResponse.json({
        success: false,
        error: {
          message: '삭제할 프로필 이미지가 없습니다.'
        }
      }, { status: 404 });
    }
    
  } catch (error) {
    console.error('프로필 이미지 삭제 중 오류:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: error instanceof Error ? error.message : '프로필 이미지 삭제 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
} 