import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';

// 파일 기반 저장소 경로
const JOBS_FILE_PATH = path.join(process.cwd(), 'temp', 'background-jobs.json');

// 안전한 cleanup 엔드포인트 - 진행 중인 작업은 보존하고 완료된 작업만 정리
export async function POST(request: NextRequest) {
  try {
    if (!fs.existsSync(JOBS_FILE_PATH)) {
      return NextResponse.json({ 
        status: 'success', 
        message: '정리할 백그라운드 작업 정보가 없습니다.' 
      });
    }

    // 현재 작업 목록 로드
    const data = fs.readFileSync(JOBS_FILE_PATH, 'utf8');
    const jobs = JSON.parse(data);
    
    // 진행 중인 작업만 필터링 (pending, running 상태 보존)
    const activeJobs = jobs.filter((job: any) => 
      job.status === 'pending' || job.status === 'running'
    );
    
    // 완료된 작업 수 계산
    const completedJobs = jobs.filter((job: any) => 
      job.status === 'completed' || job.status === 'failed'
    ).length;
    
    if (activeJobs.length === 0) {
      // 진행 중인 작업이 없으면 파일 전체 삭제
      fs.unlinkSync(JOBS_FILE_PATH);
      console.log('모든 작업이 완료되어 background-jobs.json 파일이 삭제되었습니다.');
    } else {
      // 진행 중인 작업만 남기고 파일 업데이트
      fs.writeFileSync(JOBS_FILE_PATH, JSON.stringify(activeJobs, null, 2));
      console.log(`${completedJobs}개의 완료된 작업이 정리되고, ${activeJobs.length}개의 진행 중인 작업이 보존되었습니다.`);
    }
    
    return NextResponse.json({ 
      status: 'success', 
      message: `백그라운드 작업 정보가 안전하게 정리되었습니다. (${completedJobs}개 정리, ${activeJobs.length}개 보존)`,
      cleaned: completedJobs,
      preserved: activeJobs.length
    });
  } catch (error) {
    console.error('백그라운드 작업 정보 안전 정리 실패:', error);
    return NextResponse.json(
      { 
        status: 'error', 
        message: '백그라운드 작업 정보 정리 중 오류가 발생했습니다.' 
      },
      { status: 500 }
    );
  }
}
