import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs/promises';
import path from 'path';
import { getApiServerUrl } from '@/config/serverConfig';

// 인증 헤더 가져오기 (서버 사이드용)
function getAuthHeaders(req?: NextRequest): Record<string, string> {
  const headers: Record<string, string> = {
    'Content-Type': 'application/json'
  };

  if (req) {
    // 요청에서 인증 정보 추출
    const apiKey = req.headers.get('X-API-Key');
    const authHeader = req.headers.get('Authorization');

    if (apiKey) {
      headers['X-API-Key'] = apiKey;
    } else if (authHeader) {
      headers['Authorization'] = authHeader;
      // JWT 토큰을 X-API-Key로도 전달 (API 서버 호환성)
      if (authHeader.startsWith('Bearer ')) {
        headers['X-API-Key'] = authHeader.substring(7);
      }
    }
  }

  return headers;
}

// RAG 문서 경로 가져오기 함수 (API 서버에서 조회)
async function getRagDocsPath(req?: NextRequest): Promise<string> {
  try {
    // API 서버 URL 가져오기 (중앙 설정 사용)
    const apiServerUrl = getApiServerUrl();
    const authHeaders = getAuthHeaders(req);

    const response = await fetch(`${apiServerUrl}/rag/docs-path`, {
      method: 'GET',
      headers: authHeaders
    });

    if (response.ok) {
      const data = await response.json();
      if (data.success && data.data?.path) {
        console.log('[RAG_DOCS_PATH] Retrieved from API server:', data.data.path);
        return data.data.path;
      }
    } else {
      console.warn('[RAG_DOCS_PATH] API response not ok:', response.status, response.statusText);
    }
  } catch (error) {
    console.error('[RAG_DOCS_PATH] Error fetching from API server:', error);
  }

  // 폴백: 기본 경로 사용
  const defaultPath = '/home/hamonikr/문서/rag-docs';
  console.log('[RAG_DOCS_PATH] Using default path:', defaultPath);
  return defaultPath;
}

// 문서 목록 조회 (파일 시스템 기반)
export async function GET(req: NextRequest) {
  try {
    // RAG 문서 경로 가져오기
    const ragDocsPath = await getRagDocsPath(req);
    const supportFolderPath = path.join(ragDocsPath, 'support');

    console.log('[SUPPORT_DOCS_GET] Support folder path:', supportFolderPath);

    // support 폴더가 존재하는지 확인하고, 없으면 생성
    try {
      await fs.access(supportFolderPath);
    } catch (error) {
      console.log('[SUPPORT_DOCS_GET] Support folder does not exist, creating...');
      await fs.mkdir(supportFolderPath, { recursive: true });
    }

    // support 폴더의 모든 파일 목록 가져오기
    const files = await fs.readdir(supportFolderPath);

    // 파일 정보 수집
    const documents = await Promise.all(
      files.map(async (filename) => {
        const filePath = path.join(supportFolderPath, filename);

        try {
          const stats = await fs.stat(filePath);

          // 파일인지 확인 (디렉토리 제외)
          if (!stats.isFile()) {
            return null;
          }

          // 파일 확장자로 MIME 타입 추정
          const ext = path.extname(filename).toLowerCase();
          let mimetype = 'application/octet-stream';

          if (ext === '.pdf') mimetype = 'application/pdf';
          else if (ext === '.txt') mimetype = 'text/plain';
          else if (ext === '.md') mimetype = 'text/markdown';
          else if (ext === '.doc') mimetype = 'application/msword';
          else if (ext === '.docx') mimetype = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
          else if (ext === '.xls') mimetype = 'application/vnd.ms-excel';
          else if (ext === '.xlsx') mimetype = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
          else if (ext === '.jpg' || ext === '.jpeg') mimetype = 'image/jpeg';
          else if (ext === '.png') mimetype = 'image/png';

          return {
            id: `file_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`, // 임시 ID 생성
            filename: filename,
            filepath: supportFolderPath,
            filesize: stats.size,
            mimetype: mimetype,
            user_id: 'support',
            created_at: stats.birthtime.toISOString(),
            updated_at: stats.mtime.toISOString(),
            file_exists: true
          };
        } catch (error) {
          console.error(`[SUPPORT_DOCS_GET] Error reading file ${filename}:`, error);
          return null;
        }
      })
    );

    // null 값 제거하고 생성일 기준으로 정렬
    const validDocuments = documents.filter((doc) => doc !== null).sort((a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime());

    console.log(`[SUPPORT_DOCS_GET] Found ${validDocuments.length} documents in support folder`);

    return NextResponse.json({ success: true, data: validDocuments });
  } catch (error) {
    console.error('[SUPPORT_DOCS_GET_ERROR]', error);
    return NextResponse.json(
      {
        success: false,
        error: 'Internal Server Error',
        details: error instanceof Error ? error.message : 'Unknown error'
      },
      { status: 500 }
    );
  }
}

// 문서 정보 저장 (파일 시스템 기반)
export async function POST(req: NextRequest) {
  try {
    const { filename, filepath, filesize, mimetype, user_id } = await req.json();

    if (!filename || !filepath) {
      return NextResponse.json({ success: false, error: 'Filename and filepath are required.' }, { status: 400 });
    }

    // RAG 문서 경로 가져오기
    const ragDocsPath = await getRagDocsPath(req);
    const supportFolderPath = path.join(ragDocsPath, 'support');

    // support 폴더가 존재하는지 확인하고, 없으면 생성
    try {
      await fs.access(supportFolderPath);
    } catch (error) {
      console.log('[SUPPORT_DOCS_POST] Support folder does not exist, creating...');
      await fs.mkdir(supportFolderPath, { recursive: true });
    }

    // 동일한 파일명이 있는지 확인하고 삭제
    const targetFilePath = path.join(supportFolderPath, filename);
    try {
      await fs.access(targetFilePath);
      console.log(`[SUPPORT_DOCS_DUPLICATE_REMOVAL] Found duplicate file: ${filename}`);
      await fs.unlink(targetFilePath);
      console.log(`[SUPPORT_DOCS_DUPLICATE_REMOVAL] Deleted duplicate file: ${filename}`);
    } catch (error) {
      // 파일이 존재하지 않으면 무시
      console.log(`[SUPPORT_DOCS_DUPLICATE_REMOVAL] No duplicate file found: ${filename}`);
    }

    // 파일을 support 폴더로 복사
    const sourceFilePath = path.join(filepath, filename);
    try {
      await fs.copyFile(sourceFilePath, targetFilePath);
      console.log(`[SUPPORT_DOCS_COPY] File copied to support folder: ${filename}`);
    } catch (error) {
      console.error(`[SUPPORT_DOCS_COPY_ERROR] Failed to copy file: ${filename}`, error);
      return NextResponse.json({ success: false, error: 'Failed to copy file to support folder.' }, { status: 500 });
    }

    // 파일 정보 반환
    const stats = await fs.stat(targetFilePath);
    const documentData = {
      id: `file_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      filename: filename,
      filepath: supportFolderPath,
      filesize: stats.size,
      mimetype: mimetype || 'application/octet-stream',
      user_id: 'support',
      created_at: stats.birthtime.toISOString(),
      updated_at: stats.mtime.toISOString(),
      file_exists: true
    };

    console.log(`[SUPPORT_DOCS_POST] Successfully saved document: ${filename}`);
    return NextResponse.json({ success: true, data: documentData });
  } catch (error) {
    console.error('[SUPPORT_DOCS_POST_ERROR]', error);
    return NextResponse.json({ success: false, error: 'Internal Server Error' }, { status: 500 });
  }
}

// 문서 삭제 (파일 시스템 기반)
export async function DELETE(req: NextRequest) {
  try {
    // 토큰에서 직접 사용자 정보 확인
    const token = req.cookies.get('auth_token')?.value;
    console.log('[SUPPORT_DOCS_DELETE] Token exists:', !!token);

    if (!token) {
      return NextResponse.json({ success: false, error: 'Authentication token is required.' }, { status: 401 });
    }

    const { id, filename } = await req.json();

    if (!filename) {
      return NextResponse.json({ success: false, error: 'Filename is required.' }, { status: 400 });
    }

    // RAG 문서 경로 가져오기
    const ragDocsPath = await getRagDocsPath(req);
    const supportFolderPath = path.join(ragDocsPath, 'support');

    // 와일드카드 지원: filename이 '*'인 경우 모든 파일 삭제
    if (filename === '*') {
      try {
        const files = await fs.readdir(supportFolderPath);
        let deletedCount = 0;

        for (const file of files) {
          const filePath = path.join(supportFolderPath, file);
          try {
            const stats = await fs.stat(filePath);
            if (stats.isFile()) {
              await fs.unlink(filePath);
              console.log(`[SUPPORT_DOCS_DELETE] Deleted file: ${file}`);
              deletedCount++;
            }
          } catch (error) {
            console.error(`[SUPPORT_DOCS_DELETE] Error deleting file: ${file}`, error);
          }
        }

        console.log(`[SUPPORT_DOCS_DELETE] Deleted ${deletedCount} files from support folder`);
        return NextResponse.json({ success: true, message: `Deleted ${deletedCount} files` });
      } catch (error) {
        console.error('[SUPPORT_DOCS_DELETE_ERROR]', error);
        return NextResponse.json({ success: false, error: 'Failed to delete files' }, { status: 500 });
      }
    } else {
      // 특정 파일 삭제
      const targetFilePath = path.join(supportFolderPath, filename);

      try {
        // API에서 파일 삭제를 처리하므로, 응용프로그램에서는 파일 삭제 처릭가 불필요, 
        // 하지만, 만약을 위해 파일이 있으면 삭제, 파일이 없으면 에러 없이 넘어감
        await fs.rm(targetFilePath, { force: true });

        console.log(`[SUPPORT_DOCS_DELETE] deleted if existed: ${filename}`);
        return NextResponse.json({
          success: true,
          message: `Deleted if existed: ${filename}`
        });

      } catch (error) {
        if (error instanceof Error && 'code' in error && error.code === 'ENOENT') {
          return NextResponse.json({ success: false, error: 'File not found.' }, { status: 404 });
        }
        console.error(`[SUPPORT_DOCS_DELETE_ERROR] Failed to delete file: ${filename}`, error);
        return NextResponse.json({ success: false, error: 'Failed to delete file' }, { status: 500 });
      }
    }
  } catch (error) {
    console.error('[SUPPORT_DOCS_DELETE_ERROR]', error);
    return NextResponse.json({ success: false, error: 'Internal Server Error' }, { status: 500 });
  }
}
