'use client';

import React, { useState, useRef, useCallback } from 'react';
import ReactFlow, {
  ReactFlowProvider,
  useReactFlow,
  Node,
  Edge,
  useNodesState,
  useEdgesState,
  Controls,
  Background,
} from 'reactflow';
import 'reactflow/dist/style.css';

// 간단한 사이드바 컴포넌트
const Sidebar = () => {
  const onDragStart = (event: React.DragEvent, nodeType: string) => {
    console.log('🔥 Sidebar drag start:', nodeType);
    event.dataTransfer.setData('application/reactflow', nodeType);
    event.dataTransfer.effectAllowed = 'move';
  };

  return (
    <aside style={{ width: '200px', padding: '10px', background: '#f0f0f0' }}>
      <div style={{ marginBottom: '10px' }}>드래그해서 노드 추가:</div>
      <div
        onDragStart={(event) => onDragStart(event, 'input')}
        draggable
        style={{
          padding: '10px',
          background: 'blue',
          color: 'white',
          marginBottom: '5px',
          cursor: 'grab',
        }}
      >
        입력 노드
      </div>
      <div
        onDragStart={(event) => onDragStart(event, 'default')}
        draggable
        style={{
          padding: '10px',
          background: 'green',
          color: 'white',
          cursor: 'grab',
        }}
      >
        기본 노드
      </div>
    </aside>
  );
};

// React Flow 컴포넌트
const DnDFlow = () => {
  const reactFlowWrapper = useRef(null);
  const [nodes, setNodes, onNodesChange] = useNodesState([]);
  const [edges, setEdges, onEdgesChange] = useEdgesState([]);
  const { screenToFlowPosition } = useReactFlow();

  // 노드 연결 핸들러 추가
  const onConnect = useCallback((params) => {
    console.log('🔗 Connection created:', params);
    setEdges((eds) => eds.concat({ ...params, id: `edge-${Date.now()}` }));
  }, [setEdges]);

  const onDragOver = useCallback((event: React.DragEvent) => {
    event.preventDefault();
    event.dataTransfer.dropEffect = 'move';
    console.log('🔥 Drag over');
  }, []);

  const onDrop = useCallback(
    (event: React.DragEvent) => {
      event.preventDefault();
      console.log('🔥 Drop event triggered');

      const type = event.dataTransfer.getData('application/reactflow');
      console.log('🔥 Node type:', type);

      // 타입이 없으면 리턴
      if (typeof type === 'undefined' || !type) {
        console.log('🔥 No type found');
        return;
      }

      const position = screenToFlowPosition({
        x: event.clientX,
        y: event.clientY,
      });
      console.log('🔥 Position:', position);

      const newNode: Node = {
        id: `${type}_${Date.now()}`,
        type: 'default',
        position,
        data: { label: `${type} 노드` },
      };

      console.log('🔥 Adding node:', newNode);
      setNodes((nds) => nds.concat(newNode));
    },
    [screenToFlowPosition, setNodes]
  );

  return (
    <div style={{ width: '100%', height: '500px', display: 'flex' }}>
      <Sidebar />
      <div style={{ flex: 1 }} ref={reactFlowWrapper}>
        <ReactFlow
          nodes={nodes}
          edges={edges}
          onNodesChange={onNodesChange}
          onEdgesChange={onEdgesChange}
          onConnect={onConnect}
          onDrop={onDrop}
          onDragOver={onDragOver}
        >
          <Controls />
          <Background />
        </ReactFlow>
      </div>
    </div>
  );
};

export default function SimpleFlowTest() {
  return (
    <ReactFlowProvider>
      <DnDFlow />
    </ReactFlowProvider>
  );
}