import { useState, useEffect, useCallback } from "react";
import Link from "next/link";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/src/components/ui/dialog";
import { Button } from "@/src/components/ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/src/components/ui/select";
import { Label } from "@/src/components/ui/label";
import { api } from "@/src/utils/api";
import { CopyIcon, ExternalLinkIcon } from "lucide-react";

type PromptSelectionDialogProps = {
  isOpen: boolean;
  onClose: () => void;
  onSelect?: (tag: string) => void;
  projectId: string;
};

export function PromptSelectionDialog({
  isOpen,
  onClose,
  onSelect,
  projectId,
}: PromptSelectionDialogProps) {
  const [selectedPromptName, setSelectedPromptName] = useState<string>("");
  const [selectedTag, setSelectedTag] = useState<string>("");
  const [selectionType, setSelectionType] = useState<"version" | "label">(
    "label",
  );
  const [selectedVersionOrLabel, setSelectedVersionOrLabel] =
    useState<string>("");

  const copySelectedTag = useCallback(() => {
    navigator.clipboard.writeText(selectedTag);
  }, [selectedTag]);

  useEffect(() => {
    if (isOpen) {
      setSelectedPromptName("");
      setSelectedTag("");
      setSelectionType("label");
      setSelectedVersionOrLabel("");
    }
  }, [isOpen]);

  useEffect(() => {
    if (selectedPromptName && selectedVersionOrLabel) {
      if (selectionType === "version") {
        setSelectedTag(
          `@@@langfusePrompt:name=${selectedPromptName}|version=${selectedVersionOrLabel}@@@`,
        );
      } else {
        setSelectedTag(
          `@@@langfusePrompt:name=${selectedPromptName}|label=${selectedVersionOrLabel}@@@`,
        );
      }
    } else {
      setSelectedTag("");
    }
  }, [selectedPromptName, selectedVersionOrLabel, selectionType]);

  const { data: promptOptions } = api.prompts.getPromptLinkOptions.useQuery(
    {
      projectId,
    },
    {
      enabled: Boolean(projectId),
      refetchOnMount: false,
      refetchOnWindowFocus: false,
      refetchOnReconnect: false,
      staleTime: Infinity,
    },
  );

  const selectedPrompt = promptOptions?.find(
    (option) => option.name === selectedPromptName,
  );

  const handleConfirm = useCallback(() => {
    if (!selectedTag) return;
    if (onSelect) {
      onSelect(selectedTag);
    } else {
      copySelectedTag();
    }
    onClose();
  }, [copySelectedTag, selectedTag, onSelect, onClose]);

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle>Add inline prompt reference</DialogTitle>
        </DialogHeader>
        <div className="flex flex-col gap-4 py-4">
          <p className="text-sm text-muted-foreground">
            Referenced prompts are dynamically resolved and inserted when
            fetched via API/SDK. This enables modular design—create complex
            prompts from reusable, independently maintained components.
          </p>

          <div className="flex flex-col gap-2">
            <Label htmlFor="prompt-name">Prompt name</Label>
            <Select
              value={selectedPromptName}
              onValueChange={(value) => {
                setSelectedPromptName(value);
                setSelectedVersionOrLabel("");
              }}
            >
              <SelectTrigger id="prompt-name">
                <SelectValue placeholder="Select a text prompt" />
              </SelectTrigger>
              <SelectContent>
                {promptOptions?.map((prompt) => (
                  <SelectItem key={prompt.name} value={prompt.name}>
                    {prompt.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            <p className="text-xs text-muted-foreground">
              Only text prompts can be referenced inline.
            </p>
          </div>

          {selectedPromptName && (
            <div className="flex flex-col gap-2">
              <Label htmlFor="selection-type">Reference by</Label>
              <Select
                value={selectionType}
                onValueChange={(value: "version" | "label") => {
                  setSelectionType(value);
                  setSelectedVersionOrLabel("");
                }}
              >
                <SelectTrigger id="selection-type">
                  <SelectValue placeholder="Select link type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="label">Label</SelectItem>
                  <SelectItem value="version">Version</SelectItem>
                </SelectContent>
              </Select>
            </div>
          )}

          {selectedPromptName && (
            <div className="flex flex-col gap-2">
              <Label htmlFor="version-or-label">
                {selectionType === "version" ? "Version" : "Label"}
              </Label>
              <div className="flex gap-2">
                <Select
                  value={selectedVersionOrLabel}
                  onValueChange={setSelectedVersionOrLabel}
                >
                  <SelectTrigger id="version-or-label">
                    <SelectValue
                      placeholder={
                        selectionType === "version"
                          ? "Select a version"
                          : "Select a label"
                      }
                    />
                  </SelectTrigger>
                  <SelectContent>
                    {selectionType === "version"
                      ? selectedPrompt?.versions.map((version) => (
                          <SelectItem
                            key={version.toString()}
                            value={version.toString()}
                          >
                            {version}
                          </SelectItem>
                        ))
                      : selectedPrompt?.labels.map((label) => (
                          <SelectItem key={label} value={label}>
                            {label}
                          </SelectItem>
                        ))}
                  </SelectContent>
                </Select>
                {selectedVersionOrLabel && (
                  <Link
                    href={`/project/${projectId}/prompts/${selectedPromptName}?${selectionType}=${selectedVersionOrLabel}`}
                    target="_blank"
                    passHref
                  >
                    <Button type="button" variant="outline" size="icon">
                      <ExternalLinkIcon className="h-4 w-4" />
                    </Button>
                  </Link>
                )}
              </div>
            </div>
          )}
        </div>

        {selectedTag && (
          <div className="space-y-2">
            <Label>Tag preview</Label>
            <div className="relative">
              <div className="rounded-md border bg-muted p-3 pr-10 font-mono text-xs">
                {selectedTag}
              </div>
              <Button
                type="button"
                variant="ghost"
                size="icon"
                className="absolute right-2 top-2 bg-opacity-70"
                onClick={copySelectedTag}
              >
                <CopyIcon className="h-4 w-4" />
              </Button>
            </div>
            <p className="text-xs text-muted-foreground">
              {onSelect
                ? "This tag will be inserted into the prompt content."
                : "This tag will be copied to clipboard to be then inserted into the prompt"}
            </p>
          </div>
        )}

        <DialogFooter>
          <Button type="button" variant="outline" onClick={onClose}>
            Cancel
          </Button>
          <Button type="button" onClick={handleConfirm} disabled={!selectedTag}>
            {onSelect ? "Insert" : "Copy and close"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
