import { requireAuth } from './auth.js';

export async function promptPage(chatMessages, translations, reqAPI, getText, showCustomDialog, makeEditor) {
    // 사용자 인증 체크
    const hasAccess = await requireAuth(reqAPI);
    if (!hasAccess) {
        return;
    }

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    inputContainer.style.display = 'none';

    // 현재 프롬프트 내용과 원본 프롬프트 템플릿 가져오기
    const [currentPrompt, originalPrompt] = await Promise.all([
        reqAPI('readPrompt'),
        reqAPI('readOriginalPrompt')
    ]);

    chatMessages.innerHTML = `
        <div class="page-container">
            <div class="page-header">
                <h2 class="page-title">${getText('promptEditor', 'title')}</h2>
                <p class="page-description">${getText('promptEditor', 'description')}</p>
                <div class="page-actions">
                    <button id="resetPrompt" class="btn-execute" data-key="resetToDefault" data-section="promptEditor">${getText('promptEditor', 'resetToDefault')}</button>
                    <button id="applyRagPrompt" class="btn-execute" data-key="applyRagPrompt" data-section="promptEditor">${getText('promptEditor', 'applyRagPrompt')}</button>
                </div>
            </div>
            <div class="prompt-editor-container" id="promptEditor"></div>
            <div class="page-actions" style="margin-top: 20px;">
                <button id="savePrompt" class="btn-execute" data-key="save" data-section="promptEditor">${getText('promptEditor', 'save')}</button>
            </div>
        </div>
    `;

    // 프롬프트 에디터 생성
    const editor = makeEditor(
        document.getElementById('promptEditor'),
        currentPrompt,
        'markdown',
        false
    );

    // CodeMirror 에디터 크기 조정
    editor.setSize('100%', '100%');

    // 저장 버튼 이벤트
    document.getElementById('savePrompt').addEventListener('click', async () => {
        try {
            console.log('Save button clicked');
            const newContent = editor.getValue();
            console.log('Content length:', newContent?.length);

            if (!newContent || newContent.trim() === '') {
                console.log('Empty content detected');
                const dialog = document.createElement('div');
                dialog.className = 'custom-dialog';
                dialog.innerHTML = `
                    <div class="dialog-content">
                        <h3>${getText('dialog', 'errorTitle')}</h3>
                        <p>${getText('promptEditor', 'emptyPromptError')}</p>
                        <div class="dialog-buttons">
                            <button class="btn-ok">${getText('common', 'ok')}</button>
                        </div>
                    </div>
                `;
                document.body.appendChild(dialog);
                await new Promise(resolve => {
                    dialog.querySelector('.btn-ok').onclick = () => {
                        dialog.remove();
                        resolve();
                    };
                });
                return;
            }

            console.log('Saving content...');
            const saveResult = await reqAPI('savePrompt', { content: newContent });
            console.log('Save result:', saveResult);
            
            if (!saveResult) {
                throw new Error(getText('promptEditor', 'saveError'));
            }

            const dialog = document.createElement('div');
            dialog.className = 'custom-dialog';
            dialog.innerHTML = `
                <div class="dialog-content">
                    <h3>${getText('dialog', 'successTitle')}</h3>
                    <p>${getText('promptEditor', 'saveSuccess')}</p>
                    <div class="dialog-buttons">
                        <button class="btn-ok">${getText('common', 'ok')}</button>
                    </div>
                </div>
            `;
            document.body.appendChild(dialog);
            await new Promise(resolve => {
                dialog.querySelector('.btn-ok').onclick = () => {
                    dialog.remove();
                    resolve();
                };
            });
        } catch (error) {
            console.error('Save prompt error:', error);
            // error.message가 객체인 경우를 처리
            let errorMessage = error?.message;
            if (typeof errorMessage === 'object') {
                errorMessage = JSON.stringify(errorMessage);
            }
            const dialog = document.createElement('div');
            dialog.className = 'custom-dialog';
            dialog.innerHTML = `
                <div class="dialog-content">
                    <h3>${getText('dialog', 'errorTitle')}</h3>
                    <p>${getText('promptEditor', 'saveError')}: ${errorMessage || 'Unknown error'}</p>
                    <div class="dialog-buttons">
                        <button class="btn-ok">${getText('common', 'ok')}</button>
                    </div>
                </div>
            `;
            document.body.appendChild(dialog);
            await new Promise(resolve => {
                dialog.querySelector('.btn-ok').onclick = () => {
                    dialog.remove();
                    resolve();
                };
            });
        }
    });

    // 초기화 버튼 이벤트
    document.getElementById('resetPrompt').addEventListener('click', async () => {
        const confirmed = await showCustomDialog({
            title: getText('dialog', 'resetPromptTitle'),
            message: getText('dialog', 'resetPromptMessage'),
            okText: getText('dialog', 'resetPromptOk'),
            cancelText: getText('dialog', 'resetPromptCancel')
        });

        if (confirmed) {
            try {
                // 프롬프트 초기화
                await reqAPI('resetPrompt');
                
                // 새로운 프롬프트 내용 읽기
                const newPrompt = await reqAPI('readPrompt');
                
                // 에디터 내용 업데이트
                editor.setValue(newPrompt);
                
                await showCustomDialog({
                    title: getText('dialog', 'successTitle'),
                    message: getText('promptEditor', 'resetSuccess'),
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
            } catch (error) {
                await showCustomDialog({
                    title: getText('dialog', 'errorTitle'),
                    message: getText('promptEditor', 'resetError') + ': ' + error.message,
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
            }
        }
    });

    // Chatbot 모드 전환 버튼 이벤트   
    document.getElementById('applyRagPrompt').addEventListener('click', async () => {
        const confirmed = await showCustomDialog({
            title: getText('dialog', 'applyRagPromptTitle'),
            message: getText('dialog', 'applyRagPromptMessage'),
            okText: getText('dialog', 'applyRagPromptOk'),
            cancelText: getText('dialog', 'applyRagPromptCancel')
        });

        if (confirmed) {
            try {
                // 프롬프트 초기화
                await reqAPI('resetRagPrompt');
                
                // 새로운 프롬프트 내용 읽기
                const newPrompt = await reqAPI('readRagPrompt');
                
                // 에디터 내용 업데이트
                editor.setValue(newPrompt);
                
                await showCustomDialog({
                    title: getText('dialog', 'successTitle'),
                    message: getText('promptEditor', 'applyRagPromptSuccess'),
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
            } catch (error) {
                await showCustomDialog({
                    title: getText('dialog', 'errorTitle'),
                    message: getText('promptEditor', 'applyRagPromptError') + ': ' + error.message,
                    type: 'alert',
                    okText: getText('common', 'ok')
                });
            }
        }
    });
} 