import bcrypt from "bcrypt";
import jwt from "jsonwebtoken";
import dbManager from "../database/index.js";
import crypto from "crypto";
import ActivityLogger from "./ActivityLogger.js";
import { redisClient } from "../../queues/index.js";

class AuthService {
    constructor() {
        this.secretKey = process.env.JWT_SECRET || "your-secret-key";
        this.activityLogger = new ActivityLogger();
    }

    async createUser(userData) {
        const { username, email, password, name, role = "user" } = userData;

        // 트랜잭션 시작
        await dbManager.query("BEGIN");

        try {
            // 아이디 중복 체크
            const existingUser = await dbManager.query("SELECT * FROM users WHERE username = $1", [username]);

            if (existingUser.rows.length > 0) {
                throw new Error("이미 사용 중인 아이디입니다.");
            }

            // 이메일이 제공된 경우 중복 체크
            if (email) {
                const existingEmail = await dbManager.query("SELECT * FROM users WHERE email = $1", [email]);

                if (existingEmail.rows.length > 0) {
                    throw new Error("이미 등록된 이메일입니다.");
                }
            }

            // 비밀번호 해시화
            const passwordHash = await bcrypt.hash(password, 10);

            // 사용자 생성
            const result = await dbManager.query(
                "INSERT INTO users (username, email, password_hash, name, role, language) VALUES ($1, $2, $3, $4, $5, $6) RETURNING *",
                [username, email, passwordHash, name, role, 'ko']
            );

            const user = result.rows[0];

            // API 키 자동 생성
            const permissions = {
                reports: {
                    read: true,
                    write: true
                },
                users: {
                    read: true
                }
            };

            // API 키 생성 (16바이트 랜덤 문자열)
            const randomKey = crypto.randomBytes(16).toString("hex");
            const keyValue = `airun_${user.id}_${randomKey}`;

            const apiKeyResult = await dbManager.query(
                `INSERT INTO api_keys (
                    user_id, 
                    name, 
                    key_value, 
                    permissions, 
                    rate_limit, 
                    status
                ) VALUES ($1, $2, $3, $4, $5, $6) RETURNING *`,
                [user.id, `기본 API 키`, keyValue, permissions, 1000, "active"]
            );

            // 활동 로그 기록
            await this.activityLogger.logActivity(user.id, "USER_CREATED", {
                username: user.username,
                email: user.email,
                role: user.role,
                api_key_created: true
            });

            // 트랜잭션 커밋
            await dbManager.query("COMMIT");

            delete user.password_hash;
            return {
                user,
                apiKey: apiKeyResult.rows[0]
            };
        } catch (error) {
            // 오류 발생 시 롤백
            await dbManager.query("ROLLBACK");
            throw error;
        }
    }

    async authenticateUser(username, password, ipAddress = null) {
        const result = await dbManager.query("SELECT * FROM users WHERE username = $1", [username]);

        if (result.rows.length === 0) {
            throw new Error("사용자를 찾을 수 없습니다.");
        }

        const user = result.rows[0];
        const validPassword = await bcrypt.compare(password, user.password_hash);

        if (!validPassword) {
            throw new Error("잘못된 비밀번호입니다.");
        }

        // 마지막 로그인 시간 업데이트
        await dbManager.query("UPDATE users SET last_login = CURRENT_TIMESTAMP WHERE id = $1", [user.id]);

        // IP 주소 로그 기록 (IP 주소가 없는 경우 'unknown'으로 기록)
        await dbManager.query("INSERT INTO user_ip_logs (user_id, ip_address) VALUES ($1, $2)", [user.id, ipAddress || "unknown"]);

        // 활동 로그에도 IP 정보 기록
        await this.activityLogger.logActivity(
            user.id,
            "USER_LOGIN",
            {
                username: user.username,
                ip_address: ipAddress || "unknown"
            },
            ipAddress
        );

        // JWT 토큰 생성
        const token = jwt.sign(
            {
                id: user.id,
                username: user.username,
                email: user.email,
                role: user.role
            },
            this.secretKey,
            { expiresIn: "24h" }
        );

        // API 키 조회
        const apiKeysQuery = `
            SELECT 
                id,
                name,
                key_value as key,
                status,
                created_at,
                last_used,
                permissions
            FROM api_keys 
            WHERE user_id = $1 AND status = 'active'
        `;

        const apiKeysResult = await dbManager.query(apiKeysQuery, [user.id]);
        const apiKeys = apiKeysResult.rows;

        delete user.password_hash;
        return {
            user,
            token,
            apiKeys
        };
    }

    async getUser(userId) {
        try {
            const query = `
                SELECT 
                    u.id,
                    u.username,
                    u.email,
                    u.name,
                    u.role,
                    u.status,
                    u.created_at,
                    u.last_login,
                    u.language,
                    COALESCE(
                        (
                            SELECT json_agg(
                                json_build_object(
                                    'id', ak.id,
                                    'key', ak.key_value,
                                    'name', ak.name,
                                    'status', ak.status,
                                    'createdAt', ak.created_at
                                )
                            )
                            FROM api_keys ak 
                            WHERE ak.user_id = u.id
                        ),
                        '[]'::json
                    ) as "apiKeys"
                FROM users u
                WHERE u.id = $1
            `;

            const result = await dbManager.query(query, [userId]);
            if (!result.rows[0]) {
                throw new Error("사용자를 찾을 수 없습니다.");
            }
            return result.rows[0];
        } catch (error) {
            console.error("사용자 조회 중 오류:", error);
            throw new Error("사용자 정보를 조회할 수 없습니다.");
        }
    }

    // 토큰을 블랙리스트에 추가
    async addToBlacklist(token) {
        try {
            const decoded = jwt.decode(token);
            if (!decoded || !decoded.exp) {
                throw new Error("유효하지 않은 토큰입니다.");
            }

            // 토큰의 남은 유효 시간 계산
            const timeToExpiry = decoded.exp - Math.floor(Date.now() / 1000);
            if (timeToExpiry > 0) {
                // 토큰의 남은 유효 시간만큼만 블랙리스트에 저장
                await redisClient.setex(`auth:bl_${token}`, timeToExpiry, "true");
                return true;
            }
            return false;
        } catch (error) {
            console.error("토큰 블랙리스트 추가 실패:", error);
            throw error;
        }
    }

    // 토큰이 블랙리스트에 있는지 확인
    async isBlacklisted(token) {
        try {
            const exists = await redisClient.exists(`auth:bl_${token}`);
            return exists === 1;
        } catch (error) {
            console.error("토큰 블랙리스트 확인 실패:", error);
            throw error;
        }
    }

    async verifyToken(token) {
        try {
            // 먼저 토큰이 블랙리스트에 있는지 확인 (Redis 연결 실패 시 건너뛰기)
            try {
                const isBlacklisted = await this.isBlacklisted(token);
                if (isBlacklisted) {
                    throw new Error("로그아웃된 토큰입니다.");
                }
            } catch (redisError) {
                console.warn("Redis 연결 실패, 블랙리스트 확인을 건너뜁니다:", redisError.message);
                // Redis 연결 실패 시에도 토큰 검증은 계속 진행
            }

            const decoded = jwt.verify(token, this.secretKey);

            // 데이터베이스에서 사용자 확인 (재시도 로직 추가)
            let result;
            let retryCount = 0;
            const maxRetries = 3;
            
            while (retryCount < maxRetries) {
                try {
                    result = await dbManager.query("SELECT id, username, email, role, status FROM users WHERE id = $1", [decoded.id]);
                    break;
                } catch (dbError) {
                    retryCount++;
                    console.warn(`데이터베이스 쿼리 실패 (${retryCount}/${maxRetries}):`, dbError.message);
                    if (retryCount >= maxRetries) {
                        throw new Error("데이터베이스 연결 실패");
                    }
                    // 1초 대기 후 재시도
                    await new Promise(resolve => setTimeout(resolve, 1000));
                }
            }

            if (result.rows.length === 0) {
                throw new Error("사용자를 찾을 수 없습니다.");
            }

            const user = result.rows[0];
            if (user.username !== decoded.username || user.role !== decoded.role) {
                throw new Error("토큰 정보가 일치하지 않습니다.");
            }

            // 사용자 상태가 active가 아닌 경우 별도의 에러 메시지
            if (user.status !== "active") {
                throw new Error("비활성화된 계정입니다.");
            }

            return decoded;
        } catch (error) {
            console.error("토큰 검증 오류:", error);
            throw new Error("유효하지 않은 토큰입니다.");
        }
    }

    async logout(token) {
        try {
            // 토큰을 블랙리스트에 추가
            await this.addToBlacklist(token);
            return { success: true, message: "로그아웃되었습니다." };
        } catch (error) {
            console.error("로그아웃 처리 실패:", error);
            throw error;
        }
    }

    async updateUser(userId, updates) {
        try {
            const updateFields = [];
            const values = [];
            let paramCount = 1;

            // 업데이트 가능한 필드 정의
            const allowedUpdates = ["name", "email", "role", "password", "language"];

            // [업데이트할 필드 구성]
            // 비밀번호가 있는 경우 해시화 후 필드에 추가
            if (updates.password) {
                updates.password_hash = await bcrypt.hash(updates.password, 10);
                updateFields.push(`password_hash = $${paramCount}`);
                values.push(updates.password_hash);
                paramCount++;
                delete updates.password; 
            }

            // 나머지 업데이트 필드 구성
            for (const [key, value] of Object.entries(updates)) {
                if (allowedUpdates.includes(key) && value !== undefined && key !== "password") {
                    updateFields.push(`${key} = $${paramCount}`);
                    values.push(value);
                    paramCount++;
                }
            }

            if (updateFields.length === 0) {
                throw new Error("업데이트할 필드가 없습니다.");
            }

            values.push(userId);
            const query = `
                UPDATE users 
                SET ${updateFields.join(", ")}
                WHERE id = $${paramCount} 
                RETURNING id, username, email, name, role, status, created_at, last_login, language
            `;

            console.log("[DEBUG] Update query:", query);
            console.log("[DEBUG] Update values:", values);

            const result = await dbManager.query(query, values);

            if (result.rows.length === 0) {
                throw new Error("사용자를 찾을 수 없습니다.");
            }

            return result.rows[0];
        } catch (error) {
            console.error("[DEBUG] Error in updateUser:", error);
            throw error;
        }
    }

    async changePassword(userId, currentPassword, newPassword) {
        const result = await dbManager.query("SELECT * FROM users WHERE id = $1", [userId]);

        if (result.rows.length === 0) {
            throw new Error("사용자를 찾을 수 없습니다.");
        }

        const user = result.rows[0];
        const validPassword = await bcrypt.compare(currentPassword, user.password_hash);

        if (!validPassword) {
            throw new Error("현재 비밀번호가 일치하지 않습니다.");
        }

        const newPasswordHash = await bcrypt.hash(newPassword, 10);
        await dbManager.query("UPDATE users SET password_hash = $1 WHERE id = $2", [newPasswordHash, userId]);

        return { message: "비밀번호가 성공적으로 변경되었습니다." };
    }

    async resetPassword(username) {
        const result = await dbManager.query("SELECT * FROM users WHERE username = $1", [username]);

        if (result.rows.length === 0) {
            throw new Error("사용자를 찾을 수 없습니다.");
        }

        // 임시 비밀번호 생성
        const tempPassword = Math.random().toString(36).slice(-12);
        const passwordHash = await bcrypt.hash(tempPassword, 10);

        await dbManager.query("UPDATE users SET password_hash = $1 WHERE username = $2", [passwordHash, username]);

        return {
            message: "임시 비밀번호가 생성되었습니다.",
            tempPassword
        };
    }

    async listUsers(options = {}) {
        const { page = 1, limit = 10, sortBy = "created_at", sortOrder = "DESC", status, role, search } = options;

        const offset = (page - 1) * limit;
        const values = [];
        let paramCount = 1;

        let query = `
            SELECT 
                u.id, 
                u.username,
                u.email, 
                u.name, 
                u.role, 
                u.status, 
                u.created_at, 
                u.last_login,
                u.language,
                COALESCE(
                    (
                        SELECT json_agg(
                            json_build_object(
                                'id', ak.id,
                                'name', ak.name,
                                'key', ak.key_value,
                                'status', ak.status,
                                'createdAt', ak.created_at
                            )
                        )
                        FROM api_keys ak 
                        WHERE ak.user_id = u.id
                    ),
                    '[]'::json
                ) as "apiKeys"
            FROM users u
            WHERE 1=1
        `;

        if (status) {
            query += ` AND u.status = $${paramCount}`;
            values.push(status);
            paramCount++;
        }

        if (role) {
            query += ` AND u.role = $${paramCount}`;
            values.push(role);
            paramCount++;
        }

        if (search) {
            query += ` AND (
                u.username ILIKE $${paramCount} OR
                u.email ILIKE $${paramCount} OR
                u.name ILIKE $${paramCount}
            )`;
            values.push(`%${search}%`);
            paramCount++;
        }

        // 전체 개수 조회를 위한 서브쿼리
        const countQuery = `SELECT COUNT(*) FROM (${query}) AS count_total`;
        const countResult = await dbManager.query(countQuery, values);
        const total = parseInt(countResult.rows[0].count);

        // 정렬 및 페이징 적용
        query += ` ORDER BY u.${sortBy} ${sortOrder}`;
        query += ` LIMIT $${paramCount} OFFSET $${paramCount + 1}`;
        values.push(limit, offset);

        const result = await dbManager.query(query, values);

        return {
            users: result.rows,
            pagination: {
                total,
                page,
                limit,
                totalPages: Math.ceil(total / limit)
            }
        };
    }

    async deleteUser(userId) {
        // 사용자 존재 여부 확인
        const userResult = await dbManager.query("SELECT * FROM users WHERE id = $1", [userId]);

        if (userResult.rows.length === 0) {
            throw new Error("사용자를 찾을 수 없습니다.");
        }

        const user = userResult.rows[0];

        // 관리자 계정이 1명뿐인데 그 계정을 삭제하려는 경우 방지
        if (user.role === "admin") {
            const adminCount = await dbManager.query("SELECT COUNT(*) FROM users WHERE role = $1", ["admin"]);

            if (parseInt(adminCount.rows[0].count) <= 1) {
                throw new Error("마지막 관리자 계정은 삭제할 수 없습니다.");
            }
        }

        // 트랜잭션 시작
        await dbManager.query("BEGIN");

        try {
            // 활동 로그 기록
            await this.activityLogger.logActivity(userId, "USER_DELETED", {
                user_id: userId,
                email: user.email,
                role: user.role
            });

            // 사용자 삭제 (CASCADE로 API 키도 자동 삭제됨)
            await dbManager.query("DELETE FROM users WHERE id = $1", [userId]);

            // 트랜잭션 커밋
            await dbManager.query("COMMIT");

            return {
                success: true,
                message: "사용자가 성공적으로 삭제되었습니다.",
                deletedUser: {
                    id: user.id,
                    email: user.email,
                    name: user.name,
                    role: user.role
                }
            };
        } catch (error) {
            // 오류 발생 시 롤백
            await dbManager.query("ROLLBACK");
            throw error;
        }
    }

    async updateUserStatus(userId, status) {
        try {
            // 사용자 존재 여부 확인
            const userResult = await dbManager.query("SELECT * FROM users WHERE id = $1", [userId]);

            if (userResult.rows.length === 0) {
                throw new Error("사용자를 찾을 수 없습니다.");
            }

            const user = userResult.rows[0];

            // 관리자 계정이 1명뿐인데 그 계정을 비활성화하려는 경우 방지
            if (user.role === "admin" && status === "inactive") {
                const adminCount = await dbManager.query("SELECT COUNT(*) FROM users WHERE role = $1 AND status = $2", ["admin", "active"]);

                if (parseInt(adminCount.rows[0].count) <= 1) {
                    throw new Error("마지막 활성 관리자 계정은 비활성화할 수 없습니다.");
                }
            }

            // 사용자 상태 업데이트 (updated_at 컬럼 제거)
            const result = await dbManager.query(
                `UPDATE users 
                 SET status = $1
                 WHERE id = $2
                 RETURNING id, username, email, name, role, status, created_at, last_login, language`,
                [status, userId]
            );

            return result.rows[0];
        } catch (error) {
            console.error("사용자 상태 업데이트 중 오류:", error);
            throw error;
        }
    }

    async hashPassword(password) {
        return await bcrypt.hash(password, 10);
    }

    async updateUserDirectly(userId, updates) {
        try {
            await dbManager.query("BEGIN");

            // 업데이트할 필드 구성
            const updateFields = [];
            const values = [];
            let paramCount = 1;

            for (const [key, value] of Object.entries(updates)) {
                if (value !== undefined && value !== null) {
                    updateFields.push(`${key} = $${paramCount}`);
                    values.push(value);
                    paramCount++;
                }
            }

            if (updateFields.length === 0) {
                throw new Error("업데이트할 필드가 없습니다.");
            }

            const query = `
                UPDATE users 
                SET ${updateFields.join(", ")}
                WHERE id = $${paramCount}
                RETURNING id, username, email, name, role, status, created_at, language
            `;
            values.push(userId);

            // console.log('[DEBUG] Executing query:', query);
            // console.log('[DEBUG] Query values:', values);

            const result = await dbManager.query(query, values);

            if (result.rows.length === 0) {
                await dbManager.query("ROLLBACK");
                throw new Error("사용자를 찾을 수 없습니다.");
            }

            await dbManager.query("COMMIT");
            return result.rows[0];
        } catch (error) {
            console.error("[DEBUG] Error in updateUserDirectly:", error);
            await dbManager.query("ROLLBACK");
            throw error;
        }
    }

    async getUserByUsername(username) {
        try {
            const query = `
                SELECT 
                    u.id,
                    u.username,
                    u.email,
                    u.name,
                    u.role,
                    u.status,
                    u.created_at,
                    u.last_login,
                    u.language,
                    (
                        SELECT json_agg(
                            json_build_object(
                                'id', ak.id,
                                'key', ak.key_value,
                                'name', ak.name,
                                'status', ak.status,
                                'created_at', ak.created_at,
                                'last_used', ak.last_used,
                                'permissions', ak.permissions
                            )
                        )
                        FROM api_keys ak
                        WHERE ak.user_id = u.id AND ak.status = 'active'
                    ) as apiKeys
                FROM users u
                WHERE u.username = $1
            `;

            const result = await dbManager.query(query, [username]);
            if (result.rows.length === 0) {
                throw new Error("사용자를 찾을 수 없습니다.");
            }
            return result.rows[0];
        } catch (error) {
            console.error("사용자 조회 중 오류:", error);
            throw new Error("사용자 정보를 조회할 수 없습니다.");
        }
    }

    async updateUserStatusByUsername(username, status) {
        try {
            // 사용자 존재 여부 확인
            const userResult = await dbManager.query("SELECT * FROM users WHERE username = $1", [username]);

            if (userResult.rows.length === 0) {
                throw new Error("사용자를 찾을 수 없습니다.");
            }

            const user = userResult.rows[0];

            // 관리자 계정이 1명뿐인데 그 계정을 비활성화하려는 경우 방지
            if (user.role === "admin" && status === "inactive") {
                const adminCount = await dbManager.query("SELECT COUNT(*) FROM users WHERE role = $1 AND status = $2", ["admin", "active"]);

                if (parseInt(adminCount.rows[0].count) <= 1) {
                    throw new Error("마지막 활성 관리자 계정은 비활성화할 수 없습니다.");
                }
            }

            // 사용자 상태 업데이트 (updated_at 컬럼 제거)
            const result = await dbManager.query(
                `UPDATE users 
                 SET status = $1
                 WHERE username = $2
                 RETURNING id, username, email, name, role, status, created_at, last_login, language`,
                [status, username]
            );

            return result.rows[0];
        } catch (error) {
            console.error("사용자 상태 업데이트 중 오류:", error);
            throw error;
        }
    }
}

export default AuthService;
