/**
 * reportService.js
 * 보고서 생성 관련 기능을 제공하는 서비스 모듈
 */

import path from 'path';
import fs from 'fs';
import { fileURLToPath } from 'url';
import { spawn } from 'child_process';

// 현재 파일의 디렉토리 경로 가져오기
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const projectRoot = path.resolve(__dirname, '..', '..');

/**
 * 템플릿 목록을 가져오는 함수
 * @returns {Promise<{success: boolean, templates: Array, error: string}>}
 */
export async function getTemplateList() {
    try {
        const templatesDir = path.join(projectRoot, 'services', 'report', 'templates');
        const files = await fs.promises.readdir(templatesDir);
        
        // JSON 파일만 필터링
        const templateFiles = files.filter(file => file.endsWith('_templates.json'));
        
        // 템플릿 정보 추출
        const templates = templateFiles.map(file => {
            const templateId = file.replace('_templates.json', '');
            return {
                id: templateId,
                name: templateId.charAt(0).toUpperCase() + templateId.slice(1),
                file: file,
                description: getTemplateDescription(templateId)
            };
        });
        
        return {
            success: true,
            templates: templates
        };
    } catch (error) {
        console.error('템플릿 목록 가져오기 실패:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * 템플릿 설명을 가져오는 함수
 * @param {string} templateId 템플릿 ID
 * @returns {string} 템플릿 설명
 */
function getTemplateDescription(templateId) {
    const descriptions = {
        'simple': '기본 문서 템플릿 (보고개요, 본론, 결론)',        
        'simple01': '일반 문서 템플릿 (보고개요, 현황분석, 주요활동, 성과분석, 문제점및개선사항, 결론및제언, 추진일정)',
        'simple02': '연구 문서 템플릿 (연구배경, 선행연구, 연구방법, 연구결과, 논의, 결론및제언, 추진일정)',
        'simple03': '제안서 템플릿 (제안개요, 제안내용, 프로젝트관리, 추진일정, 투입인력, 예산계획)',
        'bizplan': '사업계획서 템플릿 (사업 개요, 시장 분석, 마케팅 전략, 재무 계획 등)',
        'marketing': '마케팅 보고서 템플릿 (시장 분석, 타겟 고객, 마케팅 전략, 예산 등)',
        'proposal': '제안서 템플릿 (제안 개요, 문제 정의, 해결 방안, 기대 효과 등)',
        'startup': '스타트업 사업계획서 템플릿 (비즈니스 모델, 시장 분석, 팀 구성, 투자 계획 등)',
        'thesis': '논문 템플릿 (서론, 본론, 결론, 참고문헌 등)',
        'khnp01': 'KHNP 보고서 템플릿 1',
        'khnp02': 'KHNP 보고서 템플릿 2'
    };
    
    return descriptions[templateId] || `${templateId} 템플릿`;
}

/**
 * 보고서를 생성하는 함수
 * @param {Object} options 보고서 생성 옵션
 * @param {string} options.prompt 보고서 주제
 * @param {string} options.template 템플릿 ID
 * @param {string} options.format 출력 형식 (pdf, hwpx, docx, pptx)
 * @param {Function} options.setVarVal 환경 변수 설정 함수
 * @param {Function} options.progressCallback 진행 상태 콜백 함수
 * @returns {Promise<{success: boolean, file_path: string, error: string}>}
 */
export async function generateReport(options) {
    try {
        const { prompt, template, format, setVarVal, progressCallback } = options;
        
        // 진행 상태 업데이트 함수
        const updateProgress = (message, percentage = null) => {
            if (typeof progressCallback === 'function') {
                progressCallback({
                    message,
                    percentage,
                    timestamp: new Date().toISOString()
                });
            }
        };
        
        updateProgress('문서 작성을 시작합니다...', null);
        
        // ReportManager 모듈 동적 로드
        updateProgress('문서 작성 모듈을 로드하는 중...', null);
        const ReportManagerPath = path.join(projectRoot, 'services', 'report', 'ReportManager.js');
        const ReportManagerModule = await import(ReportManagerPath);
        const ReportManager = ReportManagerModule.default;
        
        // RAG 및 웹 검색 활성화
        updateProgress('RAG 및 웹 검색을 활성화하는 중...', null);
        if (setVarVal) {
            await setVarVal('AIMODE', 'report', true);
            await setVarVal('USE_RAG', 'yes', true);
            await setVarVal('USE_WEB_SEARCH', 'yes', true);
        }
        
        // ReportManager 인스턴스 생성 및 초기화
        updateProgress('문서 작성 모듈을 초기화하는 중...', null);
        const reportManager = new ReportManager({ 
            debug: true,
            progressCallback: (message, percentage) => {
                // ReportManager에서 전달되는 메시지는 직접 처리하고 콘솔에는 출력하지 않음
                progressCallback({
                    message,
                    percentage,
                    timestamp: new Date().toISOString()
                });
            }
        });
        await reportManager.initialize();
        
        // 입력 검증
        if (!prompt) {
            throw new Error('보고서 주제를 입력해주세요.');
        }
        
        // 템플릿 처리
        updateProgress('템플릿을 처리하는 중...', null);
        let executiveSummary = prompt;
        if (template) {
            // 템플릿 존재 여부 확인
            const templatePath = path.join(projectRoot, 'services', 'report', 'templates', `${template}_templates.json`);
            if (!fs.existsSync(templatePath)) {
                console.warn(`템플릿 '${template}'을 찾을 수 없습니다. 기본 템플릿을 사용합니다.`);
                updateProgress(`템플릿 '${template}'을 찾을 수 없어 기본 템플릿을 사용합니다.`, null);
            } else {
                console.log(`템플릿 '${template}'을 사용합니다.`);
                updateProgress(`템플릿 '${template}'을 사용합니다.`, null);
                // 템플릿이 지정된 경우, 입력은 사업명으로 간주하고 템플릿 형식으로 변환
                executiveSummary = `# template: ${template}\n# pagebreak: false\n\n# prompt:\n\n#\n\n1. 사업명:\n${prompt}`;
            }
        }
        
        // 보고서 생성
        updateProgress('문서 작성을 시작합니다. 이 작업은 몇 분 정도 소요될 수 있습니다...', null);
        
        // Python 스크립트의 상세 출력을 캡처하기 위한 이벤트 리스너 설정
        const originalConsoleLog = console.log;
        let lastMessage = '';  // 마지막 메시지를 저장하는 변수

        console.log = function(message) {
            // 원본 메시지 출력
            originalConsoleLog.apply(console, arguments);
            
            if (typeof message === 'string') {
                // ReportManager 태그가 있는 메시지는 완전히 무시
                if (message.includes('[ReportManager]')) {
                    return;
                }
                
                // ANSI 이스케이프 시퀀스와 로그 레벨 제거
                let cleanMessage = message
                    .replace(/\u001b\[\d+m/g, '')
                    .replace(/\[(DEBUG|INFO|WARN|ERROR)\]\s*/g, '')
                    .replace(/\[JobID: [^\]]+\]\s*/g, '');
                
                // 이전 메시지와 동일한 경우 무시
                if (cleanMessage === lastMessage) {
                    return;
                }
                lastMessage = cleanMessage;

                // Python 스크립트의 진행 상황 메시지만 처리
                if (!cleanMessage.includes('[ReportManager]')) {  // ReportManager 메시지가 아닌 경우에만 처리
                    // 기존의 메시지 처리 로직...
                    if (cleanMessage.includes('작성 중') || 
                        cleanMessage.includes('생성 중') || 
                        cleanMessage.includes('분석 중') ||
                        cleanMessage.includes('🚀') || 
                        cleanMessage.includes('📊') || 
                        cleanMessage.includes('✅')) {
                        
                        const percentMatch = cleanMessage.match(/\((\d+\.\d+)%\)/);
                        const percent = percentMatch ? parseFloat(percentMatch[1]) : null;
                        updateProgress(cleanMessage, percent);
                    }
                }
            }
        };
        
        const result = await reportManager.generateBusinessPlan({
            executive_summary: executiveSummary,
            format: format || 'pdf',
            progressCallback: (message, percentage) => {
                // 백분율을 그대로 사용
                updateProgress(message, percentage);
            }
        });
        
        // 원래 console.log 복원
        console.log = originalConsoleLog;
        
        updateProgress('문서 작성이 완료되었습니다.', null);
        
        return {
            success: true,
            file_path: result.file_path || result.outputPath
        };
    } catch (error) {
        console.error('문서 작성 실패:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * 파일을 여는 함수
 * @param {string} filePath 파일 경로
 * @returns {Promise<{success: boolean, message: string, error: string}>}
 */
export async function openFile(filePath) {
    try {
        if (!filePath) {
            throw new Error('파일 경로가 필요합니다.');
        }
        
        // 플랫폼에 따라 적절한 명령어 실행
        let opened = false;
        if (process.platform === 'win32') {
            opened = await openFileWithCommand('start', [filePath]);
        } else if (process.platform === 'darwin') {
            opened = await openFileWithCommand('open', [filePath]);
        } else {
            opened = await openFileWithCommand('xdg-open', [filePath]);
        }
        
        return {
            success: opened,
            message: opened ? '파일이 열렸습니다.' : '파일을 열 수 없습니다.'
        };
    } catch (error) {
        console.error('파일 열기 실패:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * 폴더를 여는 함수
 * @param {string} filePath 파일 경로
 * @returns {Promise<{success: boolean, message: string, error: string}>}
 */
export async function openFolder(filePath) {
    try {
        if (!filePath) {
            throw new Error('폴더 경로가 필요합니다.');
        }
        
        // 파일 경로에서 디렉토리 경로 추출
        const dirPath = path.dirname(filePath);
        
        // 플랫폼에 따라 적절한 명령어 실행
        let opened = false;
        if (process.platform === 'win32') {
            opened = await openFileWithCommand('explorer', [dirPath]);
        } else if (process.platform === 'darwin') {
            opened = await openFileWithCommand('open', [dirPath]);
        } else {
            opened = await openFileWithCommand('xdg-open', [dirPath]);
        }
        
        return {
            success: opened,
            message: opened ? '폴더가 열렸습니다.' : '폴더를 열 수 없습니다.'
        };
    } catch (error) {
        console.error('폴더 열기 실패:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * 명령어를 사용하여 파일이나 폴더를 여는 함수
 * @param {string} command 명령어
 * @param {Array<string>} args 인자
 * @returns {Promise<boolean>} 성공 여부
 */
function openFileWithCommand(command, args) {
    return new Promise((resolve) => {
        const process = spawn(command, args, {
            shell: true,
            detached: true,
            stdio: 'ignore'
        });
        
        process.on('error', () => {
            resolve(false);
        });
        
        process.unref();
        resolve(true);
    });
} 