#!/bin/bash

# 컬러 출력을 위한 설정
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo -e "${GREEN}===== AI.RUN 리포트 생성기 컴파일 시작 =====${NC}"

# 현재 스크립트 위치 확인
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" &> /dev/null && pwd )"
REPORT_DIR="$(dirname "$(dirname "$SCRIPT_DIR")")"

# 가상환경 확인
if [[ -z "${VIRTUAL_ENV}" ]]; then
    echo -e "${RED}가상환경이 활성화되어 있지 않습니다.${NC}"
    echo -e "${YELLOW}먼저 가상환경을 활성화해주세요:${NC}"
    echo -e "source ~/.airun_venv/bin/activate"
    exit 1
fi

# Python 경로 확인
PYTHON_PATH="${VIRTUAL_ENV}/bin/python3"
if [ ! -f "$PYTHON_PATH" ]; then
    echo -e "${RED}가상환경의 Python을 찾을 수 없습니다: ${PYTHON_PATH}${NC}"
    exit 1
fi

# setup.py 생성
echo -e "${YELLOW}setup.py 생성 중...${NC}"
cat > "$REPORT_DIR/setup.py" << EOL
from setuptools import setup
from Cython.Build import cythonize
import numpy
import os

# 현재 디렉토리의 절대 경로
BASE_DIR = os.path.abspath(os.path.dirname(__file__))

# 컴파일할 파일들의 절대 경로 목록
source_files = [
    os.path.join(BASE_DIR, "report/generator.pyx"),
    os.path.join(BASE_DIR, "report/document.pyx"),
    os.path.join(BASE_DIR, "report/content.pyx"),
    os.path.join(BASE_DIR, "report/template.pyx"),
    os.path.join(BASE_DIR, "report/cache.pyx"),
    os.path.join(BASE_DIR, "report/search.pyx"),
    os.path.join(BASE_DIR, "report/analyzer.pyx"),
    os.path.join(BASE_DIR, "report/logger.pyx"),
    os.path.join(BASE_DIR, "report/visualization.pyx"),
    os.path.join(BASE_DIR, "report/utils.pyx")
]

setup(
    name='report_generator',
    ext_modules=cythonize(source_files,
        compiler_directives={
            'language_level': "3",
            'boundscheck': False,
            'nonecheck': False
        },
        # 파일별 컴파일러 지시문 설정
        compile_time_env={
            "UTILS_WRAPAROUND": True,      # utils.pyx
            "CACHE_WRAPAROUND": True,      # cache.pyx
            "CONTENT_WRAPAROUND": True,    # content.pyx
            "DOCUMENT_WRAPAROUND": True,   # document.pyx
            "GENERATOR_WRAPAROUND": True,  # generator.pyx
            "VISUALIZATION_WRAPAROUND": True  # visualization.pyx
        }
    ),
    include_dirs=[numpy.get_include()],
    zip_safe=False
)
EOL

# 컴파일 디렉토리 생성
BUILD_DIR="$REPORT_DIR/build"
mkdir -p "$BUILD_DIR"

# 소스 파일들을 .pyx로 복사
echo -e "${YELLOW}소스 파일 준비 중...${NC}"

for py_file in \
    "$REPORT_DIR/services/report/generator.py" \
    "$REPORT_DIR/services/report/document.py" \
    "$REPORT_DIR/services/report/content.py" \
    "$REPORT_DIR/services/report/template.py" \
    "$REPORT_DIR/services/report/cache.py" \
    "$REPORT_DIR/services/report/search.py" \
    "$REPORT_DIR/services/report/analyzer.py" \
    "$REPORT_DIR/services/report/logger.py" \
    "$REPORT_DIR/services/report/visualization.py" \
    "$REPORT_DIR/services/report/utils.py"
do
    # 파일 경로 수정
    fixed_path=$(echo "$py_file" | sed 's|/services/services/|/services/|g')
    if [ -f "$fixed_path" ]; then
        pyx_file="${fixed_path%.*}.pyx"
        cp "$fixed_path" "$pyx_file"
        echo -e "${GREEN}복사 완료: ${fixed_path} -> ${pyx_file}${NC}"
    else
        echo -e "${RED}오류: ${fixed_path} 파일을 찾을 수 없습니다${NC}"
        exit 1
    fi
done

# Cython으로 컴파일
echo -e "${YELLOW}Cython으로 컴파일 중...${NC}"
cd "$REPORT_DIR"
"$PYTHON_PATH" setup.py build_ext --inplace

# 컴파일 성공 여부 확인
if [ $? -eq 0 ]; then
    echo -e "${GREEN}컴파일 성공!${NC}"
    
    # 컴파일된 파일들을 build 디렉토리로 이동
    echo -e "${YELLOW}컴파일된 파일 정리 중...${NC}"
    
    # build 디렉토리가 없으면 생성
    mkdir -p "$BUILD_DIR"
    
    # 컴파일된 파일들을 build 디렉토리로 이동 (이미 있는 파일은 덮어쓰기)
    if [[ "$OSTYPE" == "msys" || "$OSTYPE" == "win32" || "$OSTYPE" == "cygwin" ]]; then
        # Windows 환경
        echo -e "${YELLOW}Windows 환경 감지됨${NC}"
        find "$REPORT_DIR" -name "*.pyd" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
        find "$REPORT_DIR" -name "*.c" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
        find "$REPORT_DIR" -name "*.pyx" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
    elif [[ "$OSTYPE" == "darwin"* ]]; then
        # macOS 환경
        echo -e "${YELLOW}macOS 환경 감지됨${NC}"
        find "$REPORT_DIR" -name "*.so" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
        find "$REPORT_DIR" -name "*.c" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
        find "$REPORT_DIR" -name "*.pyx" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
    else
        # Linux/Unix 환경
        echo -e "${YELLOW}Linux/Unix 환경 감지됨${NC}"
        find "$REPORT_DIR" -name "*.so" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
        find "$REPORT_DIR" -name "*.c" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
        find "$REPORT_DIR" -name "*.pyx" -not -path "$BUILD_DIR/*" -exec mv -f {} "$BUILD_DIR/" \;
    fi
    
    # 임시 파일 정리
    rm -rf "$REPORT_DIR/build" "$REPORT_DIR/report_generator.egg-info"
    
    echo -e "${GREEN}컴파일된 파일들이 ${BUILD_DIR} 디렉토리에 저장되었습니다${NC}"
    echo -e "${GREEN}===== 컴파일 완료 =====${NC}"
else
    echo -e "${RED}컴파일 실패!${NC}"
    exit 1
fi 