// AI Tool System - CLI Management Tools
// 도구 시스템을 위한 명령줄 관리 도구 (모듈 버전)

import chalk from 'chalk';
import boxen from 'boxen';
import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// 도구 레지스트리 import
import toolRegistry from './tools-registry.js';
import toolLoader from './tool-loader.js';

// 도구 시스템 초기화 플래그
let toolsInitialized = false;

// 도구 시스템 초기화 함수
async function ensureToolsLoaded() {
    if (!toolsInitialized) {
        await toolLoader.loadAllTools();
        toolsInitialized = true;
    }
}

// 도구 목록 조회
export async function listTools() {
    try {
        await ensureToolsLoaded();
        const tools = toolRegistry.getAllTools();
        
        if (tools.length === 0) {
            console.log(chalk.yellow('등록된 도구가 없습니다.'));
            return;
        }
        
        console.log(chalk.cyan.bold('\n📋 등록된 도구 목록:\n'));
        
        // 카테고리별로 그룹화
        const categories = {};
        tools.forEach(tool => {
            const category = tool.category || 'utility';
            if (!categories[category]) {
                categories[category] = [];
            }
            categories[category].push(tool);
        });
        
        // 카테고리별 출력
        Object.keys(categories).sort().forEach(category => {
            console.log(chalk.magenta.bold(`\n[${category.toUpperCase()}]`));
            categories[category].forEach(tool => {
                const status = tool.enabled !== false ? '✅' : '❌';
                console.log(`  ${status} ${chalk.green(tool.name)} (v${tool.version || '1.0.0'})`);
                console.log(`     ${chalk.gray(tool.description)}`);
            });
        });
        
        console.log(chalk.gray(`\n총 ${tools.length}개의 도구가 등록되어 있습니다.`));
        
    } catch (error) {
        console.error(chalk.red('도구 목록 조회 실패:'), error.message);
    }
}

// 도구 정보 조회
export async function showToolInfo(toolName) {
    try {
        await ensureToolsLoaded();
        const tool = toolRegistry.getToolInfo(toolName);
        
        if (!tool) {
            console.log(chalk.red(`도구 '${toolName}'을(를) 찾을 수 없습니다.`));
            return;
        }
        
        const info = [
            `${chalk.cyan.bold('📋 도구 정보')}`,
            '',
            `${chalk.green('이름:')} ${tool.name}`,
            `${chalk.green('카테고리:')} ${tool.category || 'utility'}`,
            `${chalk.green('버전:')} ${tool.version || '1.0.0'}`,
            `${chalk.green('상태:')} ${tool.enabled !== false ? '활성화' : '비활성화'}`,
            `${chalk.green('설명:')} ${tool.description}`,
            '',
            `${chalk.blue.bold('매개변수:')}`,
        ];
        
        if (tool.parameters && Object.keys(tool.parameters).length > 0) {
            Object.entries(tool.parameters).forEach(([key, param]) => {
                const required = param.required ? chalk.red('*') : ' ';
                const type = param.type || 'string';
                const defaultVal = param.default !== undefined ? ` (기본값: ${param.default})` : '';
                info.push(`  ${required} ${chalk.yellow(key)} (${type})${defaultVal}`);
                if (param.description) {
                    info.push(`    ${chalk.gray(param.description)}`);
                }
            });
        } else {
            info.push('  매개변수 없음');
        }
        
        console.log(boxen(info.join('\n'), {
            padding: 1,
            margin: 1,
            borderStyle: 'single',
            borderColor: 'cyan'
        }));
        
    } catch (error) {
        console.error(chalk.red('도구 정보 조회 실패:'), error.message);
    }
}

// 도구 테스트
export async function testTool(toolName, testArgs = {}) {
    try {
        await ensureToolsLoaded();
        console.log(chalk.cyan(`🧪 도구 '${toolName}' 테스트 중...`));
        
        const result = await toolRegistry.executeTool(toolName, testArgs, 'test-user');
        
        if (result.success) {
            console.log(chalk.green('✅ 테스트 성공!'));
            console.log(chalk.gray('결과:'), JSON.stringify(result.data, null, 2));
        } else {
            console.log(chalk.red('❌ 테스트 실패!'));
            console.log(chalk.gray('오류:'), result.error);
        }
        
    } catch (error) {
        console.error(chalk.red('도구 테스트 실패:'), error.message);
    }
}

// 도구 템플릿 생성
export async function createToolTemplate(toolName) {
    try {
        const template = `// ${toolName} 도구 정의
// 이 파일을 수정하여 새로운 도구를 만들 수 있습니다.

const ${toolName}Tool = {
    name: '${toolName}',
    category: 'utility',
    description: '${toolName} 도구입니다. 이 설명을 수정하세요.',
    version: '1.0.0',
    parameters: {
        input: {
            type: 'string',
            required: true,
            description: '입력 값'
        }
    },
    handler: async (args, context) => {
        const { input } = args;
        const { userId, job } = context;
        
        try {
            // 여기에 도구 로직을 구현하세요
            const result = \`처리된 결과: \${input}\`;
            
            return {
                success: true,
                data: { result },
                message: '${toolName} 도구가 성공적으로 실행되었습니다.'
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: '${toolName} 도구 실행 중 오류가 발생했습니다.'
            };
        }
    }
};

// 도구 등록 (배열로 export)
export default [${toolName}Tool];
`;

        const fileName = `${toolName}-tool.js`;
        const filePath = path.join(__dirname, 'custom-tools', fileName);
        
        // custom-tools 디렉토리가 없으면 생성
        const customToolsDir = path.join(__dirname, 'custom-tools');
        if (!fs.existsSync(customToolsDir)) {
            fs.mkdirSync(customToolsDir, { recursive: true });
        }
        
        fs.writeFileSync(filePath, template, 'utf8');
        
        console.log(chalk.green(`✅ 도구 템플릿이 생성되었습니다: ${filePath}`));
        console.log(chalk.yellow('📝 파일을 편집하여 도구를 완성하세요.'));
        console.log(chalk.gray('💡 완성 후 "node index.js tools reload" 명령으로 도구를 다시 로드하세요.'));
        
    } catch (error) {
        console.error(chalk.red('템플릿 생성 실패:'), error.message);
    }
}

// 도구 다시 로드
export async function reloadTools() {
    try {
        console.log(chalk.cyan('🔄 도구 다시 로드 중...'));
        
        // 기존 도구 정리
        toolRegistry.clearTools();
        
        // 도구 다시 로드
        await toolLoader.loadAllTools();
        
        const tools = toolRegistry.getAllTools();
        console.log(chalk.green(`✅ ${tools.length}개의 도구가 다시 로드되었습니다.`));
        
    } catch (error) {
        console.error(chalk.red('도구 다시 로드 실패:'), error.message);
    }
}

// 도구 통계
export async function showToolStats() {
    try {
        await ensureToolsLoaded();
        const tools = toolRegistry.getAllTools();
        const stats = toolRegistry.getStats();
        
        console.log(chalk.cyan.bold('\n📊 도구 시스템 통계:\n'));
        
        console.log(`${chalk.green('총 도구 수:')} ${tools.length}`);
        console.log(`${chalk.green('활성화된 도구:')} ${tools.filter(t => t.enabled !== false).length}`);
        console.log(`${chalk.green('비활성화된 도구:')} ${tools.filter(t => t.enabled === false).length}`);
        
        // 카테고리별 통계
        const categories = {};
        tools.forEach(tool => {
            const category = tool.category || 'utility';
            categories[category] = (categories[category] || 0) + 1;
        });
        
        console.log(chalk.blue.bold('\n📂 카테고리별 분포:'));
        Object.entries(categories).forEach(([category, count]) => {
            console.log(`  ${category}: ${count}개`);
        });
        
        // 실행 통계
        if (stats && Object.keys(stats).length > 0) {
            console.log(chalk.blue.bold('\n🎯 실행 통계:'));
            Object.entries(stats).forEach(([toolName, execCount]) => {
                console.log(`  ${toolName}: ${execCount}회 실행`);
            });
        }
        
    } catch (error) {
        console.error(chalk.red('통계 조회 실패:'), error.message);
    }
} 