#!/usr/bin/env node

import { SimpleMCPServer } from './simple_mcp_server.js';
import { spawn } from 'child_process';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

class MCPStdioServer {
  constructor() {
    this.server = new SimpleMCPServer();
    this.requestId = 0;
    this.dummyServerProcess = null;
    this.healthCheckInterval = null;
    this.isShuttingDown = false;
    this.connectionRetryCount = 0;
    this.maxRetryCount = 5;
    this.retryDelay = 2000; // 2초
  }

  // 더미 서버 시작
  async startDummyServer() {
    if (this.dummyServerProcess) {
      console.error("⚠️ 더미 서버가 이미 실행 중입니다.");
      return;
    }

    try {
      console.log("🚀 더미 서버 시작 중...");
      
      // 더미 서버 스크립트 경로 (poc_dummy_server 디렉토리의 server.js)
      const dummyServerPath = path.join(__dirname, '..', 'poc_dummy_server', 'server.js');
      
      this.dummyServerProcess = spawn('node', [dummyServerPath], {
        stdio: ['pipe', 'pipe', 'pipe'],
        detached: false
      });

      // 더미 서버 출력 처리
      // this.dummyServerProcess.stdout.on('data', (data) => {
      //   console.error(`📡 더미 서버: ${data.toString().trim()}`);
      // });

      this.dummyServerProcess.stderr.on('data', (data) => {
        console.error(`❌ 더미 서버 오류: ${data.toString().trim()}`);
      });

      this.dummyServerProcess.on('close', (code) => {
        // console.error(`🔚 더미 서버 종료 (코드: ${code})`);
        this.dummyServerProcess = null;
        
        // 종료 중이 아니라면 재시작 시도
        if (!this.isShuttingDown) {
          this.scheduleRestart();
        }
      });

      this.dummyServerProcess.on('error', (error) => {
        console.error(`💥 더미 서버 시작 실패:`, error.message);
        this.dummyServerProcess = null;
        
        if (!this.isShuttingDown) {
          this.scheduleRestart();
        }
      });

      // 서버 시작 대기
      await this.waitForServerStart();
      
    } catch (error) {
      console.error("💥 더미 서버 시작 중 오류:", error.message);
      this.dummyServerProcess = null;
      
      if (!this.isShuttingDown) {
        this.scheduleRestart();
      }
    }
  }

  // 서버 시작 대기
  async waitForServerStart() {
    const maxWaitTime = 10000; // 10초
    const checkInterval = 500; // 0.5초마다 확인
    let elapsedTime = 0;

    while (elapsedTime < maxWaitTime) {
      try {
        await this.server.testConnection();
        console.log("✅ 더미 서버가 성공적으로 시작되었습니다.");
        this.connectionRetryCount = 0; // 성공 시 재시도 카운트 리셋
        return true;
      } catch (error) {
        await new Promise(resolve => setTimeout(resolve, checkInterval));
        elapsedTime += checkInterval;
      }
    }

    console.error("⏰ 더미 서버 시작 대기 시간 초과");
    return false;
  }

  // 더미 서버 재시작 스케줄링
  scheduleRestart() {
    if (this.connectionRetryCount >= this.maxRetryCount) {
      console.error(`💀 더미 서버 재시작 시도 횟수 초과 (${this.maxRetryCount}회)`);
      return;
    }

    this.connectionRetryCount++;
    const delay = this.retryDelay * this.connectionRetryCount; // 지수 백오프
    
    console.log(`🔄 더미 서버 재시작 예약 (${this.connectionRetryCount}/${this.maxRetryCount}) - ${delay}ms 후`);
    
    setTimeout(async () => {
      if (!this.isShuttingDown) {
        await this.startDummyServer();
      }
    }, delay);
  }

  // 더미 서버 종료
  stopDummyServer() {
    if (this.dummyServerProcess) {
      console.log("🛑 더미 서버 종료 중...");
      this.dummyServerProcess.kill('SIGTERM');
      
      // 5초 후 강제 종료
      setTimeout(() => {
        if (this.dummyServerProcess) {
          console.log("💀 더미 서버 강제 종료");
          this.dummyServerProcess.kill('SIGKILL');
        }
      }, 5000);
    }
  }

  // 연결 상태 확인
  async checkConnection() {
    try {
      await this.server.testConnection();
      return true;
    } catch (error) {
      console.error("🔍 더미 서버 연결 확인 실패:", error.message);
      return false;
    }
  }

  // 헬스 체크 시작
  startHealthCheck() {
    if (this.healthCheckInterval) {
      clearInterval(this.healthCheckInterval);
    }

    this.healthCheckInterval = setInterval(async () => {
      if (this.isShuttingDown) return;

      const isConnected = await this.checkConnection();
      if (!isConnected) {
        console.error("🔍 더미 서버 연결이 끊어졌습니다. 재시작을 시도합니다.");
        this.stopDummyServer();
        await this.startDummyServer();
      }
    }, 30000); // 30초마다 확인
  }

  // 헬스 체크 중지
  stopHealthCheck() {
    if (this.healthCheckInterval) {
      clearInterval(this.healthCheckInterval);
      this.healthCheckInterval = null;
    }
  }

  // 정리 작업
  cleanup() {
    this.isShuttingDown = true;
    this.stopHealthCheck();
    this.stopDummyServer();
  }

  // JSON-RPC 응답 전송
  sendResponse(id, result = null, error = null) {
    const response = {
      jsonrpc: "2.0",
      id: id
    };

    if (error) {
      response.error = error;
    } else {
      response.result = result;
    }

    console.log(JSON.stringify(response));
  }

  // JSON-RPC 알림 전송
  sendNotification(method, params) {
    const notification = {
      jsonrpc: "2.0",
      method: method,
      params: params
    };

    console.log(JSON.stringify(notification));
  }

  // 요청 처리
  async handleRequest(request) {
    const { method, params, id } = request;

    try {
      switch (method) {
        case 'initialize':
          // 더미 서버 자동 시작
          await this.startDummyServer();
          
          const initialized = await this.server.initialize();
          if (initialized) {
            // 헬스 체크 시작
            this.startHealthCheck();
            
            this.sendResponse(id, {
              protocolVersion: "2024-11-05",
              capabilities: {
                tools: {}
              },
              serverInfo: {
                name: this.server.name,
                version: this.server.version
              }
            });
          } else {
            this.sendResponse(id, null, {
              code: -32603,
              message: "서버 초기화 실패: 더미 서버에 연결할 수 없습니다"
            });
          }
          break;

        case 'tools/list':
          const tools = this.server.getAvailableTools();
          this.sendResponse(id, { tools });
          break;

        case 'tools/call':
          const { name, arguments: args } = params;
          console.error(`🔧 MCP 도구 호출: ${name}`);
          console.error(`📋 전달된 파라미터:`, JSON.stringify(args, null, 2));
          
          // 위치 기반 파라미터 정규화
          const normalizeParams = (params, toolName) => {
            console.error(`🔧 ${toolName} 파라미터 정규화:`, params);
            
            // 이미 적절한 객체 형태면 그대로 반환
            if (typeof params === 'object' && params !== null && !Array.isArray(params)) {
              return params;
            }
            
            // 위치 기반 파라미터 처리
            if (toolName === 'get_complaints') {
              // get_complaints() -> {} 또는 get_complaints("소음") -> {keyword: "소음"}
              if (typeof params === 'string' && params.trim()) {
                return { keyword: params.trim() };
              }
              return {};
            } 
            
            else if (toolName === 'get_complaint') {
              // get_complaint(2) -> {id: "2"}
              return { id: String(params) };
            } 
            
            else if (toolName === 'update_complaint_status') {
              // update_complaint_status(3, "처리중") -> {id: "3", status: "처리중"}
              if (Array.isArray(params) && params.length >= 2) {
                return { id: String(params[0]), status: String(params[1]) };
              }
              // 단일 값인 경우는 id만
              return { id: String(params) };
            } 
            
            else if (toolName === 'reply_to_complaint') {
              // reply_to_complaint(1, "assistant", "답변") -> {id: "1", officer_id: "assistant", reply_content: "답변"}
              if (Array.isArray(params) && params.length >= 3) {
                return { 
                  id: String(params[0]), 
                  officer_id: String(params[1]), 
                  reply_content: String(params[2]) 
                };
              }
              // 부족한 파라미터인 경우 기본값 제공
              if (Array.isArray(params) && params.length === 2) {
                return { 
                  id: String(params[0]), 
                  officer_id: String(params[1]), 
                  reply_content: "자동 생성된 답변" 
                };
              }
              return { id: String(params) };
            }
            
            else if (toolName === 'process_complaint_workflow') {
              // process_complaint_workflow(1) -> {complaint_id: "1"}
              return { complaint_id: String(params) };
            }
            
            // 기타 도구는 원본 반환
            return params;
          };

          const normalizedArgs = normalizeParams(args, name);
          console.error(`✅ 정규화 결과:`, normalizedArgs);
          
          let result;

          switch (name) {
            case 'get_complaints':
              console.error(`🔍 get_complaints 호출 - 옵션:`, normalizedArgs);
              result = await this.server.getComplaints(normalizedArgs);
              break;
            case 'get_complaint':
              console.error(`🔍 get_complaint 호출 - ID: ${normalizedArgs?.id || 'undefined'}`);
              result = await this.server.getComplaint(normalizedArgs.id);
              break;
            case 'reply_to_complaint':
              result = await this.server.replyToComplaint(args.id, args.officer_id, args.reply_content);
              break;
            case 'update_complaint_status':
              result = await this.server.updateComplaintStatus(args.id, args.status);
              break;
            case 'process_complaint_workflow':
              result = await this.server.processComplaintWorkflow(args.complaint_id);
              break;
            default:
              throw new Error(`Unknown tool: ${name}`);
          }

          this.sendResponse(id, {
            content: [
              {
                type: "text",
                text: JSON.stringify(result, null, 2)
              }
            ]
          });
          break;

        default:
          this.sendResponse(id, null, {
            code: -32601,
            message: `Method not found: ${method}`
          });
      }
    } catch (error) {
      this.sendResponse(id, null, {
        code: -32603,
        message: error.message
      });
    }
  }

  // 메인 루프
  start() {
    process.stdin.setEncoding('utf8');
    
    let buffer = '';
    process.stdin.on('data', (chunk) => {
      buffer += chunk;
      
      // 줄 단위로 처리
      const lines = buffer.split('\n');
      buffer = lines.pop() || '';
      
      for (const line of lines) {
        if (line.trim()) {
          try {
            const request = JSON.parse(line);
            this.handleRequest(request);
          } catch (error) {
            console.error('JSON 파싱 오류:', error.message);
          }
        }
      }
    });

    process.stdin.on('end', () => {
      this.cleanup();
      process.exit(0);
    });

    // 프로세스 종료 시그널 처리
    process.on('SIGINT', () => {
      console.log('\n🛑 SIGINT 수신 - 정리 중...');
      this.cleanup();
      process.exit(0);
    });

    process.on('SIGTERM', () => {
      console.log('\n🛑 SIGTERM 수신 - 정리 중...');
      this.cleanup();
      process.exit(0);
    });

    // 서버 준비 완료 알림
    this.sendNotification('notifications/initialized', {});
  }
}

// 서버 시작
const mcpServer = new MCPStdioServer();
mcpServer.start(); 