import { NextRequest, NextResponse } from 'next/server';
import { pg } from '../../db';
import jwt from 'jsonwebtoken';
import { withTokenAuth } from '../../middlewares';

// 토큰에서 사용자 정보 추출
function getUserFromToken(request: NextRequest) {
  const authHeader = request.headers.get('Authorization');
  const token = authHeader?.startsWith('Bearer ') ? authHeader.substring(7) : null;
  
  if (!token) return null;
  
  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET || 'fallback-secret') as any;
    return decoded;
  } catch (error) {
    return null;
  }
}

// AI 메모리 도구 함수
export async function POST(request: NextRequest) {
  return withTokenAuth(request, async (req, user) => {
    try {
      if (!user || !(user.id || user.username)) {
        return NextResponse.json({ error: 'Invalid user data' }, { status: 400 });
      }

      const body = await req.json();
      const { action, data } = body;
      
      // user.username을 우선적으로 사용하도록 수정
      // const userId = user.username || user.id;

      switch (action) {
        case 'save_memory':
          return await saveMemory(user, data);
        case 'retrieve_memories':
          return await retrieveMemories(user, data);
        case 'update_memory':
          return await updateMemory(user, data);
        case 'delete_memory':
          const { id } = data;
          if (!id) {
            return NextResponse.json({ error: 'Memory ID is required' }, { status: 400 });
          }
          
          const result = await pg.query(`
            DELETE FROM user_memories
            WHERE id = $1 AND (user_id = $2 OR user_id = $3)
          `, [id, user.username, user.id]);

          if (result.rowCount === 0) {
            return NextResponse.json({ 
              success: false, 
              error: 'Memory not found or not owned by user' 
            }, { status: 404 });
          }

          return NextResponse.json({
            success: true,
            message: 'Memory deleted successfully'
          });
        case 'delete_all_memories':
          // const allMemoriesUserId = user.id; // user.id를 사용하여 모든 메모리 삭제 ( username이 아닌 ID로 처리 )
          await pg.query(`
            DELETE FROM user_memories
            WHERE (user_id = $1 OR user_id = $2)
          `, [user.username, user.id]);

          return NextResponse.json({
            success: true,
            message: 'All memories have been deleted successfully.'
          });
        default:
          return NextResponse.json({ error: 'Invalid action' }, { status: 400 });
      }

    } catch (error) {
      console.error('AI 메모리 도구 오류:', error);
      return NextResponse.json({ 
        error: 'Internal server error',
        details: error instanceof Error ? error.message : String(error)
      }, { status: 500 });
    }
  });
}

// 메모리 저장
async function saveMemory(user: any, data: any) {
  try {
    const { 
      title, 
      content, 
      category = 'conversation', 
      importance_level = 3, 
      tags = [],
      context = '' 
    } = data;

    if (!title || !content) {
      return NextResponse.json({ error: 'Title and content are required' }, { status: 400 });
    }

    const userId = user.username || user.id; // username이나 id 중 하나 사용

    // 중복 메모리 체크 (같은 제목이나 유사한 내용)
    const existingMemory = await pg.query(`
      SELECT id FROM user_memories 
      WHERE user_id = $1 AND title = $2 AND is_active = true
    `, [userId, title]);

    if (existingMemory.rows.length > 0) {
      // 기존 메모리 업데이트
      const result = await pg.query(`
        UPDATE user_memories 
        SET content = $1, updated_at = CURRENT_TIMESTAMP, importance_level = GREATEST(importance_level, $2)
        WHERE id = $3
        RETURNING *
      `, [content, importance_level, existingMemory.rows[0].id]);

      return NextResponse.json({
        success: true,
        action: 'updated',
        memory: result.rows[0],
        message: '기존 메모리가 업데이트되었습니다.'
      });
    } else {
      // 새 메모리 생성
      const result = await pg.query(`
        INSERT INTO user_memories (user_id, title, content, category, importance_level, tags)
        VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING *
      `, [userId, title, content, category, importance_level, tags]);

      return NextResponse.json({
        success: true,
        action: 'created',
        memory: result.rows[0],
        message: '새로운 메모리가 저장되었습니다.'
      });
    }
  } catch (error) {
    console.error('메모리 저장 오류:', error);
    return NextResponse.json({ 
      error: 'Failed to save memory',
      details: error instanceof Error ? error.message : String(error)
    }, { status: 500 });
  }
}

// 메모리 검색
async function retrieveMemories(user: any, data: any) {
  try {
    const { 
      query = '', 
      category = null, 
      limit = 10,
      min_importance = 1 
    } = data;

    // user.username을 우선적으로 사용하도록 수정
    const userId = user.username || user.id;
    // console.log(`[메모리 조회] user 전체 객체:`, JSON.stringify(user, null, 2));
    // console.log(`[메모리 조회] user.id: ${user.id}, user.username: ${user.username}. 사용될 userId: ${userId}`);
    if (!userId) {
      console.error('유효하지 않은 사용자 정보:', user);
      return NextResponse.json({ 
        error: 'Invalid user data',
        details: 'Username or ID is required'
      }, { status: 400 });
    }

    // 숫자 ID와 문자열 username 모두 확인하도록 수정
    let sqlQuery = `
      SELECT * FROM user_memories 
      WHERE (user_id = $1 OR user_id = $2) AND is_active = true AND importance_level >= $3
    `;
    const params: any[] = [userId, user.id?.toString(), min_importance];

    if (category) {
      sqlQuery += ` AND category = $${params.length + 1}`;
      params.push(category);
    }

    if (query) {
      sqlQuery += ` AND (title ILIKE $${params.length + 1} OR content ILIKE $${params.length + 1})`;
      params.push(`%${query}%`);
    }

    sqlQuery += ` ORDER BY importance_level DESC, last_accessed DESC LIMIT $${params.length + 1}`;
    params.push(limit);

    // console.log('실행할 쿼리:', sqlQuery);
    // console.log('쿼리 파라미터:', params);

    const result = await pg.query(sqlQuery, params);

    // 접근 횟수 업데이트
    if (result.rows.length > 0) {
      const memoryIds = result.rows.map(row => row.id);
      await pg.query(`
        UPDATE user_memories 
        SET last_accessed = CURRENT_TIMESTAMP, access_count = access_count + 1
        WHERE id = ANY($1)
      `, [memoryIds]);
    }

    return NextResponse.json({
      success: true,
      memories: result.rows,
      count: result.rows.length,
      message: `${result.rows.length}개의 관련 메모리를 찾았습니다.`
    });
  } catch (error) {
    console.error('메모리 검색 오류:', error);
    return NextResponse.json({ 
      error: 'Failed to retrieve memories',
      details: error instanceof Error ? error.message : String(error)
    }, { status: 500 });
  }
}

// 메모리 업데이트
async function updateMemory(user: any, data: any) {
  try {
    const { id, title, content, category, importance_level, tags } = data;

    if (!id) {
      return NextResponse.json({ error: 'Memory ID is required' }, { status: 400 });
    }

    const userId = user.username || user.id; // username이나 id 중 하나 사용

    const result = await pg.query(`
      UPDATE user_memories 
      SET title = COALESCE($1, title), 
          content = COALESCE($2, content), 
          category = COALESCE($3, category),
          importance_level = COALESCE($4, importance_level),
          tags = COALESCE($5, tags),
          updated_at = CURRENT_TIMESTAMP
      WHERE id = $6 AND user_id = $7
      RETURNING *
    `, [title, content, category, importance_level, tags, id, userId]);

    if (result.rows.length === 0) {
      return NextResponse.json({ error: 'Memory not found or unauthorized' }, { status: 404 });
    }

    return NextResponse.json({
      success: true,
      memory: result.rows[0],
      message: '메모리가 업데이트되었습니다.'
    });
  } catch (error) {
    console.error('메모리 업데이트 오류:', error);
    return NextResponse.json({ 
      error: 'Failed to update memory',
      details: error instanceof Error ? error.message : String(error)
    }, { status: 500 });
  }
} 