import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';

const OLLAMA_URL = process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || process.env.OLLAMA_URL || "https://api.hamonize.com/ollama";
const VLLM_SERVER = process.env.NEXT_PUBLIC_VLLM_SERVER || "https://api.hamonize.com/vllm";
const OPENAI_API_KEY = process.env.OPENAI_API_KEY;
const OPENAI_MODEL = process.env.OPENAI_MODEL || "gpt-4.1-nano";
const VLLM_JUDGE_MODEL = process.env.VLLM_JUDGE_MODEL || "unsloth/gemma-3-12b-it-unsloth-bnb-4bit"; // VLLM 폴백 모델
const OLLAMA_JUDGE_MODEL = process.env.OLLAMA_JUDGE_MODEL || "airun-chat:latest"; // Ollama 백업 모델

// 확장된 50개 문제 세트 - 더 나은 변별력과 통계적 신뢰도
const QUESTIONS = [
  // ========== 수학 (Math) - 10문제 ==========
  // Easy (3문제)
  { category: "math", question: "2 + 2는?", answer: "4", difficulty: "easy", acceptable: ["4", "four", "넷"] },
  { category: "math", question: "10 - 7은?", answer: "3", difficulty: "easy", acceptable: ["3", "three", "셋"] },
  { category: "math", question: "5 × 4는?", answer: "20", difficulty: "easy", acceptable: ["20", "twenty", "이십"] },
  
  // Medium (5문제)
  { category: "math", question: "15 × 7 + 23 - 8은?", answer: "120", difficulty: "medium", acceptable: ["120", "백이십"] },
  { category: "math", question: "√144는?", answer: "12", difficulty: "medium", acceptable: ["12", "twelve", "열둘"] },
  { category: "math", question: "2³ + 3²는?", answer: "17", difficulty: "medium", acceptable: ["17", "seventeen", "열일곱"] },
  { category: "math", question: "100의 25%는?", answer: "25", difficulty: "medium", acceptable: ["25", "twenty-five", "이십오"] },
  { category: "math", question: "다음 중 소수는? 13, 15, 21, 27", answer: "13", difficulty: "medium", acceptable: ["13", "thirteen", "열셋"] },
  
  // Hard (2문제)
  { category: "math", question: "피보나치 수열의 10번째 항은? (첫 두 항이 1, 1일 때)", answer: "55", difficulty: "hard", acceptable: ["55", "쉰다섯"] },
  { category: "math", question: "log₂(32)는?", answer: "5", difficulty: "hard", acceptable: ["5", "five", "다섯"] },

  // ========== 코딩 (Coding) - 10문제 ==========
  // Easy (3문제)
  { category: "coding", question: "다음 코드의 출력은? print(1+1)", answer: "2", difficulty: "easy", acceptable: ["2", "2\n"] },
  { category: "coding", question: "Python에서 문자열 길이를 구하는 함수는?", answer: "len", difficulty: "easy", acceptable: ["len", "len()", "length"] },
  { category: "coding", question: "리스트 [1,2,3]의 첫 번째 요소는? (0-indexed)", answer: "1", difficulty: "easy", acceptable: ["1", "one"] },
  
  // Medium (5문제)
  { category: "coding", question: "Python에서 리스트 [1,2,3,4,5]의 평균을 구하는 코드를 작성하세요.", 
    answer: "3", difficulty: "medium", acceptable: ["3", "3.0", "sum([1,2,3,4,5])/len([1,2,3,4,5])", "sum([1,2,3,4,5])/5"], freeform: true },
  { category: "coding", question: "for i in range(3): print(i) 의 출력은?", answer: "0 1 2", difficulty: "medium", 
    acceptable: ["0 1 2", "0\n1\n2", "0,1,2"] },
  { category: "coding", question: "배열 [3,1,4,1,5]를 오름차순 정렬하면?", answer: "[1,1,3,4,5]", difficulty: "medium", 
    acceptable: ["[1,1,3,4,5]", "1,1,3,4,5", "1 1 3 4 5"] },
  { category: "coding", question: "x = 5; x += 3; x의 값은?", answer: "8", difficulty: "medium", acceptable: ["8", "eight"] },
  { category: "coding", question: "문자열 'hello'를 거꾸로 뒤집으면?", answer: "olleh", difficulty: "medium", acceptable: ["olleh", "'olleh'"] },
  
  // Hard (2문제)
  { category: "coding", question: "시간복잡도 O(n log n)인 정렬 알고리즘 하나는?", answer: "퀵소트", difficulty: "hard", 
    acceptable: ["퀵소트", "퀵정렬", "quick sort", "quicksort", "merge sort", "병합정렬", "힙정렬", "heap sort"] },
  { category: "coding", question: "재귀함수로 n! 구현시 base case는?", answer: "n==0 또는 n==1", difficulty: "hard", 
    acceptable: ["n==0", "n==1", "n<=1", "n이 0", "n이 1", "0 또는 1"], freeform: true },

  // ========== 추론 (Reasoning) - 10문제 ==========
  // Easy (3문제)
  { category: "reasoning", question: "고양이와 개의 공통점은?", answer: "동물", difficulty: "easy", 
    acceptable: ["동물", "포유류", "애완동물", "반려동물", "네발동물", "척추동물"] },
  { category: "reasoning", question: "빨강과 파랑을 섞으면?", answer: "보라", difficulty: "easy", 
    acceptable: ["보라", "보라색", "purple", "violet"] },
  { category: "reasoning", question: "얼음이 녹으면?", answer: "물", difficulty: "easy", 
    acceptable: ["물", "water", "액체", "liquid"] },
  
  // Medium (5문제)
  { category: "reasoning", question: "만약 모든 A가 B이고, 모든 B가 C라면, A와 C의 관계는?", 
    answer: "모든 A는 C이다", difficulty: "medium", 
    acceptable: ["모든 A는 C이다", "A는 C이다", "A⊆C", "A is C", "A는 C의 부분집합"] },
  { category: "reasoning", question: "시계가 3시를 가리킬 때 시침과 분침의 각도는?", answer: "90도", difficulty: "medium", 
    acceptable: ["90도", "90", "직각", "90 degrees"] },
  { category: "reasoning", question: "A는 B보다 크고, B는 C보다 크다. 가장 작은 것은?", answer: "C", difficulty: "medium", 
    acceptable: ["C", "씨"] },
  { category: "reasoning", question: "1, 1, 2, 3, 5, 8, ? 다음 수는?", answer: "13", difficulty: "medium", 
    acceptable: ["13", "thirteen", "열셋"] },
  { category: "reasoning", question: "동쪽의 반대 방향은?", answer: "서쪽", difficulty: "medium", 
    acceptable: ["서쪽", "서", "west", "W"] },
  
  // Hard (2문제)
  { category: "reasoning", question: "다음 수열의 규칙을 찾고 다음 수를 예측하세요: 2, 6, 12, 20, 30, ?", 
    answer: "42", difficulty: "hard", acceptable: ["42", "마흔둘"] },
  { category: "reasoning", question: "3개의 상자 중 하나에 상금이 있다. 당신이 1번을 선택했고, 진행자가 3번은 비어있다고 공개했다. 바꿔야 할까?", 
    answer: "예", difficulty: "hard", acceptable: ["예", "네", "바꾼다", "yes", "change"], freeform: true },

  // ========== 지식 (Knowledge) - 10문제 ==========
  // Easy (3문제)
  { category: "knowledge", question: "한국의 수도는?", answer: "서울", difficulty: "easy", 
    acceptable: ["서울", "서울특별시", "Seoul"] },
  { category: "knowledge", question: "태양계에서 가장 큰 행성은?", answer: "목성", difficulty: "easy", 
    acceptable: ["목성", "Jupiter", "주피터"] },
  { category: "knowledge", question: "물의 화학식은?", answer: "H2O", difficulty: "easy", 
    acceptable: ["H2O", "H₂O"] },
  
  // Medium (5문제)
  { category: "knowledge", question: "파이썬을 만든 사람은?", answer: "귀도 반 로섬", difficulty: "medium", 
    acceptable: ["귀도 반 로섬", "Guido van Rossum", "귀도", "Guido"] },
  { category: "knowledge", question: "인공지능의 아버지로 불리는 사람은?", answer: "앨런 튜링", difficulty: "medium", 
    acceptable: ["앨런 튜링", "Alan Turing", "튜링", "Turing"] },
  { category: "knowledge", question: "HTTP의 기본 포트 번호는?", answer: "80", difficulty: "medium", 
    acceptable: ["80", "eighty"] },
  { category: "knowledge", question: "비트코인을 만든 사람(또는 그룹)의 이름은?", answer: "사토시 나카모토", difficulty: "medium", 
    acceptable: ["사토시 나카모토", "Satoshi Nakamoto", "나카모토 사토시"] },
  { category: "knowledge", question: "CPU의 풀네임은?", answer: "Central Processing Unit", difficulty: "medium", 
    acceptable: ["Central Processing Unit", "중앙처리장치", "중앙 처리 장치"] },
  
  // Hard (2문제)
  { category: "knowledge", question: "양자컴퓨터의 기본 단위는?", answer: "큐비트", difficulty: "hard", 
    acceptable: ["큐비트", "qubit", "quantum bit", "양자비트"] },
  { category: "knowledge", question: "딥러닝에서 'Transformer' 아키텍처를 제안한 논문 제목은?", 
    answer: "Attention is All You Need", difficulty: "hard", 
    acceptable: ["Attention is All You Need", "attention is all you need"] },

  // ========== 언어 (Language) - 10문제 ==========
  // Easy (3문제)
  { category: "language", question: "다음 문장을 영어로 번역: 나는 학생입니다.", answer: "I am a student", difficulty: "easy", 
    acceptable: ["I am a student", "I'm a student"] },
  { category: "language", question: "'안녕하세요'를 영어로?", answer: "Hello", difficulty: "easy", 
    acceptable: ["Hello", "Hi", "Hey", "Good morning", "Good afternoon"] },
  { category: "language", question: "'Thank you'를 한국어로?", answer: "감사합니다", difficulty: "easy", 
    acceptable: ["감사합니다", "고맙습니다", "고마워요", "감사해요"] },
  
  // Medium (5문제)
  { category: "language", question: "'Time flies like an arrow'를 한국어로 번역하면?", 
    answer: "시간은 화살처럼 날아간다", difficulty: "medium", 
    acceptable: ["시간은 화살처럼 날아간다", "세월은 화살처럼 빠르다", "시간이 화살처럼 지나간다"] },
  { category: "language", question: "다음 문장이 문법적으로 올바른가? '나는 어제 학교에 갔습니다.'", 
    answer: "예", difficulty: "medium", acceptable: ["예", "네", "맞다", "올바르다", "yes", "correct"] },
  { category: "language", question: "'Break a leg'의 의미는?", answer: "행운을 빌어", difficulty: "medium", 
    acceptable: ["행운을 빌어", "행운을 빈다", "good luck", "화이팅", "파이팅"] },
  { category: "language", question: "주어를 찾으세요: '아름다운 꽃이 정원에 피었다.'", answer: "꽃", difficulty: "medium", 
    acceptable: ["꽃", "꽃이", "아름다운 꽃", "아름다운 꽃이"] },
  { category: "language", question: "'Piece of cake'의 의미는?", answer: "매우 쉬운", difficulty: "medium", 
    acceptable: ["매우 쉬운", "쉬운", "매우 쉬운 일", "간단한", "easy", "very easy"] },
  
  // Hard (2문제)
  { category: "language", question: "능동태를 수동태로: 'The team developed a new algorithm'", 
    answer: "A new algorithm was developed by the team", difficulty: "hard", 
    acceptable: ["A new algorithm was developed by the team", "A new algorithm has been developed by the team"] },
  { category: "language", question: "'To be or not to be'를 쓴 작가는?", answer: "셰익스피어", difficulty: "hard", 
    acceptable: ["셰익스피어", "Shakespeare", "윌리엄 셰익스피어", "William Shakespeare"] }
];

const RESULTS_FILE = path.join(process.cwd(), 'ollama_eval_judge_results.json');
const STATUS_FILE = path.join(process.cwd(), 'eval_status.json');

// app.py 패턴을 적용한 개선된 평가 클라이언트
class ImprovedEvaluationClient {
  private static judgeCache = new Map<string, { score: number; judge_model: string; source: string; timestamp: number }>();
  private static CACHE_DURATION = 3600000; // 1시간 캐싱
  private abortController: AbortController;
  private judgeProvider: 'openai' | 'ollama' | 'vllm';

  constructor(judgeProvider: 'openai' | 'ollama' | 'vllm' = 'ollama') {
    this.abortController = new AbortController();
    this.judgeProvider = judgeProvider;
    console.log(`🎯 Judge 모델 제공자: ${judgeProvider}`);
  }

  // app.py 패턴: 타임아웃이 적용된 fetch 함수
  private async fetchWithTimeout(url: string, options: any, timeoutMs: number = 30000): Promise<Response> {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), timeoutMs);
    
    try {
      const response = await fetch(url, {
        ...options,
        signal: controller.signal
      });
      clearTimeout(timeoutId);
      return response;
    } catch (error) {
      clearTimeout(timeoutId);
      throw error;
    }
  }

  // app.py 패턴: 캐시를 활용한 Judge 평가
  private getCachedJudgeResult(key: string): { score: number; judge_model: string; source: string } | null {
    const cached = ImprovedEvaluationClient.judgeCache.get(key);
    if (cached && (Date.now() - cached.timestamp) < ImprovedEvaluationClient.CACHE_DURATION) {
      console.debug(`캐시에서 Judge 결과 반환: ${key}`);
      return { score: cached.score, judge_model: cached.judge_model, source: cached.source };
    }
    return null;
  }

  private setCachedJudgeResult(key: string, result: { score: number; judge_model: string; source: string }): void {
    ImprovedEvaluationClient.judgeCache.set(key, {
      ...result,
      timestamp: Date.now()
    });
  }

  // 개선된 Judge 점수 평가 함수 - 문제 유형과 난이도 고려
  async judgeScoreWithInfo(judgePrompt: string, modelAnswer: string, goldAnswer: string, question?: any): Promise<{score: number, judge_model: string, source: string}> {
    // 캐시 키 생성
    const cacheKey = `${judgePrompt}_${modelAnswer}_${goldAnswer}`;
    
    // 캐시에서 먼저 확인
    const cachedResult = this.getCachedJudgeResult(cacheKey);
    if (cachedResult) {
      return cachedResult;
    }

    // 자유형식 답변인 경우 다른 평가 기준 적용
    const isFreeform = question?.freeform || goldAnswer === "자유작성";
    const difficulty = question?.difficulty || "medium";
    const acceptable = question?.acceptable || [goldAnswer];

    // 허용 가능한 답변 목록에 있으면 만점
    if (acceptable.some((ans: string) => 
      modelAnswer.toLowerCase().trim() === ans.toLowerCase().trim() ||
      modelAnswer.includes(ans) || ans.includes(modelAnswer)
    )) {
      const perfectResult = { score: 10, judge_model: 'exact-match', source: 'rule-based' };
      this.setCachedJudgeResult(cacheKey, perfectResult);
      return perfectResult;
    }

    // 개선된 프롬프트 - 문제 유형과 난이도 고려
    const prompt = `
당신은 AI 모델의 답변을 평가하는 전문가입니다.

[문제 정보]
- 카테고리: ${question?.category || '일반'}
- 난이도: ${difficulty}
- 자유형식 답변: ${isFreeform ? '예' : '아니오'}

[질문]
${judgePrompt}

[기준 답변${acceptable.length > 1 ? ' (여러 정답 가능)' : ''}]
${goldAnswer}
${acceptable.length > 1 ? '허용 답변: ' + acceptable.join(', ') : ''}

[모델 답변]
${modelAnswer}

평가 기준:
${isFreeform ? `
자유형식 답변이므로 다음을 평가하세요:
- 주제 관련성 (질문과의 연관성)
- 답변의 완성도와 논리성
- 언어 사용의 적절성
- 창의성과 통찰력
` : `
정답과의 비교를 통해 평가하세요:
- 의미적 정확성 (핵심 내용이 맞는가)
- 표현의 적절성 (다른 표현이지만 같은 의미인가)
- 완전성 (필요한 정보가 모두 포함되었는가)
`}

난이도 가중치:
- easy: 기본 지식만 요구하므로 엄격하게 평가
- medium: 적절한 이해도를 보이면 관대하게 평가  
- hard: 복잡한 문제이므로 부분 점수를 관대하게 부여

점수 기준:
- 10점: 완벽한 답변 (${isFreeform ? '매우 우수한 자유 답변' : '정답과 완전히 일치'})
- 8-9점: 우수한 답변 (핵심 내용 정확, 사소한 차이만 존재)
- 6-7점: 양호한 답변 (대체로 맞지만 일부 개선 필요)
- 4-5점: 부분적 정답 (일부만 맞거나 불완전)
- 2-3점: 미흡한 답변 (대부분 틀리지만 일부 관련성 있음)
- 0-1점: 오답 (완전히 틀리거나 무관한 답변)

**반드시 0-10 사이의 정수로만 답하세요. 점수만 출력하세요.**
`;

    // Judge 모델 우선순위에 따라 시도
    // 선택된 provider를 먼저 시도하고, 실패시에만 fallback 시도
    let judgeResult = null;
    
    // 1. 선택된 provider 우선 시도
    console.log(`🎯 선택된 Judge 제공자로 평가 시도: ${this.judgeProvider}`);
    
    if (this.judgeProvider === 'ollama') {
      judgeResult = await this.tryOllamaJudge(prompt).catch(error => {
        console.debug("Ollama Judge 실패:", error.message);
        return null;
      });
    } else if (this.judgeProvider === 'openai' && OPENAI_API_KEY) {
      judgeResult = await this.tryOpenAIJudge(prompt).catch(error => {
        console.debug("OpenAI Judge 실패:", error.message);
        return null;
      });
    } else if (this.judgeProvider === 'vllm') {
      judgeResult = await this.tryVLLMJudge(prompt).catch(error => {
        console.debug("VLLM Judge 실패:", error.message);
        return null;
      });
    }
    
    // 2. 선택된 provider가 실패한 경우에만 fallback 시도
    if (!judgeResult) {
      console.log("⚠️ 선택된 Judge 제공자 실패, fallback 시도");
      
      // Fallback 순서: Ollama -> OpenAI -> VLLM
      const fallbackProviders = [];
      
      if (this.judgeProvider !== 'ollama') {
        fallbackProviders.push({
          name: 'ollama',
          fn: () => this.tryOllamaJudge(prompt)
        });
      }
      
      if (this.judgeProvider !== 'openai' && OPENAI_API_KEY) {
        fallbackProviders.push({
          name: 'openai',
          fn: () => this.tryOpenAIJudge(prompt)
        });
      }
      
      if (this.judgeProvider !== 'vllm') {
        fallbackProviders.push({
          name: 'vllm',
          fn: () => this.tryVLLMJudge(prompt)
        });
      }
      
      // Fallback providers를 순차적으로 시도
      for (const provider of fallbackProviders) {
        console.log(`🔄 Fallback 시도: ${provider.name}`);
        judgeResult = await provider.fn().catch(error => {
          console.debug(`${provider.name} Judge (fallback) 실패:`, error.message);
          return null;
        });
        
        if (judgeResult) {
          console.log(`✅ Fallback 성공: ${provider.name}`);
          break;
        }
      }
    }
    
    // 3. 결과 처리
    if (judgeResult) {
      this.setCachedJudgeResult(cacheKey, judgeResult);
      return judgeResult;
    }

    // app.py 패턴: 모든 Judge 모델 실패 시 기본값
    console.error("모든 Judge 모델 사용 실패. 기본 점수 5점을 반환합니다.");
    const fallbackResult = { score: 5, judge_model: 'fallback', source: 'fallback' };
    this.setCachedJudgeResult(cacheKey, fallbackResult);
    return fallbackResult;
  }

  // OpenAI Judge 시도
  private async tryOpenAIJudge(prompt: string): Promise<{score: number, judge_model: string, source: string} | null> {
    console.debug(`Judge 평가 시도: OpenAI ${OPENAI_MODEL}`);
    
    const headers = {
      "Authorization": `Bearer ${OPENAI_API_KEY}`,
      "Content-Type": "application/json"
    };

    const data = {
      model: OPENAI_MODEL,
      messages: [
        { role: "system", content: "당신은 AI 답변 평가자입니다. 반드시 점수만 숫자로 출력하세요." },
        { role: "user", content: prompt }
      ],
      max_tokens: 10,
      temperature: 0
    };

    // app.py 패턴: 30초 타임아웃 적용
    const response = await this.fetchWithTimeout("https://api.openai.com/v1/chat/completions", {
      method: 'POST',
      headers,
      body: JSON.stringify(data),
      signal: this.abortController.signal
    }, 30000);

    if (response.ok) {
      const result = await response.json();
      const scoreStr = result.choices[0].message.content.trim();
      const score = parseInt(scoreStr);
      if (score >= 0 && score <= 10) {
        console.debug(`OpenAI Judge 평가 성공: ${score}점`);
        return { score, judge_model: OPENAI_MODEL, source: 'openai' };
      }
    }
    
    throw new Error(`OpenAI Judge 실패: ${response.status}`);
  }

  // Ollama Judge 시도
  private async tryOllamaJudge(prompt: string): Promise<{score: number, judge_model: string, source: string} | null> {
    console.debug(`Judge 평가 시도: Ollama ${OLLAMA_JUDGE_MODEL}`);
    
    // hamonize 모델은 chat 엔드포인트를 사용해야 함
    const useChat = OLLAMA_JUDGE_MODEL.includes('hamonize');
    
    let response;
    if (useChat) {
      // chat 엔드포인트 사용 (hamonize 모델)
      response = await this.fetchWithTimeout(`${OLLAMA_URL}/api/chat`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: OLLAMA_JUDGE_MODEL,
          messages: [
            {
              role: "system",
              content: "You are an evaluation assistant. Rate answers from 0-10 and respond only in JSON format."
            },
            {
              role: "user",
              content: `${prompt}

Return JSON: {"score": <0-10>}`
            }
          ],
          stream: false,
          options: {
            temperature: 0,
            num_predict: 30
          }
        }),
        signal: this.abortController.signal
      }, 30000);
    } else {
      // generate 엔드포인트 사용 (다른 모델들)
      const jsonSchema = {
        type: "object",
        properties: {
          score: {
            type: "integer",
            minimum: 0,
            maximum: 10
          }
        },
        required: ["score"]
      };
      
      response = await this.fetchWithTimeout(`${OLLAMA_URL}/api/generate`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: OLLAMA_JUDGE_MODEL,
          prompt: `${prompt}

You must respond with a JSON object containing only a score field with an integer value between 0 and 10.`,
          stream: false,
          format: jsonSchema,
          options: {
            temperature: 0,
            num_predict: 50
          }
        }),
        signal: this.abortController.signal
      }, 30000);
    }

    if (response.ok) {
      const result = await response.json();
      // chat은 message.content, generate는 response 필드에 결과 반환
      const rawContent = useChat ? 
        (result.message?.content?.trim() || "") : 
        (result.response?.trim() || "");
      
      // 디버깅: 실제 응답 내용 확인
      console.debug(`Ollama Judge 원본 응답: "${rawContent}"`);
      
      let score: number | null = null;
      
      // 1. JSON 파싱 시도 (주된 경로)
      try {
        const jsonData = JSON.parse(rawContent);
        if (typeof jsonData.score === 'number') {
          score = jsonData.score;
        } else if (typeof jsonData.score === 'string') {
          score = parseInt(jsonData.score);
        }
      } catch (jsonError) {
        console.debug("JSON 파싱 실패, 숫자 추출 시도");
        
        // 2. 단순 숫자 추출 (백업)
        const cleanContent = rawContent.replace(/[^0-9]/g, '');
        if (cleanContent) {
          const firstNumber = parseInt(cleanContent.charAt(0) === '1' && cleanContent.charAt(1) === '0' ? '10' : cleanContent.charAt(0));
          if (!isNaN(firstNumber)) {
            score = firstNumber;
          }
        }
        
        // 3. 텍스트 패턴 매칭 (백업)
        if (score === null) {
          const numberMatch = rawContent.match(/([0-9]|10)/);
          if (numberMatch) {
            score = parseInt(numberMatch[0]);
          }
        }
      }
      
      // 점수 검증
      if (score !== null && score >= 0 && score <= 10) {
        console.debug(`Ollama Judge 평가 성공: ${score}점`);
        return { score, judge_model: OLLAMA_JUDGE_MODEL, source: 'ollama' };
      }
      
      // 빈 응답인 경우 특별 처리 - 중간 점수 반환
      if (rawContent === "") {
        console.warn("Ollama Judge가 빈 응답을 반환함. 기본 점수 5점 사용.");
        return { score: 5, judge_model: OLLAMA_JUDGE_MODEL, source: 'ollama-default' };
      }
      
      // 점수를 찾을 수 없는 경우
      console.warn(`Ollama Judge 점수 파싱 실패. 응답: "${rawContent}"`);
    }

    throw new Error(`Ollama Judge 실패: ${response.status}`);
  }

  // VLLM Judge 시도
  private async tryVLLMJudge(prompt: string): Promise<{score: number, judge_model: string, source: string} | null> {
    console.debug(`Judge 평가 시도: VLLM ${VLLM_JUDGE_MODEL}`);
    
    // app.py 패턴: 30초 타임아웃 적용
    const response = await this.fetchWithTimeout(`${VLLM_SERVER}/v1/chat/completions`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        model: VLLM_JUDGE_MODEL,
        messages: [
          { role: "system", content: "당신은 AI 답변 평가자입니다. 반드시 점수만 숫자로 출력하세요." },
          { role: "user", content: prompt }
        ],
        max_tokens: 10,
        temperature: 0
      }),
      signal: this.abortController.signal
    }, 30000);

    if (response.ok) {
      const result = await response.json();
      const rawContent = result.choices?.[0]?.message?.content?.trim() || "";
      
      // 디버깅: 실제 응답 내용 확인
      if (rawContent.length > 20) {
        console.debug(`VLLM Judge 원본 응답 (처음 100자): ${rawContent.substring(0, 100)}...`);
      }
      
      // 숫자만 추출 (0-10 사이)
      const numberMatch = rawContent.match(/([0-9]|10)/);
      if (numberMatch) {
        const score = parseInt(numberMatch[0]);
        if (score >= 0 && score <= 10) {
          console.debug(`VLLM Judge 평가 성공: ${score}점`);
          return { score, judge_model: VLLM_JUDGE_MODEL, source: 'vllm' };
        }
      }
      
      // 점수를 찾을 수 없는 경우
      console.warn(`VLLM Judge 점수 파싱 실패. 응답: "${rawContent}"`);
    }

    throw new Error(`VLLM Judge 실패: ${response.status}`);
  }

  // app.py 패턴: 모델별 추론 함수 (타임아웃 적용)
  async queryModel(modelName: string, question: string): Promise<string> {
    const modelType = this.getModelType(modelName);
    
    if (modelType === 'vllm') {
      // app.py 패턴: 60초 타임아웃으로 VLLM API 호출
      const response = await this.fetchWithTimeout(`${VLLM_SERVER}/v1/chat/completions`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: modelName,
          messages: [
            { role: 'user', content: question }
          ],
          max_tokens: 150,
          temperature: 0.1
        }),
        signal: this.abortController.signal
      }, 60000);

      if (response.ok) {
        const data = await response.json();
        return data.choices[0]?.message?.content?.trim() || "";
      } else {
        throw new Error(`VLLM API 오류: ${response.status}`);
      }
    } else {
      // app.py 패턴: 60초 타임아웃으로 Ollama API 호출
      const response = await this.fetchWithTimeout(`${OLLAMA_URL}/api/generate`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: modelName,
          prompt: question,
          stream: false
        }),
        signal: this.abortController.signal
      }, 60000);

      if (response.ok) {
        const data = await response.json();
        return data.response?.trim() || "";
      } else {
        throw new Error(`Ollama API 오류: ${response.status}`);
      }
    }
  }

  private getModelType(modelName: string): 'ollama' | 'vllm' {
    if (modelName.includes('/')) return 'vllm';
    if (modelName.includes('gpt-') || modelName.includes('claude-')) return 'vllm';
    return 'ollama';
  }

  // 각 카테고리에서 무작위로 문제 선택 (통계적 신뢰성 향상)
  private selectRandomQuestions(allQuestions: typeof QUESTIONS): typeof QUESTIONS {
    const categories = ['math', 'coding', 'reasoning', 'knowledge', 'language'];
    const selectedQuestions: typeof QUESTIONS = [];
    
    categories.forEach(category => {
      const categoryQuestions = allQuestions.filter(q => q.category === category);
      
      if (categoryQuestions.length === 0) return;
      
      // 각 카테고리에서 최소 1개, 최대 2개 선택 (난이도별로)
      const difficulties = ['easy', 'medium', 'hard'];
      
      difficulties.forEach(difficulty => {
        const difficultyQuestions = categoryQuestions.filter(q => q.difficulty === difficulty);
        if (difficultyQuestions.length > 0) {
          // 무작위로 하나 선택
          const randomIndex = Math.floor(Math.random() * difficultyQuestions.length);
          selectedQuestions.push(difficultyQuestions[randomIndex]);
        }
      });
    });
    
    // 최소 5개 문제는 보장 (카테고리당 1개)
    if (selectedQuestions.length < 5) {
      categories.forEach(category => {
        if (!selectedQuestions.some(q => q.category === category)) {
          const categoryQuestions = allQuestions.filter(q => q.category === category);
          if (categoryQuestions.length > 0) {
            selectedQuestions.push(categoryQuestions[0]);
          }
        }
      });
    }
    
    console.log(`총 ${selectedQuestions.length}개 문제 선택됨 (전체 ${allQuestions.length}개 중)`);
    return selectedQuestions;
  }

  // app.py 패턴: 질문별 병렬 평가 (모델당)
  async evaluateModelQuestions(model: string, questions: typeof QUESTIONS, results: any, reEvaluate: boolean): Promise<void> {
    console.log(`모델 ${model} 질문들을 병렬로 평가 시작`);
    
    // 개선: 각 카테고리에서 무작위로 문제 선택하여 다양성 확보
    const selectedQuestions = this.selectRandomQuestions(questions);
    
    // app.py 패턴: Promise.allSettled로 모든 질문을 병렬 처리
    const questionPromises = selectedQuestions.map(async (question, questionIndex) => {
      try {
        // 재평가 모드가 아니고 이미 결과가 있으면 건너뛰기
        if (!reEvaluate && results[model]?.[question.category]) {
          console.debug(`  ${question.category}: 이미 완료됨 (건너뛰기)`);
          return { success: true, skipped: true, category: question.category };
        }

        console.debug(`  카테고리 ${questionIndex + 1}/${questions.length}: ${question.category} 평가 중...`);

        // 모델에게 질문하기
        console.debug(`    질문: ${question.question}`);
        const startTime = Date.now();
        const answer = await this.queryModel(model, question.question);
        const responseTime = Date.now() - startTime;
        console.debug(`    답변: ${answer} (응답시간: ${responseTime}ms)`);
        
        // Judge 모델로 점수 평가
        console.debug(`    Judge 평가 중...`);
        const judgeStartTime = Date.now();
        const judgeResult = await this.judgeScoreWithInfo(question.question, answer, question.answer);
        const judgeTime = Date.now() - judgeStartTime;
        console.debug(`    점수: ${judgeResult.score}/10 (Judge: ${judgeResult.judge_model}, 평가시간: ${judgeTime}ms)`);
        
        if (!results[model]) {
          results[model] = {};
        }
        
        results[model][question.category] = {
          question: question.question,
          model_answer: answer,
          score: judgeResult.score,
          judge_model: judgeResult.judge_model,
          judge_source: judgeResult.source,
          model_type: this.getModelType(model),
          response_time_ms: responseTime,
          judge_time_ms: judgeTime,
          evaluated_at: new Date().toISOString(),
          re_evaluated: reEvaluate
        };

        return { success: true, category: question.category };

      } catch (error) {
        console.error(`    ${question.category} 평가 오류: ${error}`);
        
        if (!results[model]) {
          results[model] = {};
        }
        
        results[model][question.category] = {
          question: question.question,
          model_answer: "",
          score: 0,
          error: String(error),
          model_type: this.getModelType(model),
          evaluated_at: new Date().toISOString(),
          re_evaluated: reEvaluate
        };

        return { success: false, category: question.category, error: String(error) };
      }
    });

    // app.py 패턴: 모든 질문을 병렬로 처리하되 결과는 기다림
    const questionResults = await Promise.allSettled(questionPromises);
    
    console.log(`모델 ${model} 병렬 평가 완료:`, {
      total: questionResults.length,
      successful: questionResults.filter(r => r.status === 'fulfilled' && r.value.success).length,
      failed: questionResults.filter(r => r.status === 'rejected' || (r.status === 'fulfilled' && !r.value.success)).length
    });
  }

  // 평가 중단
  abort() {
    this.abortController.abort();
    console.log('평가 클라이언트: 중단 신호 전송됨');
  }
}

// 평가 상태 업데이트
function updateStatus(status: any) {
  try {
    fs.writeFileSync(STATUS_FILE, JSON.stringify(status, null, 2));
  } catch (error) {
    console.error('상태 업데이트 실패:', error);
  }
}

// 평가 중단 여부 확인
function isEvaluationStopped(): boolean {
  try {
    if (fs.existsSync(STATUS_FILE)) {
      const status = JSON.parse(fs.readFileSync(STATUS_FILE, 'utf-8'));
      return status.isStopped === true;
    }
  } catch (error) {
    console.error('상태 확인 실패:', error);
  }
  return false;
}

// 결과 저장
function saveResults(results: any) {
  try {
    fs.writeFileSync(RESULTS_FILE, JSON.stringify(results, null, 2));
  } catch (error) {
    console.error('결과 저장 실패:', error);
  }
}

// 모델 유형 감지 함수
function getModelType(modelName: string): 'ollama' | 'vllm' {
  if (modelName.includes('/')) return 'vllm';
  if (modelName.includes('gpt-') || modelName.includes('claude-')) return 'vllm';
  return 'ollama';
}

// POST: app.py 패턴을 적용한 개선된 모델 평가 시작
export async function POST(request: NextRequest) {
  let evaluationClient: ImprovedEvaluationClient | null = null;
  
  try {
    const { models, reEvaluate = false, judgeProvider = 'ollama' } = await request.json();
    
    if (!models || !Array.isArray(models) || models.length === 0) {
      return NextResponse.json({ error: '평가할 모델을 선택해주세요.' }, { status: 400 });
    }
    
    // 임베딩 모델 자동 제외
    const embeddingModels = [
      'bge-m3',
      'bge-m3:latest',
      'mxbai-embed-large',
      'mxbai-embed-large:latest',
      'nomic-embed-text',
      'nomic-embed-text:latest',
      'text-embedding-ada-002',
      'text-embedding-3-small',
      'text-embedding-3-large',
      'embed',
      'embedding'
    ];
    
    // 임베딩 모델 필터링
    const filteredModels = models.filter(model => {
      const modelLower = model.toLowerCase();
      
      // 임베딩 관련 키워드 체크
      if (modelLower.includes('embed') || 
          modelLower.includes('bge') || 
          modelLower.includes('e5') ||
          modelLower.includes('gte') ||
          modelLower.includes('nomic') ||
          modelLower.includes('mxbai')) {
        console.log(`임베딩 모델 ${model}을(를) 평가에서 제외합니다.`);
        return false;
      }
      
      // 정확한 모델명 체크
      if (embeddingModels.some(embedModel => 
        modelLower === embedModel.toLowerCase() || 
        modelLower.startsWith(embedModel.toLowerCase()))) {
        console.log(`임베딩 모델 ${model}을(를) 평가에서 제외합니다.`);
        return false;
      }
      
      return true;
    });
    
    if (filteredModels.length === 0) {
      return NextResponse.json({ 
        error: '선택한 모델이 모두 임베딩 모델입니다. 언어 모델을 선택해주세요.' 
      }, { status: 400 });
    }
    
    if (filteredModels.length < models.length) {
      console.log(`${models.length - filteredModels.length}개의 임베딩 모델이 제외되었습니다.`);
    }

    console.log(`평가 시작: ${reEvaluate ? '재평가' : '새 평가'} 모드`);
    console.log('대상 모델 (임베딩 제외 후):', filteredModels);

    // app.py 패턴: 개선된 평가 클라이언트 생성
    evaluationClient = new ImprovedEvaluationClient(judgeProvider);

    // 기존 결과 로드 (재평가 모드인 경우)
    let results: any = {};
    if (reEvaluate && fs.existsSync(RESULTS_FILE)) {
      try {
        const existingData = fs.readFileSync(RESULTS_FILE, 'utf-8');
        results = JSON.parse(existingData);
        console.log('기존 결과 로드 완료');
      } catch (error) {
        console.warn('기존 결과 로드 실패, 새로 시작:', error);
        results = {};
      }
    } else if (!reEvaluate) {
      // 새 평가 시작 시 기존 결과 완전 리셋
      console.log('새 평가 시작: 기존 결과를 모두 리셋합니다.');
      
      // app.py 패턴: 안전한 파일 삭제
      try {
        if (fs.existsSync(RESULTS_FILE)) {
          fs.unlinkSync(RESULTS_FILE);
          console.log('기존 결과 파일이 삭제되었습니다.');
        }
        
        if (fs.existsSync(STATUS_FILE)) {
          fs.unlinkSync(STATUS_FILE);
          console.log('기존 상태 파일이 삭제되었습니다.');
        }
      } catch (error) {
        console.error('파일 삭제 중 오류:', error);
      }
    }

    // 재평가 모드에서는 선택된 모델의 결과만 삭제
    if (reEvaluate) {
      filteredModels.forEach(model => {
        if (results[model]) {
          console.log(`기존 ${model} 결과 삭제`);
          delete results[model];
        }
      });
    }

    // 평가 상태 초기화
    const totalTasks = filteredModels.length * QUESTIONS.length;
    const initialStatus = {
      isRunning: true,
      isStopped: false,
      progress: 0,
      currentModel: filteredModels[0] || '',
      currentCategory: 'math',
      totalModels: filteredModels.length,
      completedModels: 0,
      startTime: new Date().toISOString(),
      reEvaluate: reEvaluate,
      targetModels: models,
      implementation: 'Improved Evaluation Client (app.py pattern)'
    };
    
    updateStatus(initialStatus);
    console.log('평가 상태 초기화 완료:', initialStatus);

    // 응답 먼저 반환
    const response = NextResponse.json({ 
      message: reEvaluate ? '재평가가 시작되었습니다.' : '평가가 시작되었습니다.',
      totalTasks,
      models: filteredModels.length,
      excludedModels: models.length - filteredModels.length,
      reEvaluate,
      status: initialStatus,
      features: [
        '60초 모델 추론 타임아웃',
        '30초 Judge 평가 타임아웃',
        '질문별 병렬 처리',
        'Judge 결과 1시간 캐싱',
        'AbortController 중단 지원',
        '강화된 에러 처리'
      ]
    });

    // app.py 패턴: 백그라운드에서 개선된 평가 실행
    setImmediate(async () => {
      let completedModels = 0;
      
      console.log(`=== ${reEvaluate ? '재평가' : '평가'} 시작 (개선된 방식) ===`);
      console.log(`총 ${filteredModels.length}개 모델, ${QUESTIONS.length}개 카테고리, ${totalTasks}개 작업`);
      if (models.length - filteredModels.length > 0) {
        console.log(`${models.length - filteredModels.length}개 임베딩 모델 제외됨`);
      }
      if (reEvaluate) {
        console.log('재평가 대상 모델:', filteredModels.join(', '));
      }
      
      try {
        // app.py 패턴: 모델별 순차 처리 (질문은 병렬)
        for (let modelIndex = 0; modelIndex < filteredModels.length; modelIndex++) {
          const model = filteredModels[modelIndex];
          
          // 중단 확인
          if (isEvaluationStopped()) {
            console.log('평가가 사용자에 의해 중단되었습니다.');
            updateStatus({
              isRunning: false,
              isStopped: true,
              progress: Math.round((completedModels / filteredModels.length) * 100),
              currentModel: model,
              currentCategory: '',
              totalModels: filteredModels.length,
              completedModels,
              stoppedAt: new Date().toISOString(),
            });
            return;
          }

          console.log(`
--- 모델 ${modelIndex + 1}/${filteredModels.length}: ${model} ${reEvaluate ? '재평가' : '평가'} 시작 ---`);
          
          // 모델 시작 상태 업데이트
          updateStatus({
            isRunning: true,
            isStopped: false,
            progress: Math.round((completedModels / filteredModels.length) * 100),
            currentModel: model,
            currentCategory: '병렬 처리 중',
            totalModels: models.length,
            completedModels,
          });

          try {
            // app.py 패턴: 모델별 질문들을 병렬로 평가
            await evaluationClient!.evaluateModelQuestions(model, QUESTIONS, results, reEvaluate);
            
            completedModels++;
            console.log(`--- 모델 ${model} ${reEvaluate ? '재평가' : '평가'} 완료 ---`);
            
            // 결과 저장
            saveResults(results);
            
            // 진행률 업데이트
            updateStatus({
              isRunning: true,
              isStopped: false,
              progress: Math.round((completedModels / filteredModels.length) * 100),
              currentModel: modelIndex < filteredModels.length - 1 ? filteredModels[modelIndex + 1] : '',
              currentCategory: '완료',
              totalModels: filteredModels.length,
              completedModels,
            });
            
          } catch (error) {
            console.error(`모델 ${model} 평가 중 오류:`, error);
            completedModels++;
            
            // 에러가 발생해도 다음 모델로 계속 진행
            updateStatus({
              isRunning: true,
              isStopped: false,
              progress: Math.round((completedModels / filteredModels.length) * 100),
              currentModel: modelIndex < filteredModels.length - 1 ? filteredModels[modelIndex + 1] : '',
              currentCategory: '오류',
              totalModels: filteredModels.length,
              completedModels,
              lastError: String(error)
            });
          }
          
          // app.py 패턴: 모델 간 간격 조절 (API 부하 방지)
          if (modelIndex < filteredModels.length - 1) {
            await new Promise(resolve => setTimeout(resolve, 2000));
          }
        }

        // 전체 평가 완료
        console.log(`\n=== 전체 ${reEvaluate ? '재평가' : '평가'} 완료 (개선된 방식) ===`);
        updateStatus({
          isRunning: false,
          isStopped: false,
          progress: 100,
          currentModel: '',
          currentCategory: '완료',
          totalModels: models.length,
          completedModels: filteredModels.length,
          endTime: new Date().toISOString(),
          implementation: 'Improved Evaluation Client (app.py pattern)'
        });

      } catch (error) {
        console.error('평가 중 치명적 오류 발생:', error);
        updateStatus({
          isRunning: false,
          isStopped: false,
          progress: Math.round((completedModels / models.length) * 100),
          error: String(error),
          endTime: new Date().toISOString(),
          implementation: 'Improved Evaluation Client (app.py pattern - Error)'
        });
      } finally {
        // app.py 패턴: 리소스 정리
        if (evaluationClient) {
          try {
            evaluationClient.abort();
          } catch (error) {
            console.error('평가 클라이언트 정리 중 오류:', error);
          }
        }
      }
    });

    return response;

  } catch (error) {
    console.error('평가 시작 오류:', error);
    
    // app.py 패턴: 에러 발생 시에도 리소스 정리
    if (evaluationClient) {
      try {
        evaluationClient.abort();
      } catch (cleanupError) {
        console.error('에러 후 정리 중 오류:', cleanupError);
      }
    }
    
    return NextResponse.json({ 
      error: '평가를 시작할 수 없습니다.',
      details: (error as Error).message,
      implementation: 'Improved Evaluation Client (app.py pattern - Error)' 
    }, { status: 500 });
  }
} 