import { NextRequest, NextResponse } from 'next/server';
import { createReactAgent } from '@langchain/langgraph/prebuilt';
import { ChatOpenAI } from '@langchain/openai';
import { ChatAnthropic } from '@langchain/anthropic';
import { tool } from '@langchain/core/tools';
import { z } from 'zod';
import fs from 'fs';
import path from 'path';
import os from 'os';
import { spawn, ChildProcess } from 'child_process';
import Anthropic from '@anthropic-ai/sdk';
import { GoogleGenerativeAI } from '@google/generative-ai';
import Groq from 'groq-sdk';

// MCP 결과 타입 정의
interface MCPResult {
  content: Array<{
    type: string;
    text: string;
  }>;
}

// 툴 스키마 타입 정의
type ToolSchema = {
  [key: string]: any;
};

// 설정 파일 경로
const getAirunConfigPath = () => {
  const homeDir = os.homedir();
  return path.join(homeDir, '.airun', 'airun.conf');
};

const getMCPConfigPath = () => {
  return path.join(process.cwd(), 'mcp_config.json');
};

// airun.conf에서 설정 읽기
const getAirunConfig = () => {
  const configPath = getAirunConfigPath();
  const config: any = {};
  
  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      for (const line of content.split('\n')) {
        const trimmed = line.trim();
        if (trimmed.startsWith('export ')) {
          const match = trimmed.match(/export\s+([A-Z_]+)=["']?([^"']+)["']?/);
          if (match) {
            config[match[1]] = match[2];
          }
        }
      }
    } catch (error) {
      console.error('airun.conf 읽기 실패:', error);
    }
  }
  
  return config;
};

// MCP 설정 로드
const loadMCPConfig = () => {
  const configPath = getMCPConfigPath();
  const defaultConfig = {
    "hamonize": {
      "command": "node",
      "args": ["./mcp_tools/hamonize/dist/index.js"],
      "transport": "stdio"
    },
    "postgres": {
      "transport": "stdio",
      "command": "node",
      "args": ["./mcp_tools/mcp-postgres-server/build/index.js"]
    }
  };

  if (fs.existsSync(configPath)) {
    try {
      const content = fs.readFileSync(configPath, 'utf8');
      return JSON.parse(content);
    } catch (error) {
      console.error('MCP 설정 로드 실패:', error);
      return defaultConfig;
    }
  }
  
  return defaultConfig;
};

// MCP 클라이언트 클래스
class MCPClient {
  private processes: Map<string, ChildProcess> = new Map();
  private messageId = 1;

  async initialize(mcpConfig: any) {
    console.log('🔄 MCP 클라이언트 초기화 시작...');
    
    for (const [serverName, config] of Object.entries(mcpConfig)) {
      await this.startServer(serverName, config as any);
    }
    
    await new Promise(resolve => setTimeout(resolve, 2000));
    console.log('✅ MCP 클라이언트 초기화 완료');
  }

  private async startServer(serverName: string, config: any) {
    const { command, args, transport, env } = config;
    
    if (transport !== 'stdio') {
      console.warn(`${serverName}: stdio transport만 지원됩니다.`);
      return;
    }

    try {
      const spawnEnv = { ...process.env };
      if (env && typeof env === 'object') {
        Object.assign(spawnEnv, env);
      }

      const childProcess = spawn(command, args, {
        stdio: ['pipe', 'pipe', 'pipe'],
        cwd: process.cwd(),
        env: spawnEnv
      });

      childProcess.stderr?.on('data', (data: Buffer) => {
        console.error(`${serverName} stderr:`, data.toString());
      });

      this.processes.set(serverName, childProcess);
      
      await new Promise(resolve => setTimeout(resolve, 1000));
      
      // 서버 초기화
      await this.sendRequest(serverName, 'initialize', {
        protocolVersion: '2024-11-05',
        capabilities: {
          roots: { listChanged: true },
          sampling: {}
        },
        clientInfo: {
          name: 'hamonize-langgraph-client',
          version: '1.0.0'
        }
      });

      console.log(`${serverName} MCP 서버가 초기화되었습니다.`);
    } catch (error) {
      console.error(`${serverName} MCP 서버 시작 실패:`, error);
    }
  }

  private async sendRequest(serverName: string, method: string, params: any = {}) {
    const childProcess = this.processes.get(serverName);
    if (!childProcess || !childProcess.stdin) {
      throw new Error(`${serverName} 서버가 실행 중이지 않습니다.`);
    }

    const request: any = {
      jsonrpc: '2.0',
      id: this.messageId++,
      method
    };

    if (params !== null && params !== undefined) {
      request.params = params;
    }

    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error(`${serverName} 요청 시간 초과`));
      }, 30000);

      let buffer = '';
      const onData = (data: Buffer) => {
        buffer += data.toString();
        
        while (true) {
          const newlineIndex = buffer.indexOf('\n');
          if (newlineIndex === -1) break;
          
          const line = buffer.slice(0, newlineIndex);
          buffer = buffer.slice(newlineIndex + 1);
          
          if (line.trim()) {
            try {
              const response = JSON.parse(line);
              if (response.id === request.id) {
                clearTimeout(timeout);
                childProcess.stdout?.off('data', onData);
                
                if (response.error) {
                  reject(new Error(`${serverName}: ${response.error.message}`));
                } else {
                  resolve(response.result);
                }
                return;
              }
            } catch (e) {
              // JSON 파싱 오류는 무시하고 계속 진행
            }
          }
        }
      };

      childProcess.stdout?.on('data', onData);
      childProcess.stdin?.write(JSON.stringify(request) + '\n');
    });
  }

  async callTool(serverName: string, toolName: string, arguments_: any): Promise<MCPResult> {
    try {
      const result = await this.sendRequest(serverName, 'tools/call', {
        name: toolName,
        arguments: arguments_
      });
      
      return result as MCPResult;
    } catch (error) {
      console.error(`${serverName} 도구 호출 실패 (${toolName}):`, error);
      throw error;
    }
  }

  cleanup() {
    for (const [serverName, childProcess] of this.processes) {
      try {
        childProcess.kill();
        console.log(`${serverName} MCP 서버 종료됨`);
      } catch (error) {
        console.error(`${serverName} 서버 종료 실패:`, error);
      }
    }
    this.processes.clear();
  }
}

// 글로벌 MCP 클라이언트
let globalMCPClient: MCPClient | null = null;
let isInitializing = false;
let initializationComplete = false;

// MCP 클라이언트 초기화
const initializeMCPClient = async () => {
  if (globalMCPClient && initializationComplete) {
    return globalMCPClient;
  }

  if (isInitializing) {
    while (isInitializing) {
      await new Promise(resolve => setTimeout(resolve, 100));
    }
    return globalMCPClient;
  }

  isInitializing = true;
  
  try {
    const mcpConfig = loadMCPConfig();
    globalMCPClient = new MCPClient();
    await globalMCPClient.initialize(mcpConfig);
    initializationComplete = true;
    console.log('✅ 글로벌 MCP 클라이언트 초기화 완료');
  } catch (error) {
    console.error('❌ MCP 클라이언트 초기화 실패:', error);
    globalMCPClient = null;
    initializationComplete = false;
  } finally {
    isInitializing = false;
  }

  return globalMCPClient;
};

// LangGraph.js tools 정의
const webSearchTool = tool(
  async (input: any) => {
    const { query } = input;
    console.log(`🔍 웹 검색 도구 호출됨: "${query}"`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'web_search', { query });
      console.log(`✅ 웹 검색 완료: ${query}`);
      return result.content[0]?.text || '검색 결과가 없습니다.';
    } catch (error) {
      console.error('웹 검색 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`웹 검색 실패: ${errorMessage}`);
    }
  },
  {
    name: "web_search",
    description: "웹에서 정보를 검색합니다. 최신 뉴스, 기술 동향, 일반적인 정보 등을 찾을 때 사용합니다.",
    schema: z.object({
      query: z.string().describe("검색할 키워드나 질문"),
    }),
  }
);

const translateTool = tool(
  async (input: any) => {
    const { text, target_lang } = input;
    console.log(`🌍 번역 도구 호출됨: "${text}" -> ${target_lang}`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'translate_text', { 
        text, 
        target_lang: target_lang || 'KO' 
      });
      console.log(`✅ 번역 완료: ${text} -> ${target_lang}`);
      return result.content[0]?.text || '번역 결과가 없습니다.';
    } catch (error) {
      console.error('번역 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`번역 실패: ${errorMessage}`);
    }
  },
  {
    name: "translate_text",
    description: "텍스트를 다른 언어로 번역합니다. 한국어, 영어, 일본어, 중국어 등 다양한 언어를 지원합니다.",
    schema: z.object({
      text: z.string().describe("번역할 텍스트"),
      target_lang: z.string().default("KO").describe("번역할 언어 (KO, EN, JA, ZH 등)"),
    }),
  }
);

const generateCodeTool = tool(
  async (input: any) => {
    const { prompt, language } = input;
    console.log(`⚡ 코드 생성 도구 호출됨: "${prompt}" (${language})`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'generate_code', { 
        prompt, 
        language: language || 'python' 
      });
      console.log(`✅ 코드 생성 완료: ${prompt}`);
      return result.content[0]?.text || '코드 생성 결과가 없습니다.';
    } catch (error) {
      console.error('코드 생성 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`코드 생성 실패: ${errorMessage}`);
    }
  },
  {
    name: "generate_code",
    description: "지정된 프로그래밍 언어로 코드를 생성합니다. 함수, 클래스, 스크립트 등을 생성할 수 있습니다.",
    schema: z.object({
      prompt: z.string().describe("코드 생성에 사용할 프롬프트나 요구사항"),
      language: z.string().default("python").describe("생성할 코드의 프로그래밍 언어 (python, javascript, java, c++ 등)"),
    }),
  }
);

const ragSearchTool = tool(
  async (input: any) => {
    const { query } = input;
    console.log(`📚 RAG 검색 도구 호출됨: "${query}"`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'rag_search', { query });
      console.log(`✅ RAG 검색 완료: ${query}`);
      return result.content[0]?.text || 'RAG 검색 결과가 없습니다.';
    } catch (error) {
      console.error('RAG 검색 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`RAG 검색 실패: ${errorMessage}`);
    }
  },
  {
    name: "rag_search",
    description: "문서 데이터베이스에서 관련 정보를 검색합니다. 업로드된 문서나 지식 베이스에서 정보를 찾을 때 사용합니다.",
    schema: z.object({
      query: z.string().describe("검색할 키워드나 질문"),
    }),
  }
);

const createReportTool = tool(
  async (input: any) => {
    const { content, format, template } = input;
    console.log(`📄 리포트 생성 도구 호출됨: ${format} 형식, ${template} 템플릿`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'create_report', { 
        content, 
        format: format || 'pdf', 
        template: template || 'simple' 
      });
      console.log(`✅ 리포트 생성 완료: ${format} 형식`);
      return result.content[0]?.text || '리포트 생성 결과가 없습니다.';
    } catch (error) {
      console.error('리포트 생성 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`리포트 생성 실패: ${errorMessage}`);
    }
  },
  {
    name: "create_report",
    description: "지정된 내용으로 리포트를 생성합니다. PDF, Word, HTML 등 다양한 형식을 지원합니다.",
    schema: z.object({
      content: z.string().describe("리포트에 포함할 내용"),
      format: z.string().default("pdf").describe("리포트 형식 (pdf, docx, html 등)"),
      template: z.string().default("simple").describe("리포트 템플릿 (simple, professional, academic 등)"),
    }),
  }
);

const extractDocumentTextTool = tool(
  async (input: any) => {
    const { document_path, use_ocr, lang } = input;
    console.log(`📖 문서 텍스트 추출 도구 호출됨: "${document_path}"`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'extract_document_text', { 
        document_path, 
        use_ocr: use_ocr || false, 
        lang: lang || 'kor' 
      });
      console.log(`✅ 문서 텍스트 추출 완료: ${document_path}`);
      return result.content[0]?.text || '문서 텍스트 추출 결과가 없습니다.';
    } catch (error) {
      console.error('문서 텍스트 추출 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`문서 텍스트 추출 실패: ${errorMessage}`);
    }
  },
  {
    name: "extract_document_text",
    description: "문서 파일에서 텍스트를 추출합니다. PDF, Word, HWP 등 다양한 형식을 지원하며, OCR 기능도 제공합니다.",
    schema: z.object({
      document_path: z.string().describe("추출할 문서 파일의 경로"),
      use_ocr: z.boolean().default(false).describe("OCR(광학 문자 인식) 사용 여부"),
      lang: z.string().default("kor").describe("OCR 언어 설정 (kor, eng, jpn 등)"),
    }),
  }
);

const createDocumentTool = tool(
  async (input: any) => {
    const { content, format, output_path, template } = input;
    console.log(`📝 문서 생성 도구 호출됨: ${format} 형식`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'create_document', { 
        content, 
        format: format || 'pdf', 
        output_path, 
        template: template || 'simple' 
      });
      console.log(`✅ 문서 생성 완료: ${format} 형식`);
      return result.content[0]?.text || '문서 생성 결과가 없습니다.';
    } catch (error) {
      console.error('문서 생성 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`문서 생성 실패: ${errorMessage}`);
    }
  },
  {
    name: "create_document",
    description: "지정된 내용으로 문서를 생성합니다. PDF, Word, HTML 등 다양한 형식을 지원합니다.",
    schema: z.object({
      content: z.string().describe("문서에 포함할 내용"),
      format: z.string().default("pdf").describe("문서 형식 (pdf, docx, html 등)"),
      output_path: z.string().optional().describe("출력 파일 경로 (옵션)"),
      template: z.string().default("simple").describe("문서 템플릿 (simple, professional, academic 등)"),
    }),
  }
);

const createChartTool = tool(
  async (input: any) => {
    const { data, chart_type, output_path } = input;
    console.log(`📊 차트 생성 도구 호출됨: ${chart_type} 타입`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'create_chart', { 
        data, 
        chart_type: chart_type || 'line', 
        output_path 
      });
      console.log(`✅ 차트 생성 완료: ${chart_type} 타입`);
      return result.content[0]?.text || '차트 생성 결과가 없습니다.';
    } catch (error) {
      console.error('차트 생성 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`차트 생성 실패: ${errorMessage}`);
    }
  },
  {
    name: "create_chart",
    description: "데이터를 시각화한 차트를 생성합니다. 선형, 막대, 원형 등 다양한 차트 타입을 지원합니다.",
    schema: z.object({
      data: z.record(z.any()).describe("차트에 사용할 데이터 (JSON 형식)"),
      chart_type: z.string().default("line").describe("차트 타입 (line, bar, pie, scatter 등)"),
      output_path: z.string().optional().describe("출력 파일 경로 (옵션)"),
    }),
  }
);

const fileOperationTool = tool(
  async (input: any) => {
    const { operation, path, content, mode } = input;
    console.log(`📁 파일 작업 도구 호출됨: ${operation} 작업`);
    
    const mcpClient = await initializeMCPClient();
    if (!mcpClient) {
      throw new Error('MCP 클라이언트 초기화 실패');
    }

    try {
      const result = await mcpClient.callTool('hamonize', 'file_operation', { 
        operation, 
        path, 
        content, 
        mode 
      });
      console.log(`✅ 파일 작업 완료: ${operation}`);
      return result.content[0]?.text || '파일 작업 결과가 없습니다.';
    } catch (error) {
      console.error('파일 작업 실패:', error);
      const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
      throw new Error(`파일 작업 실패: ${errorMessage}`);
    }
  },
  {
    name: "file_operation",
    description: "파일 및 디렉토리 작업을 수행합니다. 읽기, 쓰기, 생성, 삭제, 이동 등을 지원합니다.",
    schema: z.object({
      operation: z.string().describe("작업 타입 (read, write, create, delete, move, copy 등)"),
      path: z.string().describe("대상 파일이나 디렉토리 경로"),
      content: z.any().optional().describe("파일 내용 (쓰기 작업 시 필요)"),
      mode: z.string().optional().describe("파일 모드 (권한 설정)"),
    }),
  }
);

// 모든 도구를 배열로 정의
const allTools = [
  webSearchTool,
  translateTool,
  generateCodeTool,
  ragSearchTool,
  createReportTool,
  extractDocumentTextTool,
  createDocumentTool,
  createChartTool,
  fileOperationTool,
];

// LLM 모델 생성 함수
function createLLMModel(provider: string, model: string, config: any, temperature?: number) {
  switch (provider) {
    case 'openai':
      return new ChatOpenAI({
        apiKey: config.OPENAI_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    case 'anthropic':
      return new ChatAnthropic({
        apiKey: config.ANTHROPIC_API_KEY,
        modelName: model,
        temperature: temperature || 0.7,
        streaming: true,
      });
    
    default:
      throw new Error(`지원하지 않는 프로바이더: ${provider}`);
  }
}

// 스트리밍 응답 생성 함수
const createStreamingResponse = (generator: () => AsyncGenerator<string>) => {
  return new Response(
    new ReadableStream({
      async start(controller) {
        try {
          for await (const chunk of generator()) {
            controller.enqueue(new TextEncoder().encode(chunk));
          }
        } catch (error) {
          console.error('스트리밍 오류:', error);
          const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
          controller.enqueue(new TextEncoder().encode(`오류: ${errorMessage}`));
        } finally {
          controller.close();
        }
      },
    }),
    {
      headers: {
        'Content-Type': 'text/plain; charset=utf-8',
        'Cache-Control': 'no-cache',
        'Connection': 'keep-alive',
      },
    }
  );
};

// LangGraph 에이전트로 처리
async function* processWithLangGraph(
  message: string,
  files: any[],
  provider: string,
  model: string,
  temperature?: number,
  history?: Array<{role: 'user' | 'assistant', content: string}>
) {
  console.log(`🚀 LangGraph 에이전트 처리 시작: ${provider}/${model}`);
  
  try {
    // 설정 로드
    const config = getAirunConfig();
    
    // LLM 모델 생성
    const llm = createLLMModel(provider, model, config, temperature);
    
    // LangGraph 에이전트 생성
    console.log('🔧 LangGraph 에이전트 생성 중...');
    const agent = createReactAgent({
      llm,
      tools: allTools,
    });
    
    // console.log('✅ LangGraph 에이전트 생성 완료');
    
    // 메시지 준비
    let finalMessage = message;
    if (files && files.length > 0) {
      const fileDescriptions = files.map(file => {
        let desc = `파일: "${file.name}" (${file.type})`;
        if (file.extractedText) {
          desc += `\n텍스트 내용:\n${file.extractedText}`;
        }
        if (file.isImage && file.base64) {
          desc += `\n이미지가 첨부되었습니다.`;
        }
        return desc;
      }).join('\n\n');
      
      finalMessage = `${fileDescriptions}\n\n질문: ${message}`;
    }
    
    // 히스토리를 메시지 형식으로 변환
    const messages = [];
    if (history && history.length > 0) {
      for (const msg of history) {
        messages.push({
          role: msg.role === 'user' ? 'human' : 'assistant',
          content: msg.content,
        });
      }
    }
    
    // 현재 메시지 추가
    messages.push({
      role: 'human',
      content: finalMessage,
    });
    
    console.log('📤 LangGraph 에이전트 실행 중...');
    
    // 에이전트 실행
    const result = await agent.invoke({
      messages: messages,
    });
    
    console.log('✅ LangGraph 에이전트 처리 완료');
    
    // 결과 처리
    if (result && result.messages && result.messages.length > 0) {
      const lastMessage = result.messages[result.messages.length - 1];
      
      // 응답 텍스트 추출 - 타입 안전하게 처리
      let responseText = '';
      if (lastMessage.content) {
        // content가 string이면 그대로 사용, 배열이면 첫 번째 요소의 text 사용
        if (typeof lastMessage.content === 'string') {
          responseText = lastMessage.content;
        } else if (Array.isArray(lastMessage.content) && lastMessage.content.length > 0) {
          const firstContent = lastMessage.content[0];
          if (typeof firstContent === 'object' && 'text' in firstContent) {
            responseText = firstContent.text;
          } else {
            responseText = String(firstContent);
          }
        } else {
          responseText = String(lastMessage.content);
        }
      } else {
        responseText = '응답을 생성했습니다.';
      }
      
      // 스트리밍 방식으로 응답 전송
      const words = responseText.split(' ');
      for (let i = 0; i < words.length; i++) {
        yield words[i] + (i < words.length - 1 ? ' ' : '');
        await new Promise(resolve => setTimeout(resolve, 50)); // 50ms 지연
      }
      
    } else {
      yield '응답을 생성할 수 없습니다.';
    }
    
  } catch (error) {
    console.error('❌ LangGraph 에이전트 처리 실패:', error);
    const errorMessage = error instanceof Error ? error.message : '알 수 없는 오류';
    yield `처리 중 오류가 발생했습니다: ${errorMessage}`;
  }
}

// POST: 채팅 메시지 처리 (LangGraph 사용)
export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const message = formData.get('message') as string;
    const provider = formData.get('provider') as string;
    const model = formData.get('model') as string;
    const temperature = parseFloat(formData.get('temperature') as string) || undefined;
    const historyString = formData.get('history') as string;
    
    let history: Array<{role: 'user' | 'assistant', content: string}> = [];
    if (historyString) {
      try {
        history = JSON.parse(historyString);
      } catch (error) {
        console.error('히스토리 파싱 실패:', error);
      }
    }
    
    if (!message || message.trim() === '') {
      return NextResponse.json(
        { error: '메시지가 필요합니다.' },
        { status: 400 }
      );
    }
    
    // 지원하는 프로바이더 확인
    if (!['openai', 'anthropic'].includes(provider)) {
      return NextResponse.json(
        { error: `현재 LangGraph 모드에서는 OpenAI와 Anthropic만 지원됩니다. (요청: ${provider})` },
        { status: 400 }
      );
    }
    
    // 다중 파일 처리
    const files: any[] = [];
    let fileIndex = 0;
    
    while (true) {
      const file = formData.get(`file_${fileIndex}`) as File | null;
      if (!file) break;
      
      const fileInfo: any = {
        name: file.name,
        type: file.type,
        size: file.size,
        file: file
      };
      
      // 텍스트 추출 데이터
      const extractedText = formData.get(`file_${fileIndex}_text`) as string | null;
      if (extractedText) {
        fileInfo.extractedText = extractedText;
      }
      
      // Base64 데이터 (이미지)
      const base64Data = formData.get(`file_${fileIndex}_base64`) as string | null;
      if (base64Data) {
        fileInfo.base64 = base64Data;
        fileInfo.isImage = true;
      }
      
      files.push(fileInfo);
      fileIndex++;
    }
    
    console.log(`🦜 LangGraph 채팅 요청: ${message}`);
    if (files.length > 0) {
      console.log(`📎 첨부 파일 ${files.length}개:`, files.map(f => `${f.name} (${f.type})`).join(', '));
    }

    // 스트리밍 응답 생성
    return createStreamingResponse(async function* () {
      yield* processWithLangGraph(message, files, provider, model, temperature, history);
    });
    
  } catch (error) {
    console.error('LangGraph 채팅 처리 실패:', error);
    return NextResponse.json(
      { error: '채팅 처리 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
}

// GET: 채팅 상태 확인
export async function GET() {
  try {
    const config = getAirunConfig();
    
    return NextResponse.json({
      available: true,
      mode: 'langgraph',
      provider: config.USE_LLM || 'anthropic',
      model: config.ANTHROPIC_MODEL || config.OPENAI_MODEL || 'default',
      supportedProviders: ['openai', 'anthropic'],
      toolsCount: allTools.length,
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error('LangGraph 채팅 상태 확인 실패:', error);
    return NextResponse.json(
      { error: '채팅 상태 확인에 실패했습니다.' },
      { status: 500 }
    );
  }
}

// 서버 종료 시 정리
process.on('SIGINT', async () => {
  if (globalMCPClient) {
    globalMCPClient.cleanup();
  }
});

process.on('SIGTERM', async () => {
  if (globalMCPClient) {
    globalMCPClient.cleanup();
  }
}); 