import { NextResponse } from 'next/server';
import { Pool } from 'pg';
import { createClient } from 'redis';
import { stat } from 'fs/promises';
import path from 'path';

interface ServiceStatus {
  name: string;
  status: 'Online' | 'Offline' | 'Error';
  details: string;
}

async function checkHttpService(name: string, url: string | undefined): Promise<ServiceStatus> {
  if (!url) {
    return { name, status: 'Offline', details: 'URL이 설정되지 않았습니다.' };
  }
  try {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 5000); // 5초 타임아웃
    const response = await fetch(url, { signal: controller.signal, method: 'HEAD' }).finally(() => clearTimeout(timeoutId));
    
    if (response.status >= 200 && response.status < 500) {
      return { name, status: 'Online', details: `HTTP ${response.status}` };
    }
    return { name, status: 'Offline', details: '서비스 응답 없음' };
  } catch (error: any) {
    if (error.name === 'AbortError') {
      return { name, status: 'Offline', details: '요청 시간 초과' };
    }
    // ECONNREFUSED 등 네트워크 오류 처리
    return { name, status: 'Offline', details: `연결 실패: ${error.message}` };
  }
}

async function checkPostgres(): Promise<ServiceStatus> {
  const name = 'PostgreSQL';
  const pool = new Pool({
    host: process.env.DB_HOST,
    port: Number(process.env.DB_PORT),
    user: process.env.DB_USER,
    password: process.env.DB_PASSWORD,
    database: process.env.DB_NAME,
    connectionTimeoutMillis: 5000,
  });

  try {
    const client = await pool.connect();
    client.release();
    await pool.end();
    return { name, status: 'Online', details: '데이터베이스 연결 성공' };
  } catch (error: any) {
    await pool.end();
    return { name, status: 'Offline', details: `연결 실패: ${error.message}` };
  }
}

async function checkRedis(): Promise<ServiceStatus> {
  const name = 'Redis';
  if (!process.env.REDIS_URL) {
    return { name, status: 'Offline', details: 'Redis URL이 설정되지 않았습니다.' };
  }
  const client = createClient({
    url: process.env.REDIS_URL,
    socket: {
      connectTimeout: 5000,
    },
  });

  try {
    await client.connect();
    await client.ping();
    await client.quit();
    return { name, status: 'Online', details: '캐시 서버 연결 성공' };
  } catch (error: any) {
    // client.quit()가 실패할 수 있으므로 강제 종료
    if (client.isOpen) {
      await client.disconnect();
    }
    return { name, status: 'Offline', details: `연결 실패: ${error.message}` };
  }
}

// GET 요청 핸들러
export async function GET() {
  const services = [
    { name: 'API', check: () => checkHttpService('API', process.env.NEXT_PUBLIC_API_SERVER_URL) },
    { name: 'RAG', check: () => checkHttpService('RAG', process.env.NEXT_PUBLIC_RAG_API_SERVER) },
    { name: '웹 검색', check: () => checkHttpService('웹 검색', process.env.NEXT_PUBLIC_WEB_API_SERVER) },
    { name: '문서 생성', check: () => checkHttpService('문서 생성', process.env.NEXT_PUBLIC_REPORT_API_SERVER) },
    { name: 'PostgreSQL', check: checkPostgres },
    { name: 'Redis', check: checkRedis },
  ];

  try {
    const results = await Promise.allSettled(services.map(s => s.check()));

    const statuses = results.map((result, index) => {
      if (result.status === 'fulfilled') {
        return result.value;
      } else {
        // Promise가 reject된 경우
        return {
          name: services[index].name,
          status: 'Error',
          details: result.reason instanceof Error ? result.reason.message : '알 수 없는 오류',
        } as ServiceStatus;
      }
    });

    return NextResponse.json({ success: true, data: statuses });
  } catch (error) {
    const errorMessage = error instanceof Error ? error.message : '알 수 없는 서버 오류';
    return NextResponse.json({ success: false, error: { message: `서비스 상태 확인 중 오류 발생: ${errorMessage}` } }, { status: 500 });
  }
} 