import { NextRequest, NextResponse } from 'next/server';
import * as XLSX from 'xlsx';

export async function POST(request: NextRequest) {
  try {
    const formData = await request.formData();
    const file = formData.get('file') as File;

    if (!file) {
      return NextResponse.json(
        { success: false, error: 'No file provided' },
        { status: 400 }
      );
    }

    // Excel 파일 분석
    const buffer = Buffer.from(await file.arrayBuffer());
    const workbook = XLSX.read(buffer, { type: 'buffer' });

    const sheets = workbook.SheetNames.map(sheetName => {
      const worksheet = workbook.Sheets[sheetName];
      const jsonData = XLSX.utils.sheet_to_json(worksheet, { header: 1, defval: '' });
      const headers = jsonData[0] as string[] || [];
      const preview = jsonData.slice(0, 11); // 첫 10행 미리보기
      
      return {
        name: sheetName,
        headers,
        preview,
        rowCount: jsonData.length,
        columnCount: headers.length
      };
    });

    return NextResponse.json({
      success: true,
      data: {
        sheets,
        totalSheets: sheets.length
      }
    });

  } catch (error) {
    console.error('Excel analysis error:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: error instanceof Error ? error.message : 'Analysis failed' 
      },
      { status: 500 }
    );
  }
}