import { NextResponse } from 'next/server';
import { pg } from "@/app/api/db";
import { format } from 'date-fns';

export async function GET() {
  try {
    const issuesQuery = `
      SELECT
        st.id,
        st.title,
        st.question,
        st.status,
        st.author_name,
        st.created_at,
        json_agg(
          json_build_object(
            'authorName', sr.author_name,
            'authorRole', sr.author_role,
            'content', sr.content,
            'createdAt', sr.created_at
          ) ORDER BY sr.created_at ASC
        ) FILTER (WHERE sr.id IS NOT NULL) as replies
      FROM support_tickets st
      LEFT JOIN support_ticket_replies sr ON st.id = sr.ticket_id
      GROUP BY st.id
      ORDER BY st.created_at DESC;
    `;
    
    const { rows: issues } = await pg.query(issuesQuery);

    // RAG 검색 최적화를 위한 설정
    const MAX_CHUNK_SIZE = 3000; // 더 큰 청크 사이즈로 조정
    const MAX_CONTENT_LENGTH = 1500; // 더 충분한 내용 길이
    
    // 텍스트 요약 함수
    const truncateText = (text: string, maxLength: number): string => {
      if (text.length <= maxLength) return text;
      return text.substring(0, maxLength - 3) + '...';
    };

    // 지식베이스 메인 헤더
    let markdownContent = "# 고객지원 티켓 지식베이스\n\n";
    markdownContent += `마지막 업데이트: ${format(new Date(), 'yyyy-MM-dd HH:mm:ss')}\n`;
    markdownContent += `총 티켓 수: ${issues.length}개\n\n`;
    markdownContent += "이 문서는 고객지원 티켓의 문제해결 과정과 솔루션을 정리한 지식베이스입니다.\n\n";
    
    issues.forEach((issue, index) => {
      // 의미있는 티켓 제목으로 헤더 구성
      let ticketContent = `## ${issue.title}\n\n`;
      
      // 티켓 메타 정보
      ticketContent += `**티켓 ID:** ${issue.id} | **상태:** ${issue.status === 'open' ? '대기중' : '해결완료'} | **작성자:** ${issue.author_name}\n`;
      ticketContent += `**등록일:** ${format(new Date(issue.created_at), 'yyyy-MM-dd')}\n\n`;
      
      // 문제 상황 (고객 문의)
      ticketContent += `### 문제 상황\n\n`;
      ticketContent += `${issue.question}\n\n`;

      // 해결 과정 및 답변
      if (issue.replies && issue.replies.length > 0) {
        // 관리자/지원팀 답변 우선 추출
        const adminReplies = issue.replies.filter((reply: any) => 
          reply.authorRole === 'admin' || reply.authorRole === 'support'
        );
        
        // 일반 사용자 답변 추출
        const userReplies = issue.replies.filter((reply: any) => 
          reply.authorRole !== 'admin' && reply.authorRole !== 'support'
        );
        
        if (adminReplies.length > 0) {
          ticketContent += `### 해결방법\n\n`;
          
          // 모든 관리자 답변 포함 (시간순)
          adminReplies.forEach((reply: any, idx: number) => {
            if (idx > 0) ticketContent += `\n**추가 답변 ${idx + 1}:**\n\n`;
            ticketContent += `${reply.content}\n\n`;
          });
          
          // 사용자 추가 문의가 있으면 포함
          if (userReplies.length > 0) {
            ticketContent += `### 추가 문의 및 피드백\n\n`;
            userReplies.forEach((reply: any, idx: number) => {
              ticketContent += `**${reply.authorName}:** ${reply.content}\n\n`;
            });
          }
          
          ticketContent += `**최종 상태:** 해결완료`;
        } else {
          // 관리자 답변이 없는 경우 - 사용자 답변만 있음
          ticketContent += `### 진행 상황\n\n`;
          issue.replies.forEach((reply: any, idx: number) => {
            ticketContent += `**${reply.authorName}:** ${reply.content}\n\n`;
          });
          ticketContent += `**최종 상태:** 답변 대기중`;
        }
      } else {
        ticketContent += `### 진행 상황\n\n`;
        ticketContent += `아직 답변이 등록되지 않은 티켓입니다.\n\n`;
        ticketContent += `**최종 상태:** 미답변`;
      }
      
      // 검색 키워드 추가 (RAG 최적화)
      ticketContent += `\n---\n\n`;
      const keywords = [];
      keywords.push(`티켓${issue.id}`);
      keywords.push(issue.status === 'open' ? '미해결' : '해결완료');
      if (issue.replies && issue.replies.length > 0) {
        keywords.push('답변완료');
      } else {
        keywords.push('미답변');
      }
      ticketContent += `**태그:** ${keywords.join(' · ')}\n`;
      
      // 청크 크기가 너무 큰 경우 간략화
      if (ticketContent.length > MAX_CHUNK_SIZE) {
        // 간략한 버전으로 재구성
        ticketContent = `## ${truncateText(issue.title, 80)}\n\n`;
        ticketContent += `**ID:** ${issue.id} | **상태:** ${issue.status === 'open' ? '대기중' : '해결완료'} | **작성자:** ${issue.author_name}\n`;
        ticketContent += `**등록일:** ${format(new Date(issue.created_at), 'yyyy-MM-dd')}\n\n`;
        ticketContent += `### 문제 요약\n\n${truncateText(issue.question, 500)}\n\n`;
        
        if (issue.replies && issue.replies.length > 0) {
          const adminReplies = issue.replies.filter((reply: any) => 
            reply.authorRole === 'admin' || reply.authorRole === 'support'
          );
          if (adminReplies.length > 0) {
            ticketContent += `### 해결책\n\n${truncateText(adminReplies[0].content, 500)}\n\n`;
            ticketContent += `**상태:** 해결완료\n`;
          } else {
            ticketContent += `**상태:** 답변 대기중\n`;
          }
        } else {
          ticketContent += `**상태:** 미답변\n`;
        }
        
        ticketContent += `\n**태그:** 티켓${issue.id} · ${issue.status === 'open' ? '미해결' : '해결완료'}\n`;
      }
      
      markdownContent += ticketContent;
      
      // 티켓 간 구분선 추가
      if (index < issues.length - 1) {
        markdownContent += "\n\n\n";
      }
    });

    // 마크다운 텍스트 직접 반환
    return new Response(markdownContent, {
      headers: {
        'Content-Type': 'text/plain; charset=utf-8',
      },
    });

  } catch (error) {
    console.error('Failed to export tickets for RAG:', error);
    return NextResponse.json(
      { success: false, message: 'Failed to export tickets.' },
      { status: 500 }
    );
  }
} 