// FlowAI 타입 정의

export type NodeType = 'input' | 'process' | 'output' | 'condition' | 'loop';

export type ProcessType = 
  | 'chat'           // /api/v1/chat
  | 'code'           // /api/v1/code  
  | 'report'         // /api/v1/report
  | 'rag-search'     // /api/v1/rag/search
  | 'rag-add'        // /api/v1/rag/add
  | 'rag-list'       // /api/v1/rag/list
  | 'rag-init'       // /api/v1/rag/init
  | 'web-search'     // /api/v1/web/search
  | 'data-transform' // For data processing
  | 'condition'      // For conditional logic
  | 'loop';          // For iteration
export type InputType = 
  | 'text'
  | 'file'
  | 'database'
  | 'api'
  | 'web-search'
  | 'rag-query';

// API endpoint configurations
export interface ChatConfig {
  model?: string;
  temperature?: number;
  maxTokens?: number;
  systemPrompt?: string;
}

export interface CodeConfig {
  model?: string;
  language?: string;
  framework?: string;
  style?: string;
  systemPrompt?: string;
}

export interface ReportConfig {
  template?: 'bizplan' | 'proposal' | 'thesis' | 'simple' | 'marketing' | 'startup';
  format?: 'pdf' | 'docx' | 'pptx' | 'hwpx';
  includeCharts?: boolean;
  pageSize?: 'A4' | 'Letter';
}

export interface RAGConfig {
  query?: string;
  similarity_threshold?: number;
  max_results?: number;
  collection?: string;
}

export interface WebSearchConfig {
  query?: string;
  engine?: 'auto' | 'google' | 'naver' | 'daum';
  maxResults?: number;
}

export type OutputFormat = 
  | 'text'
  | 'pdf'
  | 'docx'
  | 'pptx'
  | 'hwpx'
  | 'markdown'
  | 'html'
  | 'json'
  | 'csv';

export interface Position {
  x: number;
  y: number;
}

export interface NodeData {
  id: string;
  type: NodeType;
  position: Position;
  data: {
    label: string;
    description?: string;
    config?: any;
    // Input 관련
    inputType?: InputType;
    placeholder?: string;
    allowedTypes?: string[];
    // Process 관련
    processType?: ProcessType;
    // Endpoint-specific configurations
    chatConfig?: ChatConfig;
    codeConfig?: CodeConfig;
    reportConfig?: ReportConfig;
    ragConfig?: RAGConfig;
    webSearchConfig?: WebSearchConfig;
    // Common fields
    prompt?: string;
    // 조건 및 반복
    condition?: string;
    operator?: string;
    loopCount?: number;
    loopVariable?: string;
    // API 관련
    apiEndpoint?: string;
    headers?: Record<string, string>;
    body?: any;
    // 실행 상태
    isRunning?: boolean;
    hasError?: boolean;
    // 레거시 플래그 (하위 호환성)
    ragEnabled?: boolean;
    webSearchEnabled?: boolean;
  };
  inputs?: string[];
  outputs?: string[];
}

export interface Connection {
  id: string;
  source: string;
  target: string;
  sourceHandle?: string;
  targetHandle?: string;
}

export interface WorkflowData {
  id: string;
  name: string;
  description?: string;
  nodes: NodeData[];
  connections: Connection[];
  variables?: Record<string, any>;
  createdAt: string;
  updatedAt: string;
  userId?: string;
  isPublic?: boolean;
  tags?: string[];
}

export interface ExecutionResult {
  nodeId: string;
  status: 'pending' | 'running' | 'success' | 'error' | 'skipped';
  output?: any;
  error?: string;
  startTime?: string;
  endTime?: string;
  duration?: number;
}

export interface WorkflowExecution {
  id: string;
  workflowId: string;
  status: 'pending' | 'running' | 'completed' | 'failed' | 'cancelled';
  results: ExecutionResult[];
  startTime: string;
  endTime?: string;
  totalDuration?: number;
  error?: string;
  finalOutput?: any;
}

export interface NodeTemplate {
  type: NodeType;
  label: string;
  description: string;
  icon: string;
  defaultConfig?: any;
  category: string;
}

export interface FlowAIContextType {
  workflow: WorkflowData | null;
  setWorkflow: (workflow: WorkflowData | null) => void;
  selectedNode: NodeData | null;
  setSelectedNode: (node: NodeData | null) => void;
  isExecuting: boolean;
  executionResults: ExecutionResult[];
  executeWorkflow: () => Promise<void>;
  saveWorkflow: () => Promise<void>;
  loadWorkflow: (id: string) => Promise<void>;
}