/* global process */
/* eslint-disable no-unused-vars, no-control-regex */
import { setContinousNetworkTryCount, getContinousNetworkTryCount, aiChat, geminiChat, anthropicChat, groqChat, openaiChat, ollamaChat, turnOnOllamaAndGetModelList, combindMessageHistory, code_generator, getModelName, getContextWindowSize, resultTemplate, axiosPostWrap, ask_prompt_text } from './aiFeatures.js'
import { makePreprocessingCode, shell_exec, execInVenv, attatchWatcher, execAdv } from './codeExecution.js'
import { isCorrectCode, code_validator, makeVEnvCmd } from './codeModifiers.js'
import { createVENV, disableAllVariable, disableVariable, getRCPath, readRCData, getVarVal, findMissingVars, isKeyInConfig, setVarVal } from './configuration.js'
import { threeticks, threespaces, disableOra, limitline, annn, responseTokenRatio, preprocessing, traceError, contextWindows, colors, forignLanguage, greetings, howAreYou, whatAreYouDoing, langtable } from './constants.js'
import { installProcess, realworld_which_python, which, getPythonVenvPath, getActivatePath, getPythonPipPath, venvCandidatePath, checkPythonForTermination } from './envLoaders.js'
import { oraSucceed, oraFail, oraStop, oraStart, oraBackupAndStopCurrent, print, strout } from './oraManager.js'
import promptTemplate from './translationPromptTemplate.js';
import chalk from 'chalk';
import { highlight } from 'cli-highlight';
import axios from 'axios';
import readline from 'readline';
import path from 'path';
import fs from 'fs';
import ora from 'ora';
import boxen from 'boxen';
import readlineSync from 'readline-sync';
import figlet from 'figlet';
import { fileURLToPath } from 'url';
import { Command } from 'commander';
import { promises as fsPromises } from 'fs';
import os from 'os';
import inquirer from 'inquirer';
import singleton from './singleton.js';
import stripAnsi from 'strip-ansi';
import { compareVersions as compareVersionsPackage } from 'compare-versions';

export async function currentLatestVersion() {
    try {
        const response = await axios.get('https://repo.invesume.com/airun/package.json', {
            headers: {
                'Accept': 'application/json',
                'Cache-Control': 'no-cache'
            },
            timeout: 10000
        });
        
        if (response.data && response.data.version) {
            return response.data.version.trim().replace(/^v/, '');
        }
        throw new Error('Invalid response format');
    } catch (error) {
        console.error('Failed to fetch latest version:', error.message);
        return null;
    }
}

export async function downloadUpdate(version) {
    const updateDir = path.join(os.homedir(), '.airun', 'updates');
    const backupDir = path.join(os.homedir(), '.airun', 'backups');
    const filename = `airun-v${version}-linux-x64.tar.gz`;
    const downloadUrl = `https://repo.invesume.com/airun/${filename}`;
    const downloadPath = path.join(updateDir, filename);
    
    try {
        // 업데이트 디렉토리 생성
        await fs.promises.mkdir(updateDir, { recursive: true });
        await fs.promises.mkdir(backupDir, { recursive: true });
        
        // 이전 다운로드 파일 정리
        if (await is_file(downloadPath)) {
            await fs.promises.unlink(downloadPath);
        }
        
        console.log(`Downloading from ${downloadUrl}...`);
        
        // 파일 다운로드
        const response = await axios({
            method: 'get',
            url: downloadUrl,
            responseType: 'stream',
            timeout: 30000
        });

        // Content-Length 확인
        const expectedSize = parseInt(response.headers['content-length'], 10);
        if (!expectedSize) {
            throw new Error('Invalid download: Content-Length header missing');
        }

        const writer = fs.createWriteStream(downloadPath);
        response.data.pipe(writer);

        await new Promise((resolve, reject) => {
            writer.on('finish', resolve);
            writer.on('error', reject);
        });

        // 다운로드된 파일 검증
        const stats = await fs.promises.stat(downloadPath);
        if (stats.size === 0) {
            throw new Error('Downloaded file is empty');
        }
        if (expectedSize && stats.size !== expectedSize) {
            throw new Error(`File size mismatch. Expected: ${expectedSize}, Got: ${stats.size}`);
        }

        // tar 파일 형식 검증
        const verifyResult = await execAdv(`tar -tzf "${downloadPath}"`);
        if (verifyResult.code !== 0) {
            throw new Error('Downloaded file is not a valid tar archive');
        }

        // 필수 파일 존재 확인
        const requiredFiles = ['index.js', 'package.json'];
        const fileList = verifyResult.stdout.split('\n');
        for (const required of requiredFiles) {
            if (!fileList.some(f => f.endsWith(required))) {
                throw new Error(`Required file missing in update package: ${required}`);
            }
        }

        console.log('Update package downloaded and verified successfully');
        return downloadPath;
    } catch (error) {
        // 다운로드 실패 시 임시 파일 정리
        try {
            if (await is_file(downloadPath)) {
                await fs.promises.unlink(downloadPath);
            }
        } catch (cleanupError) {
            console.warn('Failed to clean up incomplete download:', cleanupError);
        }
        
        if (error.response) {
            if (error.response.status === 404) {
                throw new Error(`Update package not found for version ${version}`);
            }
            throw new Error(`Download failed with status ${error.response.status}`);
        }
        throw new Error(`Update download failed: ${error.message}`);
    }
}

export async function installUpdate(version) {
    const updateDir = path.join(os.homedir(), '.airun', 'updates');
    const backupDir = path.join(os.homedir(), '.airun', 'backups');
    const filename = `airun-v${version}-linux-x64.tar.gz`;
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
    let projectDir = null;
    let hasBackup = false;
    let backupPath = null;
    let tempDir = null;
    let updateTempDir = null;
    const updatePath = path.join(updateDir, filename);
    let chromeSandboxPath = null;
    let sandboxExists = false;
    let sandboxPermissions = null;
    
    try {
        // 업데이트 파일 존재 확인
        if (!await is_file(updatePath)) {
            throw new Error('Update package not found. Please try downloading again.');
        }

        // 실제 프로젝트 경로 찾기
        const result = await execAdv('readlink -f $(which airun)');
        if (!result || result.code !== 0 || !result.stdout) {
            throw new Error('Failed to locate airun installation directory');
        }
        projectDir = path.dirname(result.stdout.trim());
        console.log(`Project directory: ${projectDir}`);
        
        // 디렉토리 존재 및 쓰기 권한 확인
        try {
            await fs.promises.access(projectDir, fs.constants.W_OK);
        } catch (error) {
            throw new Error(`No write permission for ${projectDir}. Please run with sudo or check permissions.`);
        }
        
        // 업데이트/백업 디렉토리 생성
        await fs.promises.mkdir(updateDir, { recursive: true });
        await fs.promises.mkdir(backupDir, { recursive: true });
        
        // 백업 디렉토리 쓰기 권한 확인
        try {
            await fs.promises.access(backupDir, fs.constants.W_OK);
        } catch (error) {
            throw new Error(`No write permission for backup directory ${backupDir}. Please check permissions.`);
        }
        
        // 기존 백업 파일 정리 (새 백업 생성 전)
        console.log('Cleaning up old backup files...');
        try {
            const existingBackups = await fs.promises.readdir(backupDir);
            const airunBackups = existingBackups.filter(file => 
                file.startsWith('airun-backup-') && file.endsWith('.tar.gz')
            );
            
            if (airunBackups.length > 0) {
                console.log(`Found ${airunBackups.length} existing backup file(s) to remove`);
                
                for (const backupFile of airunBackups) {
                    const oldBackupPath = path.join(backupDir, backupFile);
                    try {
                        await fs.promises.unlink(oldBackupPath);
                        console.log(`✓ Removed old backup: ${backupFile}`);
                    } catch (deleteError) {
                        console.warn(`⚠️  Could not remove old backup ${backupFile}: ${deleteError.message}`);
                    }
                }
            } else {
                console.log('No old backup files found');
            }
        } catch (readError) {
            console.warn(`⚠️  Could not read backup directory for cleanup: ${readError.message}`);
        }
        
                 // 환경 설정 파일들 백업 (업데이트 전)
         let envBackupPaths = [];
         const envPatterns = [
             '.env',
             '.env.local',
             '.env.production',
             '.env.production.local',
             '.env.development',
             '.env.development.local',
             '.env.test',
             '.env.test.local',
             'workspaces/web/.env',
             'workspaces/web/.env.local',
             'workspaces/web/.env.production',
             'workspaces/web/.env.production.local',
             'workspaces/web/.env.development',
             'workspaces/web/.env.development.local',
             'workspaces/web/.env.test',
             'workspaces/web/.env.test.local',
             'langfuse/web/.env',
             'langfuse/web/.env.local',
             'langfuse/web/.env.production',
             'langfuse/web/.env.production.local',
             'langfuse/web/.env.development',
             'langfuse/web/.env.development.local',
             'harmonize-client/.env',
             'harmonize-client/.env.local',
             'harmonize-client/.env.production',
             'harmonize-client/.env.production.local'
         ];
        
        console.log('Backing up environment configuration files...');
        for (const envPattern of envPatterns) {
            const envFilePath = path.join(projectDir, envPattern);
            if (await is_file(envFilePath)) {
                const backupFileName = envPattern.replace(/\//g, '_') + `.backup.${timestamp}`;
                const envBackupPath = path.join(backupDir, backupFileName);
                try {
                    // 백업 디렉토리 생성 (필요한 경우)
                    await fs.promises.mkdir(path.dirname(envBackupPath), { recursive: true });
                    await fs.promises.copyFile(envFilePath, envBackupPath);
                    envBackupPaths.push({
                        original: envFilePath,
                        backup: envBackupPath,
                        relative: envPattern
                    });
                    console.log(`✅ 환경 설정 파일 백업 완료: ${envPattern}`);
                } catch (error) {
                    console.warn(`⚠️ 환경 설정 파일 백업 실패 (${envPattern}): ${error.message}`);
                }
            }
        }

        // 현재 버전 백업
        backupPath = path.join(backupDir, `airun-backup-${timestamp}.tar.gz`);
        const tempBackupPath = `${backupPath}.tmp`;
        console.log(`Creating backup at: ${backupPath}`);
        
        // 백업 생성 전 디렉토리 확인
        console.log('Creating backup with exclusions...');
        
        // 백업 제외 디렉토리 목록 정의
        const excludeDirs = [
            'node_modules',
            '.git',
            'uploads',
            'cache',
            'logs',
            'test_output',
            'output',
            '.npm',
            '.cache',
            'tmp',
            'temp',
            'workspaces'
            // workspaces는 제외하되, 환경 설정 파일들은 별도로 백업
        ];
        
        // tar 명령어 구성 (상대 경로 사용)
        const excludeArgs = excludeDirs.map(dir => `--exclude='./${dir}'`).join(' ');
        const tarCommand = `tar ${excludeArgs} -czf "${tempBackupPath}" -C "${projectDir}" .`;
        
        console.log('Executing backup command:', tarCommand);
        const backupResult = await execAdv(tarCommand);
        
        if (backupResult.code !== 0) {
            console.error('Backup creation failed:', backupResult.stderr);
            console.error('Backup command was:', tarCommand);
            
            // 실패한 임시 백업 파일 정리
            try {
                if (await is_file(tempBackupPath)) {
                    await fs.promises.unlink(tempBackupPath);
                }
            } catch (cleanupError) {
                console.warn('Failed to clean up failed backup:', cleanupError);
            }
            throw new Error(`Failed to create backup: ${backupResult.stderr || 'Unknown tar error'}`);
        }
        console.log('Backup file created successfully');
        
        // 임시 백업 파일 기본 검증
        if (!await is_file(tempBackupPath)) {
            throw new Error('Temporary backup file was not created');
        }
        
        // 임시 백업 파일 크기 확인
        const tempStats = await fs.promises.stat(tempBackupPath);
        console.log(`Temporary backup file size: ${tempStats.size} bytes`);
        
        if (tempStats.size === 0) {
            await fs.promises.unlink(tempBackupPath);
            throw new Error('Temporary backup file is empty');
        }
        
        // 최소 크기 확인 (1KB 이상)
        if (tempStats.size < 1024) {
            console.warn(`Warning: Backup file is very small (${tempStats.size} bytes)`);
        }
        
        // 임시 백업 파일을 최종 백업 파일로 이동
        await fs.promises.rename(tempBackupPath, backupPath);
        console.log('Backup file moved to final location');
        
        // 백업 파일 존재 및 크기 확인
        try {
            const backupStats = await fs.promises.stat(backupPath);
            console.log(`Final backup file size: ${backupStats.size} bytes`);
            
            if (backupStats.size === 0) {
                throw new Error('Backup file is empty');
            }
            
            // 백업 파일 검증 (tar 파일 리스트 확인) - 타임아웃 설정
            console.log('Verifying backup file integrity...');
            const verifyBackup = await execAdv(`timeout 30 tar -tzf "${backupPath}" | head -20`);
            
            if (verifyBackup.code !== 0) {
                console.error('Backup verification failed. Exit code:', verifyBackup.code);
                console.error('Backup verification stderr:', verifyBackup.stderr);
                
                // 백업 파일이 손상되었을 수 있으므로 더 간단한 검증 시도
                console.log('Attempting basic file validation...');
                const basicCheck = await execAdv(`file "${backupPath}"`);
                console.log('File type check result:', basicCheck.stdout);
                
                if (basicCheck.stdout && basicCheck.stdout.includes('gzip compressed')) {
                    console.log('Backup file appears to be a valid gzip file, proceeding with caution...');
                } else {
                    throw new Error(`Backup file verification failed: ${verifyBackup.stderr || 'Invalid tar file'}`);
                }
            } else {
                console.log('Backup file verification successful');
                
                if (!verifyBackup.stdout || verifyBackup.stdout.trim() === '') {
                    console.warn('Warning: Backup file verification returned empty output');
                    
                    // 대체 검증 방법 시도
                    const altCheck = await execAdv(`tar -tf "${backupPath}" | wc -l`);
                    if (altCheck.code === 0 && parseInt(altCheck.stdout.trim()) > 0) {
                        console.log(`Backup contains ${altCheck.stdout.trim()} files`);
                    } else {
                        throw new Error('Backup file appears to be empty or corrupted');
                    }
                } else {
                    // 백업 파일 내용 간단 확인
                    const backupFiles = verifyBackup.stdout.split('\n').filter(f => f.trim());
                    console.log(`Backup contains ${backupFiles.length} files (showing first 20)`);
                    
                    // 필수 파일 존재 확인 (더 유연하게)
                    const requiredFiles = ['index.js', 'package.json', 'commons.js'];
                    const missingFiles = requiredFiles.filter(file => 
                        !backupFiles.some(f => f.includes(file))
                    );
                    
                    if (missingFiles.length > 0) {
                        console.warn('Warning: Some required files might be missing in backup:', missingFiles);
                        console.log('First 10 files in backup:', backupFiles.slice(0, 10));
                        
                        // 치명적이지 않은 경고로 처리 (백업 자체는 유효할 수 있음)
                        console.log('Proceeding with backup despite missing file warnings...');
                    }
                }
            }
            
            hasBackup = true;
            console.log('Backup created and verified successfully');
        } catch (error) {
            console.error('Backup verification error details:', error);
            
            // 백업 파일 크기 재확인
            try {
                const finalStats = await fs.promises.stat(backupPath);
                console.log(`Backup file final size: ${finalStats.size} bytes`);
                
                if (finalStats.size > 1024) {
                    console.log('Backup file has reasonable size, keeping it despite verification issues');
                    hasBackup = true;
                } else {
                    throw new Error('Backup file is too small to be valid');
                }
            } catch (statError) {
                console.error('Failed to check backup file stats:', statError);
                
                // 백업 실패 시 정리
                try {
                    if (await is_file(backupPath)) {
                        await fs.promises.unlink(backupPath);
                    }
                    if (await is_file(tempBackupPath)) {
                        await fs.promises.unlink(tempBackupPath);
                    }
                } catch (cleanupError) {
                    console.warn('Failed to clean up corrupted backup:', cleanupError);
                }
                
                // 백업 검증 실패 시에도 업데이트 진행 (경고만 출력)
                console.warn(`⚠️  Backup verification failed: ${error.message}`);
                console.warn('⚠️  Proceeding with update without verified backup...');
                console.warn('⚠️  Manual backup is recommended before continuing.');
                hasBackup = false; // 백업 실패 상태로 설정하지만 업데이트는 계속
            }
        }
        
        // 기존 npm link 상태 확인
        console.log('Checking existing npm link...');
        const npmLinkResult = await execAdv('npm ls -g airun');
        const hasExistingLink = npmLinkResult.code === 0;
        
        // 임시 디렉토리 생성
        tempDir = path.join(updateDir, `temp_${timestamp}`);
        updateTempDir = path.join(updateDir, `update_${timestamp}`);
        await fs.promises.mkdir(tempDir, { recursive: true });
        await fs.promises.mkdir(updateTempDir, { recursive: true });
        
        // 현재 설치본 임시 디렉토리로 백업
        console.log('Creating working copy...');
        const workingCopyResult = await execAdv(`cp -r "${projectDir}/"* "${tempDir}/"`);
        if (workingCopyResult.code !== 0) {
            throw new Error('Failed to create working copy');
        }
        
        // 새 버전 압축 해제
        const updateFile = path.join(updateDir, filename);
        console.log(`Extracting update to temporary directory...`);
        const extractResult = await execAdv(`tar -xzf "${updateFile}" -C "${updateTempDir}"`);
        if (extractResult.code !== 0) {
            throw new Error(`Failed to extract update file: ${extractResult.stderr}`);
        }
        
        // 업데이트 파일 검증
        try {
            // 필수 파일 목록
            const requiredFiles = ['index.js', 'package.json', 'commons.js'];
            const missingFiles = [];
            
            // 각 필수 파일 존재 여부 확인
            for (const file of requiredFiles) {
                const filePath = path.join(updateTempDir, 'airun', file);
                try {
                    await fs.promises.access(filePath, fs.constants.F_OK);
                } catch (error) {
                    missingFiles.push(file);
                }
            }
            
            if (missingFiles.length > 0) {
                throw new Error(`Required files missing in update package: ${missingFiles.join(', ')}`);
            }
            
            // package.json 유효성 검증
            const packageJsonPath = path.join(updateTempDir, 'airun', 'package.json');
            try {
                const packageContent = await fs.promises.readFile(packageJsonPath, 'utf8');
                const packageJson = JSON.parse(packageContent);
                
                // 필수 필드 확인
                const requiredFields = ['name', 'version', 'main'];
                const missingFields = requiredFields.filter(field => !packageJson[field]);
                
                if (missingFields.length > 0) {
                    throw new Error(`package.json is missing required fields: ${missingFields.join(', ')}`);
                }
                
                // 버전 확인
                if (packageJson.version !== version) {
                    throw new Error(`Version mismatch in package.json: expected ${version}, got ${packageJson.version}`);
                }
            } catch (error) {
                if (error instanceof SyntaxError) {
                    throw new Error('package.json is not a valid JSON file');
                }
                throw error;
            }
            
            console.log('Update package validated successfully');
        } catch (error) {
            throw new Error(`Update package validation failed: ${error.message}`);
        }
        
        try {
            // node_modules와 .git 디렉토리 보존
            const preservedDirs = {
                'node_modules': false,
                '.git': false
            };
            
            // electron/dist 디렉토리 경로
            const electronDistPath = path.join(projectDir, 'node_modules/electron/dist');
            let electronDistExists = false;
            let electronDistBackupPath = null;
            
            // electron/dist 디렉토리가 존재하는지 확인
            try {
                if (await is_dir(electronDistPath)) {
                    electronDistExists = true;
                    electronDistBackupPath = path.join(updateTempDir, 'electron_dist_backup');
                    
                    // electron/dist 디렉토리를 별도로 백업
                    console.log('Preserving electron/dist directory with its permissions...');
                    await fs.promises.mkdir(electronDistBackupPath, { recursive: true });
                    const backupDistResult = await execAdv(`cp -rp "${electronDistPath}" "${electronDistBackupPath}/"`);
                    
                    if (backupDistResult.code !== 0) {
                        console.warn('Warning: Failed to backup electron/dist directory. Chrome sandbox permissions might be lost.');
                    } else {
                        console.log('Successfully backed up electron/dist directory');
                    }
                }
            } catch (error) {
                console.log('No electron/dist directory found or error backing it up:', error.message);
            }
            
            for (const [dir, preserved] of Object.entries(preservedDirs)) {
                if (await is_dir(path.join(projectDir, dir))) {
                    const preserveResult = await execAdv(`mv "${projectDir}/${dir}" "${updateTempDir}/"`);
                    preservedDirs[dir] = preserveResult.code === 0;
                }
            }
            
            // 기존 파일 삭제 (보존된 디렉토리 제외)
            const spinner = ora('🗑️  Removing old files...').start();
            const currentFiles = await fs.promises.readdir(projectDir);
            for (const file of currentFiles) {
                if (!Object.keys(preservedDirs).includes(file)) {
                    const filePath = path.join(projectDir, file);
                    
                    try {
                        // 먼저 Node.js API로 시도
                        await fs.promises.rm(filePath, { recursive: true, force: true });
                    } catch (error) {
                        // Node.js API 실패 시 시스템 명령어로 강제 삭제 시도
                        console.log(`⚠️  Node.js rm failed for ${file}, trying system command...`);
                        
                        try {
                            // 읽기 전용 속성 제거 후 강제 삭제
                            await execAdv(`chmod -R +w "${filePath}" 2>/dev/null || true`);
                            const rmResult = await execAdv(`rm -rf "${filePath}"`);
                            
                            if (rmResult.code !== 0) {
                                // 여전히 실패하면 개별 파일 삭제 시도
                                console.log(`⚠️  System rm also failed for ${file}, trying individual file removal...`);
                                
                                if (await is_dir(filePath)) {
                                    // 디렉토리인 경우 내용을 먼저 삭제
                                    await execAdv(`find "${filePath}" -type f -delete 2>/dev/null || true`);
                                    await execAdv(`find "${filePath}" -type d -empty -delete 2>/dev/null || true`);
                                    await execAdv(`rmdir "${filePath}" 2>/dev/null || rm -rf "${filePath}"`);
                                } else {
                                    // 파일인 경우 강제 삭제
                                    await execAdv(`rm -f "${filePath}"`);
                                }
                            }
                        } catch (systemError) {
                            console.warn(`⚠️  Could not remove ${file}: ${systemError.message}`);
                            // 삭제 실패해도 계속 진행
                        }
                    }
                }
            }
            spinner.succeed('🗑️  Old files removed successfully');
            
            // 새 파일 복사
            spinner.start('📦 Installing new version...');
            // airun 디렉토리에서 파일 복사
            const copyResult = await execAdv(`cp -r "${updateTempDir}/airun/"* "${projectDir}/"`);
            if (copyResult.code !== 0) {
                spinner.fail('❌ Failed to copy new files');
                throw new Error('Failed to copy new files');
            }
            spinner.succeed('📦 New files installed successfully');
            
            // node_modules 디렉토리 복원
            if (preservedDirs['node_modules']) {
                spinner.start('📦 Restoring node_modules directory...');
                const restoreModulesResult = await execAdv(`mv "${updateTempDir}/node_modules" "${projectDir}/"`);
                if (restoreModulesResult.code !== 0) {
                    spinner.fail('❌ Failed to restore node_modules directory');
                    throw new Error('Failed to restore node_modules directory');
                }
                spinner.succeed('📦 node_modules directory restored successfully');
                
                // electron/dist 디렉토리 복원 (권한 유지를 위해)
                if (electronDistExists && electronDistBackupPath) {
                    spinner.start('🔒 Restoring electron/dist directory with original permissions...');
                    
                    // 기존 electron/dist 디렉토리 제거 (새로 설치된 것)
                    const newElectronDistPath = path.join(projectDir, 'node_modules/electron/dist');
                    if (await is_dir(newElectronDistPath)) {
                        await fs.promises.rm(newElectronDistPath, { recursive: true, force: true });
                    }
                    
                    // 백업한 electron/dist 디렉토리 복원 (권한 유지)
                    const restoreDistResult = await execAdv(`cp -rp "${electronDistBackupPath}/dist" "${path.join(projectDir, 'node_modules/electron/')}"`);
                    
                    if (restoreDistResult.code !== 0) {
                        spinner.warn('⚠️ Failed to restore electron/dist directory with original permissions');
                        console.log('You may need to manually set permissions for chrome-sandbox:');
                        console.log('sudo chown root:root "/path/to/chrome-sandbox"');
                        console.log('sudo chmod 4755 "/path/to/chrome-sandbox"');
                    } else {
                        spinner.succeed('🔒 electron/dist directory restored with original permissions');
                    }
                }
            }
            
            // package.json 버전 업데이트
            spinner.start('📝 Updating package.json version...');
            const packageJsonPath = path.join(projectDir, 'package.json');
            try {
                // package.json이 존재하는지 확인
                if (!await is_file(packageJsonPath)) {
                    spinner.fail('❌ package.json not found after copying files');
                    throw new Error('package.json not found after copying files');
                }
                
                const packageJson = JSON.parse(await fs.promises.readFile(packageJsonPath, 'utf8'));
                packageJson.version = version;
                await fs.promises.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2));
                spinner.succeed('📝 package.json version updated successfully');
            } catch (error) {
                spinner.fail(`❌ Failed to update package.json: ${error.message}`);
                
                // 파일 시스템 상태 확인
                try {
                    const files = await fs.promises.readdir(projectDir);
                    console.error('Current directory contents:', files);
                } catch (fsError) {
                    console.error('Failed to read directory:', fsError);
                }
                
                throw new Error(`Failed to update package.json: ${error.message}`);
            }
            
            // npm install 실행
            spinner.start('📥 Installing dependencies...');
            const installResult = await execAdv(`cd "${projectDir}" && npm install`);
            if (installResult.code !== 0) {
                spinner.fail('❌ Failed to install dependencies');
                throw new Error('Failed to install dependencies');
            }
            spinner.succeed('📥 Dependencies installed successfully');

            // electron 로컬 설치
            spinner.start('⚛️  Installing electron locally...');
            const electronLocalResult = await execAdv(`cd "${projectDir}" && npm install electron`);
            if (electronLocalResult.code !== 0) {
                spinner.fail('❌ Failed to install electron locally');
                throw new Error('Failed to install electron locally');
            }
            
            // chrome-sandbox 권한 설정
            spinner.start('🔒 Configuring Electron sandbox...');
            
            // sudo 명령 대신 --no-sandbox 옵션을 사용하도록 package.json 수정
            try {
                const packageJsonPath = path.join(projectDir, 'package.json');
                if (await is_file(packageJsonPath)) {
                    const packageJson = JSON.parse(await fs.promises.readFile(packageJsonPath, 'utf8'));
                    
                    // scripts 섹션이 없으면 생성
                    if (!packageJson.scripts) {
                        packageJson.scripts = {};
                    }
                    
                    // 기존 start 스크립트가 있으면 --no-sandbox 추가
                    if (packageJson.scripts.start) {
                        if (!packageJson.scripts.start.includes('--no-sandbox')) {
                            packageJson.scripts.start = packageJson.scripts.start.replace('electron .', 'electron --no-sandbox .');
                        }
                    } else {
                        packageJson.scripts.start = 'electron --no-sandbox .';
                    }
                    
                    // 기존 electron 관련 스크립트에 --no-sandbox 추가
                    Object.keys(packageJson.scripts).forEach(key => {
                        if (key.includes('electron') && !packageJson.scripts[key].includes('--no-sandbox')) {
                            packageJson.scripts[key] = packageJson.scripts[key].replace('electron', 'electron --no-sandbox');
                        }
                    });
                    
                    // package.json 저장
                    await fs.promises.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2), 'utf8');
                    spinner.succeed('📄 Added --no-sandbox option to package.json scripts');
                }
                
                // main.js 파일이 있으면 app.commandLine.appendSwitch 추가
                const mainJsPath = path.join(projectDir, 'main.js');
                if (await is_file(mainJsPath)) {
                    let mainJs = await fs.promises.readFile(mainJsPath, 'utf8');
                    
                    // app.commandLine.appendSwitch가 없으면 추가
                    if (!mainJs.includes('app.commandLine.appendSwitch(\'--no-sandbox\')')) {
                        // app 모듈 import 확인
                        if (!mainJs.includes('const app = ')) {
                            if (mainJs.includes('const { app') || mainJs.includes('const {app')) {
                                // 이미 구조분해할당으로 app을 가져오고 있음
                            } else if (mainJs.includes('const electron = require(\'electron\')')) {
                                // electron에서 app 참조 추가
                                mainJs = mainJs.replace('const electron = require(\'electron\')', 
                                    'const electron = require(\'electron\')\nconst app = electron.app');
                            } else {
                                // electron 모듈 import 추가
                                mainJs = 'const { app } = require(\'electron\')\n' + mainJs;
                            }
                        }
                        
                        // app.ready 이벤트 전에 no-sandbox 스위치 추가
                        const insertPosition = mainJs.indexOf('app.whenReady()') > -1 ? 
                            mainJs.indexOf('app.whenReady()') : 
                            (mainJs.indexOf('app.on(\'ready\'') > -1 ? 
                                mainJs.indexOf('app.on(\'ready\'') : 0);
                        
                        if (insertPosition > 0) {
                            mainJs = mainJs.slice(0, insertPosition) + 
                                'app.commandLine.appendSwitch(\'--no-sandbox\')\n\n' + 
                                mainJs.slice(insertPosition);
                        } else {
                            // 파일 시작 부분에 추가
                            mainJs = 'app.commandLine.appendSwitch(\'--no-sandbox\')\n\n' + mainJs;
                        }
                        
                        await fs.promises.writeFile(mainJsPath, mainJs, 'utf8');
                        spinner.succeed('📄 Added --no-sandbox option to main.js');
                    }
                }
                
                spinner.succeed('🔒 Configured Electron to run without sandbox (no sudo required)');
            } catch (error) {
                spinner.warn('⚠️  Warning: Could not configure --no-sandbox option');
                console.log('You may need to manually add --no-sandbox option when running the application');
                console.log('For example: electron --no-sandbox .');
                console.log('Or add to main.js: app.commandLine.appendSwitch(\'--no-sandbox\')');
            }
            
            spinner.succeed('⚛️  Electron installed successfully');

            // index.js에 실행 권한 추가
            spinner.start('🔐 Setting execute permission...');
            const indexChmodResult = await execAdv(`chmod +x "${projectDir}/index.js"`);
            if (indexChmodResult.code !== 0) {
                spinner.fail('❌ Failed to set execute permission on index.js');
                throw new Error('Failed to set execute permission on index.js');
            }
            spinner.succeed('🔐 Execute permission set successfully');
            
            // npm link 재실행
            spinner.start('🔗 Updating npm link...');
            const linkResult = await execAdv(`cd "${projectDir}" && npm link`);
            if (linkResult.code !== 0) {
                spinner.fail('❌ Failed to create global command');
                throw new Error('Failed to create global command');
            }
            spinner.succeed('🔗 Global command updated successfully');
            
            // 환경 설정 파일들 복원 (업데이트 후)
            if (envBackupPaths.length > 0) {
                spinner.start('🔄 Restoring environment configuration files...');
                let restoredCount = 0;
                let failedCount = 0;
                
                for (const envBackup of envBackupPaths) {
                    try {
                        // 복원할 디렉토리가 존재하지 않으면 생성
                        const targetDir = path.dirname(envBackup.original);
                        await fs.promises.mkdir(targetDir, { recursive: true });
                        
                        await fs.promises.copyFile(envBackup.backup, envBackup.original);
                        console.log(`✅ 환경 설정 파일 복원 완료: ${envBackup.relative}`);
                        restoredCount++;
                    } catch (error) {
                        console.warn(`⚠️ 환경 설정 파일 복원 실패 (${envBackup.relative}): ${error.message}`);
                        console.log(`💡 수동으로 복원하려면: cp "${envBackup.backup}" "${envBackup.original}"`);
                        failedCount++;
                    }
                }
                
                if (restoredCount > 0) {
                    spinner.succeed(`✅ 환경 설정 파일 복원 완료 (${restoredCount}개 성공, ${failedCount}개 실패)`);
                } else {
                    spinner.fail(`❌ 환경 설정 파일 복원 실패 (${failedCount}개 실패)`);
                }
                
                // 백업 파일 정리 (최대 5개 유지)
                try {
                    const backupFiles = await fs.promises.readdir(backupDir);
                    const envBackups = backupFiles
                        .filter(file => file.includes('.env') && file.includes('.backup.'))
                        .sort()
                        .reverse(); // 최신 순으로 정렬
                    
                    if (envBackups.length > 20) { // 여러 환경 파일이 있으므로 더 많이 유지
                        const filesToDelete = envBackups.slice(20);
                        for (const file of filesToDelete) {
                            try {
                                await fs.promises.unlink(path.join(backupDir, file));
                                console.log(`🗑️ 오래된 환경 설정 백업 파일 삭제: ${file}`);
                            } catch (deleteError) {
                                console.warn(`⚠️ 백업 파일 삭제 실패: ${deleteError.message}`);
                            }
                        }
                    }
                } catch (cleanupError) {
                    console.warn(`⚠️ 환경 설정 백업 파일 정리 실패: ${cleanupError.message}`);
                }
            } else {
                console.log('ℹ️ 복원할 환경 설정 파일이 없습니다.');
            }
            
            console.log(chalk.green('\n✨ Update completed successfully! ✨'));
            return true;
        } catch (error) {
            // 설치 실패 시 복원
            console.error('Installation failed:', error.message);
            console.log('Restoring previous version...');
            
            // 복원 전 현재 상태 백업 (디버깅용)
            const failedStatePath = path.join(backupDir, `airun-failed-${timestamp}.tar.gz`);
            try {
                await execAdv(`tar -czf "${failedStatePath}" -C "${projectDir}" .`);
            } catch (error) {
                console.warn('Failed to save failed state:', error);
            }
            
            try {
                // 현재 파일 모두 제거 (안전하게)
                const currentFiles = await fs.promises.readdir(projectDir);
                for (const file of currentFiles) {
                    const filePath = path.join(projectDir, file);
                    await fs.promises.rm(filePath, { recursive: true, force: true });
                }
                
                // 백업에서 복원 시도 (임시 복사본에서 먼저 시도)
                if (tempDir && await is_dir(tempDir)) {
                    console.log('Restoring from working copy...');
                    const restoreResult = await execAdv(`cp -r "${tempDir}/"* "${projectDir}/"`);
                    if (restoreResult.code === 0) {
                        console.log('Successfully restored from working copy');
                    } else {
                        throw new Error('Failed to restore from working copy');
                    }
                } else if (hasBackup && backupPath) {
                    // 임시 복사본 복원 실패 시 백업 파일에서 복원
                    console.log('Restoring from backup file...');
                    const restoreResult = await execAdv(`tar -xzf "${backupPath}" -C "${projectDir}"`);
                    if (restoreResult.code !== 0) {
                        throw new Error('Failed to restore from backup file');
                    }
                } else {
                    throw new Error('No valid backup available for restoration');
                }
                
                // npm install 및 link 복원
                console.log('Restoring dependencies...');
                await execAdv(`cd "${projectDir}" && npm install`);
                await execAdv(`cd "${projectDir}" && npm link`);
                
                // 환경 설정 파일들 복원 (업데이트 실패 시)
                if (envBackupPaths.length > 0) {
                    console.log('Restoring environment configuration files...');
                    let restoredCount = 0;
                    
                    for (const envBackup of envBackupPaths) {
                        try {
                            // 복원할 디렉토리가 존재하지 않으면 생성
                            const targetDir = path.dirname(envBackup.original);
                            await fs.promises.mkdir(targetDir, { recursive: true });
                            
                            await fs.promises.copyFile(envBackup.backup, envBackup.original);
                            console.log(`✅ 환경 설정 파일 복원 완료: ${envBackup.relative}`);
                            restoredCount++;
                        } catch (error) {
                            console.warn(`⚠️ 환경 설정 파일 복원 실패 (${envBackup.relative}): ${error.message}`);
                            console.log(`💡 수동으로 복원하려면: cp "${envBackup.backup}" "${envBackup.original}"`);
                        }
                    }
                    
                    if (restoredCount > 0) {
                        console.log(`✅ ${restoredCount}개의 환경 설정 파일이 복원되었습니다.`);
                    }
                }
                
                console.log(chalk.yellow('⚠️  Update failed, but previous version has been successfully restored'));
                console.log(chalk.blue('ℹ️  You can try updating again after resolving the issue'));
                
                // 복원 성공 시 에러 던지지 않고 false 반환
                return false;
            } catch (restoreError) {
                console.error('Critical error: Both update and restore failed');
                console.error('Update error:', error.message);
                console.error('Restore error:', restoreError.message);
                throw new Error(`Critical error: Both update and restore failed - Update: ${error.message}, Restore: ${restoreError.message}`);
            }
        }
    } catch (error) {
        console.error('Update process failed:', error);
        throw error;
    } finally {
        // 임시 파일 정리
        try {
            const cleanupPaths = [tempDir, updateTempDir];
            for (const dir of cleanupPaths) {
                if (dir && await is_dir(dir)) {
                    await fs.promises.rm(dir, { recursive: true, force: true });
                }
            }
            
            // 업데이트 파일 정리
            const updateFile = path.join(updateDir, filename);
            if (await is_file(updateFile)) {
                await fs.promises.unlink(updateFile);
            }
        } catch (error) {
            console.warn('Warning: Failed to clean up temporary files:', error);
        }
    }
}

export async function checkAndUpdate() {
    try {
        const currentVersion = process.env.npm_package_version;
        const latestVersion = await currentLatestVersion();
        
        if (!latestVersion) {
            throw new Error('Cannot get the latest version information.');
        }
        
        // 버전 비교
        if (latestVersion === currentVersion) {
            return { updated: false, message: 'Already the latest version.' };
        }
        
        // 다운로드
        await downloadUpdate(latestVersion);
        
        // 설치
        await installUpdate(latestVersion);
        
        return {
            updated: true,
            message: `Updated to version ${latestVersion}. Please restart the program.`
        };
    } catch (error) {
        return {
            updated: false,
            error: true,
            message: `Update failed: ${error.message}`
        };
    }
}

export async function reqRenderer(mode, arg) {
    if (!isElectron()) return;
    if (!singleton?.reqsAPI) return;
    return await singleton?.reqsAPI(mode, arg);
}
export async function outNotifier(message, leave = false) {
    if (leave) await letNotifier();
    if (!isElectron()) return;
    return await reqRenderer('outnotify', message)
}
export async function disNotifier(message) {
    if (!isElectron()) return;
    return await reqRenderer('disnotify', message)
}
export async function letNotifier(message) {
    if (!isElectron()) return;
    return await reqRenderer('letnotify', message)
}
export async function errNotifier(message, leave = false) {
    if (leave) await letNotifier();
    if (!isElectron()) return;
    return await reqRenderer('errnotify', message)
    /*
        ask_prompt_text
        openEndedPrompt
        promptChoices
        multipleChoicePrompt
        inquirer.prompt    
    */
}
export function isElectron() {
    return !!(process?.versions?.electron);
}
export async function promptChoices(choices, message) {
    await errNotifier('Option selection input request error occurred');
    const indexedChoices = choices.map((choice, index) => `${index + 1} - ${choice}`);
    const answers = await inquirer.prompt([
        {
            name: 'choice',
            type: 'list',
            message,
            choices: indexedChoices,
            filter: (value) => {
                const numberPattern = /\d+/;
                const match = value.match(numberPattern);
                return choices[parseInt(match[0], 10) - 1];
            }
        }
    ]);
    return choices.indexOf(answers.choice);
}
export function printError(e) {
    if (!traceError) return;
    console.log(e);
}
export function isBadStr(ppath) {
    if (ppath.indexOf(`"`) > -1) return !false;
    if (ppath.indexOf(`'`) > -1) return !false;
    return !true;
}
export function addslashes(str) { return str.replace(/[\\"]/g, '\\$&').replace(/\u0000/g, '\\0'); }
export function getCurrentDateTime() {
    const now = new Date();
    const year = now.getFullYear();
    const month = String(now.getMonth() + 1).padStart(2, '0');
    const day = String(now.getDate()).padStart(2, '0');
    const hours = String(now.getHours()).padStart(2, '0');
    const minutes = String(now.getMinutes()).padStart(2, '0');
    const seconds = String(now.getSeconds()).padStart(2, '0');
    const milliseconds = String(now.getMilliseconds()).padStart(3, '0');
    return `${year}${month}${day}_${hours}${minutes}${seconds}_${milliseconds}`;
}
export async function is_dir(path) {
    try {
        const stat = await fsPromises.stat(path);
        return stat.isDirectory();
    } catch (error) {
        printError(error);
        return false;
    }
}
export async function is_file(path) {
    try {
        const stat = await fsPromises.stat(path);
        return stat.isFile();
    } catch (error) {
        printError(error);
        return false;
    }
}
export async function isItem(itemPath) {
    try {
        await fsPromises.access(itemPath, fsPromises.constants.F_OK);
        return true;
    } catch (e) {
        printError(e);
        return false;
    }
}

export function splitStringIntoTokens(inputString) {
    return inputString.split(/(\w+|\S)/g).filter(token => token.trim() !== '');
}

export function measureColumns(min = 2) {
    const terminallWidth = process.stdout.columns;
    return terminallWidth - min;
}

export function isWindows() { return process.platform === 'win32'; }
export function isLinux() { return process.platform === 'linux'; }
export function isMacOS() { return process.platform === 'darwin'; }

export function compareVersions(v1, v2) {
    return compareVersionsPackage.compare(v1, v2);
}