/**
 * 컨텍스트 추출기 - 대화에서 중요한 정보를 추출하는 모듈
 * Ollama/Hamonize 모델과 호환되는 프롬프트 기반 추출 방식 사용
 */

import { logger } from './utils/logger.js';

export class ContextExtractor {
    constructor() {
        this.ollamaClient = null; // 나중에 주입받을 예정
        this.extractionCache = new Map(); // 중복 추출 방지 캐시
    }

    /**
     * Ollama 클라이언트 설정
     */
    setOllamaClient(client) {
        this.ollamaClient = client;
    }

    /**
     * 대화에서 컨텍스트 추출 (메인 함수)
     */
    async extractContext(conversation, userId, sessionId) {
        try {
            if (!this.ollamaClient) {
                logger.warn('[ContextExtractor] Ollama 클라이언트가 설정되지 않음 - 기본 추출 사용');
                return this.extractBasicContext(conversation);
            }

            // 캐시 확인
            const cacheKey = this.generateCacheKey(conversation);
            if (this.extractionCache.has(cacheKey)) {
                logger.debug('[ContextExtractor] 캐시된 추출 결과 사용');
                return this.extractionCache.get(cacheKey);
            }

            // AI 기반 컨텍스트 추출
            const extracted = await this.extractWithAI(conversation, userId, sessionId);
            
            // 캐시 저장 (최대 100개 항목 유지)
            if (this.extractionCache.size >= 100) {
                const firstKey = this.extractionCache.keys().next().value;
                this.extractionCache.delete(firstKey);
            }
            this.extractionCache.set(cacheKey, extracted);

            return extracted;

        } catch (error) {
            logger.error('[ContextExtractor] 컨텍스트 추출 실패:', error);
            // 실패시 기본 추출 방식 사용
            return this.extractBasicContext(conversation);
        }
    }

    /**
     * AI 기반 컨텍스트 추출 (Hamonize 모델 사용)
     */
    async extractWithAI(conversation, userId, sessionId) {
        const extractionPrompt = this.buildExtractionPrompt(conversation);

        try {
            const response = await this.ollamaClient.chat({
                model: 'hamonize:latest',
                messages: [
                    {
                        role: 'system',
                        content: `당신은 대화에서 중요한 정보를 추출하는 전문가입니다. 사용자의 대화를 분석하여 다음 정보들을 JSON 형식으로 정확히 추출해주세요.`
                    },
                    {
                        role: 'user',
                        content: extractionPrompt
                    }
                ],
                format: 'json',
                options: {
                    temperature: 0.1,  // 일관된 추출을 위해 낮은 temperature
                    top_p: 0.9
                }
            });

            const extracted = JSON.parse(response.message.content);
            
            // 추출 결과 검증 및 정규화
            const normalized = this.normalizeExtractedData(extracted);
            
            logger.debug('[ContextExtractor] AI 추출 완료:', {
                entities: normalized.entities.length,
                facts: normalized.facts.length,
                preferences: Object.keys(normalized.preferences).length,
                sessionId
            });

            return normalized;

        } catch (error) {
            logger.error('[ContextExtractor] AI 추출 실패:', error);
            // AI 추출 실패시 기본 방식으로 폴백
            return this.extractBasicContext(conversation);
        }
    }

    /**
     * 추출 프롬프트 생성
     */
    buildExtractionPrompt(conversation) {
        // 대화를 문자열로 변환
        let conversationText = '';
        if (typeof conversation === 'string') {
            conversationText = conversation;
        } else if (Array.isArray(conversation)) {
            conversationText = conversation
                .filter(msg => msg && msg.content)
                .map(msg => `${msg.role}: ${msg.content}`)
                .join('\n');
        } else {
            conversationText = JSON.stringify(conversation);
        }

        return `
다음 대화에서 중요한 정보를 추출해주세요:

대화 내용:
"""
${conversationText}
"""

다음 형식으로 JSON 응답해주세요:
{
  "entities": [
    {
      "name": "엔티티명",
      "type": "person|technology|project|company|tool|concept",
      "context": "언급된 맥락"
    }
  ],
  "facts": [
    {
      "content": "중요한 사실 내용",
      "importance": 0.8,
      "category": "기술|업무|개인|프로젝트"
    }
  ],
  "preferences": {
    "technology": ["React", "Node.js"],
    "framework": ["Next.js"],
    "database": ["PostgreSQL"],
    "tools": ["VS Code"],
    "programming_style": ["함수형", "타입스크립트"]
  },
  "working_context": "현재 진행중인 주제나 작업",
  "topics": ["주요 주제1", "주요 주제2"],
  "sentiment": "positive|neutral|negative",
  "complexity_level": "beginner|intermediate|advanced"
}

중요한 규칙:
1. 엔티티는 사람, 기술, 도구, 회사명 등 명사형태로 추출
2. 사실은 향후 참고할 가치가 있는 정보만 추출
3. 선호도는 사용자가 좋아하거나 선호하는 것들 추출
4. working_context는 현재 대화의 주된 목적이나 진행상황
5. 확실하지 않은 정보는 제외
`;
    }

    /**
     * 기본 컨텍스트 추출 (AI 없이 규칙 기반)
     */
    extractBasicContext(conversation) {
        logger.debug('[ContextExtractor] 기본 추출 방식 사용');
        
        let text = '';
        if (typeof conversation === 'string') {
            text = conversation;
        } else if (Array.isArray(conversation)) {
            text = conversation
                .filter(msg => msg && msg.content)
                .map(msg => msg.content)
                .join(' ');
        }

        const extracted = {
            entities: this.extractEntitiesBasic(text),
            facts: this.extractFactsBasic(text),
            preferences: this.extractPreferencesBasic(text),
            working_context: this.extractWorkingContextBasic(text),
            topics: this.extractTopicsBasic(text),
            sentiment: this.extractSentimentBasic(text),
            complexity_level: this.extractComplexityBasic(text)
        };

        return extracted;
    }

    /**
     * 기본 엔티티 추출 (규칙 기반)
     */
    extractEntitiesBasic(text) {
        const entities = [];
        
        // 기술 스택 패턴
        const techPatterns = [
            /React/gi, /Vue/gi, /Angular/gi, /Node\.js/gi, /Express/gi,
            /PostgreSQL/gi, /MySQL/gi, /MongoDB/gi, /Redis/gi,
            /JavaScript/gi, /TypeScript/gi, /Python/gi, /Java/gi,
            /Docker/gi, /Kubernetes/gi, /AWS/gi, /Azure/gi
        ];

        techPatterns.forEach(pattern => {
            const matches = text.match(pattern);
            if (matches) {
                matches.forEach(match => {
                    entities.push({
                        name: match,
                        type: 'technology',
                        context: 'mentioned'
                    });
                });
            }
        });

        return entities;
    }

    /**
     * 기본 사실 추출
     */
    extractFactsBasic(text) {
        const facts = [];
        
        // 중요한 키워드가 포함된 문장 추출
        const importantKeywords = ['문제', '오류', '해결', '구현', '개발', '설치', '설정'];
        const sentences = text.split(/[.!?]/).filter(s => s.trim().length > 10);
        
        sentences.forEach(sentence => {
            const hasImportantKeyword = importantKeywords.some(keyword => 
                sentence.includes(keyword)
            );
            
            if (hasImportantKeyword && sentence.length < 200) {
                facts.push({
                    content: sentence.trim(),
                    importance: 0.5,
                    category: '기술'
                });
            }
        });

        return facts.slice(0, 5); // 최대 5개만
    }

    /**
     * 기본 선호도 추출
     */
    extractPreferencesBasic(text) {
        const preferences = {};
        
        // 선호도 표현 패턴
        const preferencePatterns = [
            { pattern: /좋아해|선호|사용해|쓰고있어/, category: 'positive' },
            { pattern: /싫어|안좋아|사용안해/, category: 'negative' }
        ];

        // 기술별 카테고리
        const techCategories = {
            'React': 'framework',
            'Vue': 'framework',
            'Angular': 'framework',
            'PostgreSQL': 'database',
            'MySQL': 'database',
            'MongoDB': 'database'
        };

        return preferences;
    }

    /**
     * 기본 작업 컨텍스트 추출
     */
    extractWorkingContextBasic(text) {
        const contextKeywords = [
            '개발중', '구현중', '작업중', '진행중', '만들고있어', '하고있어'
        ];

        for (const keyword of contextKeywords) {
            if (text.includes(keyword)) {
                // 해당 키워드 주변 텍스트 추출
                const index = text.indexOf(keyword);
                const start = Math.max(0, index - 50);
                const end = Math.min(text.length, index + 100);
                return text.substring(start, end).trim();
            }
        }

        return '';
    }

    /**
     * 기본 주제 추출
     */
    extractTopicsBasic(text) {
        const topics = [];
        const topicKeywords = [
            '웹개발', '프론트엔드', '백엔드', '데이터베이스', '서버', '배포',
            '디자인', 'API', '보안', '테스트', '성능', '최적화'
        ];

        topicKeywords.forEach(keyword => {
            if (text.includes(keyword)) {
                topics.push(keyword);
            }
        });

        return topics;
    }

    /**
     * 기본 감정 분석
     */
    extractSentimentBasic(text) {
        const positiveWords = ['좋다', '잘', '성공', '완료', '해결', '감사'];
        const negativeWords = ['문제', '오류', '실패', '어려워', '안돼', '못'];

        const positiveCount = positiveWords.filter(word => text.includes(word)).length;
        const negativeCount = negativeWords.filter(word => text.includes(word)).length;

        if (positiveCount > negativeCount) return 'positive';
        if (negativeCount > positiveCount) return 'negative';
        return 'neutral';
    }

    /**
     * 기본 복잡도 분석
     */
    extractComplexityBasic(text) {
        const beginnerKeywords = ['기초', '처음', '시작', '간단'];
        const advancedKeywords = ['고급', '복잡', '심화', '최적화', '아키텍처'];

        const hasBeginnerKeywords = beginnerKeywords.some(keyword => text.includes(keyword));
        const hasAdvancedKeywords = advancedKeywords.some(keyword => text.includes(keyword));

        if (hasAdvancedKeywords) return 'advanced';
        if (hasBeginnerKeywords) return 'beginner';
        return 'intermediate';
    }

    /**
     * 추출 결과 정규화
     */
    normalizeExtractedData(extracted) {
        return {
            entities: Array.isArray(extracted.entities) ? extracted.entities : [],
            facts: Array.isArray(extracted.facts) ? extracted.facts : [],
            preferences: typeof extracted.preferences === 'object' ? extracted.preferences : {},
            working_context: extracted.working_context || '',
            topics: Array.isArray(extracted.topics) ? extracted.topics : [],
            sentiment: extracted.sentiment || 'neutral',
            complexity_level: extracted.complexity_level || 'intermediate',
            timestamp: new Date().toISOString()
        };
    }

    /**
     * 캐시 키 생성
     */
    generateCacheKey(conversation) {
        const text = typeof conversation === 'string' ? conversation : JSON.stringify(conversation);
        // 간단한 해시 생성 (실제로는 crypto 모듈 사용 권장)
        return text.length + '_' + text.substring(0, 50).replace(/\s/g, '');
    }

    /**
     * 캐시 클리어
     */
    clearCache() {
        this.extractionCache.clear();
        logger.debug('[ContextExtractor] 캐시 클리어됨');
    }
}

export default ContextExtractor;