﻿# airun 설치하는 PowerShell 스크립트

# 기존 설치 제거
function Remove-ExistingInstallation {
    Write-Output "기존 airun 설치 제거 중..."
    
    # npm 패키지 및 링크 제거
    npm uninstall -g airun 2>$null
    npm unlink -g airun 2>$null
    npm cache clean --force 2>$null
    
    # 전역 node_modules에서 제거
    $nodePath = npm root -g
    if (Test-Path "$nodePath\airun") {
        Remove-Item -Path "$nodePath\airun" -Recurse -Force
    }
    
    # 로컬 설치 파일 제거
    if (Test-Path "node_modules") {
        Remove-Item -Path "node_modules" -Recurse -Force
    }
    if (Test-Path "package-lock.json") {
        Remove-Item -Path "package-lock.json" -Force
    }
    
    # Python 가상환경 제거
    if (Test-Path "$env:USERPROFILE\.airun_venv") {
        Remove-Item -Path "$env:USERPROFILE\.airun_venv" -Recurse -Force
    }
    
    # 설정 및 캐시 디렉토리 제거
    if (Test-Path "$env:USERPROFILE\.airun") {
        Remove-Item -Path "$env:USERPROFILE\.airun" -Recurse -Force
    }
    
    Write-Output "기존 설치 제거 완료"
}

# 관리자 권한 체크
function Test-Administrator {
    $user = [Security.Principal.WindowsIdentity]::GetCurrent();
    $principal = New-Object Security.Principal.WindowsPrincipal $user
    return $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

# 관리자 권한으로 재실행
if (-not (Test-Administrator)) {
    Write-Output "이 스크립트는 관리자 권한이 필요합니다."
    Write-Output "PowerShell을 관리자 권한으로 실행한 후 다시 시도해주세요."
    Write-Output ""
    Write-Output "관리자 권한이 필요한 이유:"
    Write-Output "1. Chrome 및 ChromeDriver 시스템 레벨 설치"
    Write-Output "2. 시스템 환경 변수 설정"
    Write-Output "3. Visual C++ Build Tools 확인"
    Write-Output "4. Node.js 패키지 전역 설치"
    Write-Output ""
    Write-Output "PowerShell을 관리자 권한으로 여는 방법:"
    Write-Output "1. Windows 시작 메뉴에서 PowerShell을 검색"
    Write-Output "2. '관리자 권한으로 실행' 선택"
    exit 1
}

# PowerShell 인코딩 설정 확인 및 설정
function Test-Encoding {
    Write-Output "현재 PowerShell 인코딩 설정:"
    Write-Output "- OutputEncoding: $([Console]::OutputEncoding.WebName)"
    Write-Output "- InputEncoding: $([Console]::InputEncoding.WebName)"
    Write-Output "- FileSystem Encoding: $($(Get-ItemProperty 'HKLM:\SYSTEM\CurrentControlSet\Control\Nls\CodePage').ACP)"
    
    # 시스템 로캘이 UTF-8이 아닌 경우 안내 메시지 출력
    $codePage = $(Get-ItemProperty 'HKLM:\SYSTEM\CurrentControlSet\Control\Nls\CodePage').ACP
    if ($codePage -ne "65001") {
        Write-Output "[주의] 시스템 로캘이 UTF-8로 설정되어 있지 않습니다."
        Write-Output "한글이 깨져 보일 수 있으니 다음과 같이 설정해주세요:"
        Write-Output "1. Windows 설정 > 시간 및 언어 > 언어 및 지역 > 관리자 언어 설정"
        Write-Output "2. 시스템 로캘 변경"
        Write-Output "3. 'Beta: Unicode UTF-8을 사용하여 전세계 언어 지원' 옵션 체크"
        Write-Output "4. 시스템 재시작"
    }
    Write-Output ""
}

# Chrome 브라우저 설치 확인 및 설치
function Install-Chrome {
    Write-Output "Chrome 브라우저 설치 확인 중..."
    $chromePath = "C:\Program Files\Google\Chrome\Application\chrome.exe"
    $chromeExists = Test-Path $chromePath
    
    if (-not $chromeExists) {
        Write-Output "Chrome 브라우저가 설치되어 있지 않습니다. 설치를 시작합니다..."
        
        # Chrome 설치 파일 다운로드
        $installerPath = "$env:TEMP\chrome_installer.exe"
        $downloadUrl = "https://dl.google.com/chrome/install/latest/chrome_installer.exe"
        
        try {
            # 이전 설치 파일이 있다면 삭제
            if (Test-Path $installerPath) {
                Remove-Item $installerPath -Force
            }
            
            # 다운로드
            Write-Output "Chrome 설치 파일 다운로드 중..."
            (New-Object System.Net.WebClient).DownloadFile($downloadUrl, $installerPath)
            Write-Output "Chrome 설치 파일 다운로드 완료"
            
            # Chrome 설치 실행
            Write-Output "Chrome 설치 중..."
            $process = Start-Process -FilePath $installerPath -Args "/silent /install" -Wait -PassThru
            
            if ($process.ExitCode -eq 0) {
                Write-Output "Chrome 설치 완료"
                # 설치 확인을 위해 5초 대기
                Start-Sleep -Seconds 5
            } else {
                Write-Output "Chrome 설치 실패 (종료 코드: $($process.ExitCode))"
            }
            
            # 설치 파일 삭제
            if (Test-Path $installerPath) {
                Remove-Item $installerPath -Force
            }
            
            # 설치 확인
            if (Test-Path $chromePath) {
                Write-Output "Chrome이 성공적으로 설치되었습니다."
            } else {
                Write-Output "Chrome 설치를 확인할 수 없습니다."
                Write-Output "Chrome을 수동으로 설치해주세요: https://www.google.com/chrome/"
            }
        }
        catch {
            Write-Output "[경고] Chrome 설치 중 오류가 발생했습니다: $($_.Exception.Message)"
            Write-Output "Chrome을 수동으로 설치해주세요: https://www.google.com/chrome/"
        }
    } else {
        Write-Output "Chrome 브라우저가 이미 설치되어 있습니다."
    }
    Write-Output ""
}

# ChromeDriver 설치
function Install-ChromeDriver {
    Write-Output "ChromeDriver 설치 확인 중..."
    
    # ChromeDriver 설치 여부 확인
    $driverPath = "$env:ProgramFiles\ChromeDriver"
    if (Test-Path "$driverPath\chromedriver.exe") {
        Write-Output "ChromeDriver가 이미 설치되어 있습니다."
        Write-Output ""
        return $true
    }
    
    # Chrome 버전 확인
    $chrome = Get-Item 'C:\Program Files\Google\Chrome\Application\chrome.exe' -ErrorAction SilentlyContinue
    if (-not $chrome) {
        Write-Output "Chrome이 설치되어 있지 않습니다. Chrome을 먼저 설치해주세요."
        return $false
    }
    
    $chromeVersion = $chrome.VersionInfo.FileVersion.Split('.')[0]
    Write-Output "Chrome 버전: $chromeVersion"
    
    try {
        # ChromeDriver 다운로드 URL (새로운 URL 사용)
        $downloadUrl = "https://googlechromelabs.github.io/chrome-for-testing/LATEST_RELEASE_STABLE"
        $response = Invoke-WebRequest -Uri $downloadUrl
        $latestVersion = [System.Text.Encoding]::UTF8.GetString($response.Content).Trim()
        Write-Output "설치할 ChromeDriver 버전: $latestVersion"
        
        # 새로운 다운로드 URL 형식 사용
        $driverZip = "$env:TEMP\chromedriver_win32.zip"
        $newDownloadUrl = "https://edgedl.me.gvt1.com/edgedl/chrome/chrome-for-testing/$latestVersion/win64/chromedriver-win64.zip"
        
        Invoke-WebRequest -Uri $newDownloadUrl -OutFile $driverZip
        
        # 압축 해제
        if (-not (Test-Path $driverPath)) {
            New-Item -ItemType Directory -Path $driverPath -Force | Out-Null
        }
        
        # 새로운 구조에 맞게 압축 해제 경로 수정
        Expand-Archive -Path $driverZip -DestinationPath "$env:TEMP\chromedriver_temp" -Force
        Copy-Item "$env:TEMP\chromedriver_temp\chromedriver-win64\*" $driverPath -Force -Recurse
        
        # 환경 변수 설정
        $currentPath = [Environment]::GetEnvironmentVariable("PATH", "Machine")
        if (-not $currentPath.Contains($driverPath)) {
            [Environment]::SetEnvironmentVariable("PATH", "$currentPath;$driverPath", "Machine")
        }
        
        # 임시 파일 삭제
        Remove-Item $driverZip -Force
        Remove-Item "$env:TEMP\chromedriver_temp" -Recurse -Force
        
        Write-Output "ChromeDriver 설치 완료"
        return $true
    }
    catch {
        Write-Output "[경고] ChromeDriver 설치 중 오류가 발생했습니다: $($_.Exception.Message)"
        Write-Output "ChromeDriver를 수동으로 설치해주세요: https://chromedriver.chromium.org/downloads"
        return $false
    }
}

# Visual C++ Build Tools 체크
function Check-BuildTools {
    Write-Output "Visual C++ Build Tools 확인 중..."
    $vsWhere = "${env:ProgramFiles(x86)}\Microsoft Visual Studio\Installer\vswhere.exe"
    if (-not (Test-Path $vsWhere)) {
        Write-Output "[경고] Visual C++ Build Tools가 설치되어 있지 않습니다."
        Write-Output "다음 링크에서 'Microsoft C++ Build Tools'를 설치해주세요:"
        Write-Output "https://visualstudio.microsoft.com/visual-cpp-build-tools/"
        Write-Output "설치 시 'Desktop development with C++'를 선택해주세요."
        return $false
    }
    return $true
}

# Python 가상환경 설정
function Setup-VirtualEnvironment {
    Write-Output "`n=== Python 가상환경 설정 중... ==="
    $venvPath = "$env:USERPROFILE\.airun_venv"
    
    # 기존 가상환경 제거
    if (Test-Path $venvPath) {
        Write-Output "기존 가상환경 제거 중: $venvPath"
        Remove-Item -Path $venvPath -Recurse -Force
        Write-Output "기존 가상환경 제거 완료"
    }
    
    # 새 가상환경 생성
    Write-Output "새 가상환경 생성 중: $venvPath"
    $process = Start-Process -FilePath "python" -ArgumentList "-m venv `"$venvPath`"" -NoNewWindow -Wait -PassThru
    
    if ($process.ExitCode -ne 0) {
        Write-Output "[실패] 가상환경 생성 실패"
        return $false
    }
    Write-Output "가상환경 생성 완료"
    
    # pip 업그레이드
    Write-Output "`npip 업그레이드 중..."
    $activateScript = Join-Path $venvPath "Scripts\activate.bat"
    
    $process = Start-Process -FilePath "cmd.exe" -ArgumentList "/c `"call `"$activateScript`" && python -m pip install --upgrade pip`"" -NoNewWindow -Wait -PassThru
    
    if ($process.ExitCode -ne 0) {
        Write-Output "[실패] pip 업그레이드 실패"
        return $false
    }
    Write-Output "pip 업그레이드 완료"
    
    return $true
}

# 임시 파일 정리 함수
function Clean-TempFiles {
    Write-Output "`n=== 임시 파일 정리 중... ==="
    
    # 버전 번호 패턴과 일치하는 파일 찾기
    $versionPattern = '^\d+\.\d+\.\d+$|^\d+\.\d+$|^\d+\.\d+\.\d+\.\d+$'
    Get-ChildItem -Path $PSScriptRoot | Where-Object {
        $_.Name -match $versionPattern
    } | ForEach-Object {
        Write-Output "임시 파일 삭제 중: $($_.Name)"
        Remove-Item $_.FullName -Force
    }
    
    Write-Output "임시 파일 정리 완료"
    Write-Output ""
}

# RAG 관련 Python 패키지 설치
function Install-RagPackages {
    Write-Output "`n=== Python 패키지 설치 중... ==="
    $venvPath = "$env:USERPROFILE\.airun_venv"
    $activateScript = Join-Path $venvPath "Scripts\activate.bat"
    
    # 핵심 의존성 패키지들
    $packages = @(
        # "chromadb",  # Replaced with PostgreSQL
        "langchain",
        "langchain-community",
        "transformers",
        "sentence-transformers",
        "torch",
        "python-docx",
        "PyPDF2",
        "Pillow",
        "selenium",
        "webdriver_manager",
        "beautifulsoup4",
        "requests",
        "tqdm",
        "pandas",
        "langchain-huggingface",
        "pyhwp",
        "pandas",
        "openpyxl",
        "PyPDF2",
        "olefile",
        "python-docx",
        "python-pptx",
        "beautifulsoup4",
        "reportlab",
        "svglib",
        "cairosvg",
        "trafilatura",
        "lxml",
        "fpdf2",
        "matplotlib"
    )
    
    foreach ($package in $packages) {
        Write-Output "`n>>> 설치 중: $package"
        $process = Start-Process -FilePath "cmd.exe" -ArgumentList "/c `"call `"$activateScript`" && pip install --no-cache-dir $package`"" -NoNewWindow -Wait -PassThru
        
        if ($process.ExitCode -ne 0) {
            Write-Output "[실패] $package 설치 실패"
            return $false
        }
        Write-Output "[성공] $package 설치 완료"
    }
    
    return $true
}

# Node.js 패키지 설치
function Install-NodePackages {
    Write-Output "`n=== Node.js 패키지 설치 중... ==="
    
    # 기존 airun 패키지 제거
    Write-Output "`n>>> 기존 airun 패키지 제거 중..."
    try {
        $output = npm unlink airun -g 2>&1
        Write-Output $output
        Write-Output "[성공] airun unlink 완료"
    } catch {
        Write-Output "[정보] unlink 실패 (이미 제거됨)"
    }
    
    try {
        $output = npm uninstall airun -g 2>&1
        Write-Output $output
        Write-Output "[성공] airun uninstall 완료"
    } catch {
        Write-Output "[정보] uninstall 실패 (이미 제거됨)"
    }
    
    # npm 업데이트
    Write-Output "`n>>> npm 업데이트 중..."
    try {
        $output = npm install -g npm@latest 2>&1
        Write-Output $output
        if ($LASTEXITCODE -ne 0) {
            Write-Output "[실패] npm 업데이트 실패"
            return $false
        }
        Write-Output "[성공] npm 업데이트 완료"
    } catch {
        Write-Output "[실패] npm 업데이트 실패: $($_.Exception.Message)"
        return $false
    }
    
    # Mermaid CLI 설치
    Write-Output "`n>>> Mermaid CLI 설치 중..."
    try {
        # 기존 설치 확인
        $mmdcExists = $null
        try {
            $mmdcExists = Get-Command mmdc -ErrorAction SilentlyContinue
        } catch {}

        if ($mmdcExists) {
            Write-Output "Mermaid CLI가 이미 설치되어 있습니다."
            $version = mmdc --version 2>&1
            Write-Output "버전: $version"
        } else {
            $output = npm install -g @mermaid-js/mermaid-cli 2>&1
            Write-Output $output
            if ($LASTEXITCODE -ne 0) {
                Write-Output "[실패] Mermaid CLI 설치 실패"
                return $false
            }
            Write-Output "[성공] Mermaid CLI 설치 완료"
            $version = mmdc --version 2>&1
            Write-Output "설치된 버전: $version"
        }
    } catch {
        Write-Output "[실패] Mermaid CLI 설치 실패: $($_.Exception.Message)"
        return $false
    }
    
    # electron 설치
    Write-Output "`n>>> electron 설치 중..."
    try {
        $output = npm install -g electron 2>&1
        Write-Output $output
        if ($LASTEXITCODE -ne 0) {
            Write-Output "[실패] electron 설치 실패"
            return $false
        }
        Write-Output "[성공] electron 설치 완료"
    } catch {
        Write-Output "[실패] electron 설치 실패: $($_.Exception.Message)"
        return $false
    }
    
    # airun 패키지 링크
    Write-Output "`n>>> airun 패키지 링크 중..."
    Push-Location $PSScriptRoot
    try {
        $output = npm link 2>&1
        Write-Output $output
        if ($LASTEXITCODE -ne 0) {
            Write-Output "[실패] airun 패키지 링크 실패"
            Pop-Location
            return $false
        }
        Write-Output "[성공] airun 패키지 링크 완료"
    } catch {
        Write-Output "[실패] airun 패키지 링크 실패: $($_.Exception.Message)"
        Pop-Location
        return $false
    }
    Pop-Location
    
    return $true
}

# 바로가기 생성
function Create-Shortcut {
    Write-Output "바탕화면에 바로가기 생성 중..."
    $DesktopPath = [Environment]::GetFolderPath("Desktop")
    $ShortcutPath = Join-Path $DesktopPath "AI.RUN.lnk"
    
    try {
        $WScriptShell = New-Object -ComObject WScript.Shell
        $NpmBinPath = Join-Path $env:APPDATA "npm"
        $AirunPath = Join-Path $NpmBinPath "airun.cmd"
        
        $Shortcut = $WScriptShell.CreateShortcut($ShortcutPath)
        # 직접 airun.cmd를 대상으로 지정
        $Shortcut.TargetPath = $AirunPath
        $Shortcut.Arguments = "-g"
        $Shortcut.WindowStyle = 7  # 최소화
        $Shortcut.Description = "AI.RUN Graphical User Interface"
        
        $IconPath = Join-Path $PSScriptRoot "assets\airun.ico"
        if (Test-Path $IconPath) {
            $Shortcut.IconLocation = $IconPath
        }
        
        $Shortcut.Save()
        Write-Output "바로가기가 성공적으로 생성되었습니다."
    }
    catch {
        Write-Output "[경고] 바로가기 생성 중 오류가 발생했습니다: $($_.Exception.Message)"
    }
}

# 설치 확인 함수
function Test-Installation {
    Write-Output "`n=== 설치 상태 확인 중... ==="
    $venvPath = "$env:USERPROFILE\.airun_venv"
    $activateScript = Join-Path $venvPath "Scripts\activate.bat"
    $testsPassed = $true
    
    # 1. 가상환경 활성화 확인
    Write-Output "`n1. 가상환경 확인 중..."
    if (-not (Test-Path $activateScript)) {
        Write-Output "[실패] 가상환경이 설치되지 않았습니다."
        return $false
    }
    Write-Output "[성공] 가상환경 확인 완료"
    
    # 2. 필수 DLL 확인
    Write-Output "`n2. 시스템 DLL 확인 중..."
    $requiredDlls = @(
        "$env:SystemRoot\System32\sqlite3.dll",
        "$env:SystemRoot\System32\msvcp140.dll",
        "$env:SystemRoot\System32\vcruntime140.dll"
    )
    
    foreach ($dll in $requiredDlls) {
        if (-not (Test-Path $dll)) {
            Write-Output "[실패] $dll 이(가) 없습니다."
            $testsPassed = $false
        }
    }
    if ($testsPassed) {
        Write-Output "[성공] 필수 DLL 확인 완료"
    }
    
    # 3. 필수 패키지 설치 확인
    Write-Output "`n3. Python 패키지 확인 중..."
    $requiredPackages = @(
        # "chromadb",  # Replaced with PostgreSQL
        "langchain",
        "langchain_community",
        "transformers",
        "sentence_transformers",
        "torch"
    )
    
    $cmd = "call `"$activateScript`" && python -c `""
    foreach ($package in $requiredPackages) {
        $cmd += "import $package; "
    }
    $cmd += "print('모든 패키지 import 성공')`""
    
    $result = cmd /c $cmd 2>&1
    if ($LASTEXITCODE -ne 0) {
        Write-Output "[실패] 패키지 import 테스트 실패:"
        Write-Output $result
        $testsPassed = $false
    } else {
        Write-Output "[성공] 패키지 import 테스트 완료"
        
        # 버전 정보 출력
        Write-Output "`n=== 설치된 패키지 버전 정보 ==="
        $versionCmd = "call `"$activateScript`" && python -c `""
        # $versionCmd += "import chromadb; print('chromadb:', chromadb.__version__); "  # Replaced with PostgreSQL
        $versionCmd += "import langchain; print('langchain:', langchain.__version__); "
        $versionCmd += "import torch; print('torch:', torch.__version__)`""
        $versions = cmd /c $versionCmd 2>&1
        Write-Output $versions
    }
    
    # 4. RAG 기능 테스트 (PostgreSQL 기반)
    Write-Output "`n4. RAG 기능 테스트 중... (PostgreSQL 기반)"
    Write-Output "PostgreSQL 기반 RAG 시스템은 Python 백엔드에서 처리됩니다."
    Write-Output "[정보] RAG 기능 테스트는 PostgreSQL 서버 실행 후 별도로 수행됩니다."
    
    # 최종 결과
    Write-Output "`n=== 설치 확인 결과 ==="
    if ($testsPassed) {
        Write-Output "모든 테스트가 성공적으로 완료되었습니다."
        Write-Output "AI.RUN의 RAG 기능을 사용할 준비가 되었습니다."
    } else {
        Write-Output "일부 테스트가 실패했습니다. 위의 오류 메시지를 확인해주세요."
        Write-Output "문제 해결을 위한 제안:"
        Write-Output "1. Visual C++ Redistributable를 재설치해보세요"
        Write-Output "2. Python 가상환경을 재생성해보세요"
        Write-Output "3. 필요한 패키지들을 수동으로 재설치해보세요"
    }
    
    return $testsPassed
}

function Install-WindowsDependencies {
    Write-Output "`n=== Windows 시스템 의존성 설치 중... ==="
    
    # Visual C++ Redistributable 설치 확인
    $vcRedistRegKey = "HKLM:\SOFTWARE\Microsoft\VisualStudio\14.0\VC\Runtimes\x64"
    if (Test-Path $vcRedistRegKey) {
        Write-Output "Visual C++ Redistributable이 이미 설치되어 있습니다."
    } else {
        Write-Output "Visual C++ Redistributable 설치 중..."
        $vcRedistUrl = "https://aka.ms/vs/17/release/vc_redist.x64.exe"
        $vcRedistPath = "$env:TEMP\vc_redist.x64.exe"
        
        try {
            Write-Output "다운로드 중: $vcRedistUrl"
            Invoke-WebRequest -Uri $vcRedistUrl -OutFile $vcRedistPath
            Write-Output "설치 실행 중..."
            Start-Process -FilePath $vcRedistPath -Args "/quiet /norestart" -Wait
            Write-Output "임시 파일 정리 중..."
            Remove-Item $vcRedistPath -Force
        } catch {
            Write-Output "[경고] Visual C++ Redistributable 설치 실패: $($_.Exception.Message)"
        }
    }
    return $true
}

function Setup-ConfigDirectory {
    Write-Output "`n=== 환경 설정 디렉토리 생성 중... ==="
    $configPath = "$env:USERPROFILE\.airun"
    
    # 기존 설정 폴더 제거
    if (Test-Path $configPath) {
        Write-Output "기존 설정 폴더 제거 중: $configPath"
        Remove-Item -Path $configPath -Recurse -Force
        Write-Output "기존 설정 폴더 제거 완료"
    }
    
    # 새 설정 폴더 생성
    Write-Output "새 설정 폴더 생성 중: $configPath"
    New-Item -ItemType Directory -Path $configPath -Force | Out-Null
    Write-Output "설정 폴더 생성 완료"
    
    # 기본 설정 파일 생성
    Write-Output "기본 설정 파일 생성 중..."
    $defaultConfig = @{
        "model" = "gpt-3.5-turbo"
        "temperature" = 0.7
        "max_tokens" = 2000
    } | ConvertTo-Json
    
    $configFile = "$configPath\config.json"
    $defaultConfig | Out-File -FilePath $configFile -Encoding utf8
    Write-Output "설정 파일 생성 완료: $configFile"
    
    return $true
}

# Python 설치 확인
$pythonInstalled = $false
try {
    $pythonVersion = python --version 2>&1
    if ($pythonVersion -match "Python 3") {
        Write-Output "Python이 설치되어 있습니다: $pythonVersion"
        $pythonInstalled = $true
    }
} catch {
    Write-Output "Python이 설치되어 있지 않습니다."
}

# Node.js 설치 및 버전 확인
$nodeInstalled = $false
try {
    $nodeVersion = node --version 2>&1
    if ($nodeVersion -match "v22.18.0") {
        Write-Output "Node.js가 올바른 버전으로 설치되어 있습니다: $nodeVersion"
        $nodeInstalled = $true
    } else {
        Write-Output "Node.js 버전이 맞지 않습니다. 현재 버전: $nodeVersion"
        Write-Output "Node.js 22.18.0 버전을 설치해주세요: https://nodejs.org/"
        Write-Output "1. Node.js 22.18.0 LTS 버전을 다운로드"
        Write-Output "2. 기존 Node.js를 제거"
        Write-Output "3. 다운로드한 설치 파일을 실행"
        exit 1
    }
} catch {
    Write-Output "Node.js가 설치되어 있지 않습니다."
    Write-Output "Node.js 22.18.0 버전을 설치해주세요: https://nodejs.org/"
    exit 1
}

# npm 버전 확인 및 업데이트
try {
    $npmVersion = npm -v
    if ($npmVersion -ne "10.0.0") {
        Write-Output "npm 버전 업데이트 중..."
        npm install -g npm@10.0.0
        $npmVersion = npm -v
        if ($npmVersion -ne "10.0.0") {
            Write-Output "npm 버전 업데이트 실패. 현재 버전: $npmVersion"
            exit 1
        }
    }
    Write-Output "npm 버전: $npmVersion"
} catch {
    Write-Output "npm 버전 확인 중 오류 발생"
    exit 1
}

# 메인 설치 프로세스
if ($pythonInstalled -and $nodeInstalled) {
    # 기존 설치 제거
    Remove-ExistingInstallation
    
    # 인코딩 설정 확인
    Test-Encoding
    
    # Chrome 및 ChromeDriver 설치
    if (-not (Install-Chrome)) {
        Write-Output "[오류] Chrome 설치 실패"
        exit 1
    }
    
    if (-not (Install-ChromeDriver)) {
        Write-Output "[오류] ChromeDriver 설치 실패"
        exit 1
    }
    
    # 환경 설정 디렉토리 설정
    if (-not (Setup-ConfigDirectory)) {
        Write-Output "[오류] 환경 설정 디렉토리 생성 실패"
        exit 1
    }
    Write-Output "환경 설정 디렉토리 생성 완료"
    
    # Windows 시스템 의존성 설치
    Write-Output "`n시스템 의존성 설치 시작..."
    if (-not (Install-WindowsDependencies)) {
        Write-Output "[오류] Windows 시스템 의존성 설치 실패"
        exit 1
    }
    Write-Output "시스템 의존성 설치 완료"
    
    # Python 가상환경 설정
    Write-Output "`n가상환경 설정 시작..."
    if (-not (Setup-VirtualEnvironment)) {
        Write-Output "[오류] Python 가상환경 설정 실패"
        exit 1
    }
    Write-Output "가상환경 설정 완료"
    
    # Windows용 requirements 설치
    Write-Output "`nWindows용 Python 패키지 설치 시작..."
    $venvPath = "$env:USERPROFILE\.airun_venv"
    $activateScript = Join-Path $venvPath "Scripts\activate.bat"
    
    if (Test-Path "requirements-windows.txt") {
        $process = Start-Process -FilePath "cmd.exe" -ArgumentList "/c `"call `"$activateScript`" && pip install -r requirements-windows.txt`"" -NoNewWindow -Wait -PassThru
        if ($process.ExitCode -ne 0) {
            Write-Output "[오류] Windows용 Python 패키지 설치 실패"
            exit 1
        }
    } else {
        Write-Output "[경고] requirements-windows.txt 파일이 없습니다. 기본 패키지만 설치합니다."
    }
    
    # RAG 관련 패키지 설치
    Write-Output "`nPython 패키지 설치 시작..."
    if (-not (Install-RagPackages)) {
        Write-Output "[오류] Python 패키지 설치 실패"
        exit 1
    }
    Write-Output "Python 패키지 설치 완료"
    
    # Node.js 패키지 설치
    Write-Output "`nNode.js 패키지 설치 시작..."
    if (-not (Install-NodePackages)) {
        Write-Output "[오류] Node.js 패키지 설치 실패"
        exit 1
    }
    Write-Output "Node.js 패키지 설치 완료"
    
    # 설치 확인 테스트 실행
    Write-Output "`n설치 확인 테스트 시작..."
    if (-not (Test-Installation)) {
        Write-Output "[오류] 설치 확인 테스트 실패"
        exit 1
    }
    
    Write-Output "`n설치가 성공적으로 완료되었습니다!"
    
    # 바로가기 생성
    Create-Shortcut
    
    Write-Output "`n프로그램 실행 방법:"
    Write-Output "1. CLI 모드: airun ""your prompt"""
    Write-Output "2. GUI 모드: airun -g"
} else {
    Write-Output "필수 환경을 설치한 후 다시 실행해주세요."
    Write-Output "필요한 환경:"
    Write-Output "1. Python 3.10 이상: https://www.python.org/downloads/"
    Write-Output "2. Node.js 22.18.0: https://nodejs.org/"
    exit 1
}

function Install-PyhwpxPackage {
    Write-Output "`n=== pyhwpx 패키지 설치 중... ==="
    $venvPath = "$env:USERPROFILE\.airun_venv"
    $activateScript = Join-Path $venvPath "Scripts\activate.bat"
    
    if (Test-Path "pyhwpx") {
        # 이전 설치 제거
        $process = Start-Process -FilePath "cmd.exe" -ArgumentList "/c `"call `"$activateScript`" && pip uninstall -y pyhwpx`"" -NoNewWindow -Wait -PassThru
        
        # pyproject.toml을 통한 설치
        Push-Location "pyhwpx"
        try {
            if (Test-Path "pyproject.toml") {
                Write-Output "pyproject.toml을 통한 설치 시작..."
                $process = Start-Process -FilePath "cmd.exe" -ArgumentList "/c `"call `"$activateScript`" && pip install .`"" -NoNewWindow -Wait -PassThru
                
                if ($process.ExitCode -eq 0) {
                    Write-Output "[성공] pyhwpx 패키지가 성공적으로 설치되었습니다."
                } else {
                    Write-Output "[실패] pyhwpx 패키지 설치 실패"
                    return $false
                }
            } else {
                Write-Output "[실패] pyproject.toml을 찾을 수 없습니다."
                return $false
            }
        } finally {
            Pop-Location
        }
    } else {
        Write-Output "[경고] pyhwpx 디렉토리를 찾을 수 없습니다. 설치를 건너뜁니다..."
    }
    
    return $true
}

# 기존 설치 스크립트 내에 함수 호출 추가
if (-not (Install-PyhwpxPackage)) {
    Write-Output "[오류] pyhwpx 패키지 설치 실패"
    exit 1
}
