/**
 * 지능적 컨텍스트 활용 시스템 개선안
 * AI가 스스로 컨텍스트 필요성을 판단하도록 개선
 */

class IntelligentContextManager {
    constructor() {
        // 컨텍스트가 필요한 질문 패턴들
        this.contextTriggerPatterns = [
            // 기술 관련 질문
            /기술|프레임워크|언어|데이터베이스|IDE|도구/i,
            // 선호도 관련 질문
            /선호|좋아|사용.*하는|쓰는/i,
            // 프로젝트 관련 질문
            /프로젝트|개발|구축|만들|설계/i,
            // 추천 요청
            /추천|제안|방법|어떻게/i,
            // 현재 상황 질문
            /현재|지금|내.*스택|내.*환경/i
        ];

        // 컨텍스트가 불필요한 질문 패턴들
        this.nonContextPatterns = [
            /^안녕|^하이|^hello/i,
            /날씨|시간|^감사/i,
            /^좋은.*아침|^좋은.*저녁/i,
            /일반적인|보편적인|대부분/i
        ];
    }

    /**
     * 질문이 컨텍스트를 필요로 하는지 판단
     * @param {string} userPrompt - 사용자 질문
     * @returns {boolean} - 컨텍스트 필요 여부
     */
    shouldUseContext(userPrompt) {
        // 1단계: 명확히 불필요한 패턴 체크
        if (this.nonContextPatterns.some(pattern => pattern.test(userPrompt))) {
            return false;
        }

        // 2단계: 컨텍스트 필요 패턴 체크
        if (this.contextTriggerPatterns.some(pattern => pattern.test(userPrompt))) {
            return true;
        }

        // 3단계: AI 기반 관련성 판단 (향후 구현)
        return this.aiBasedRelevanceCheck(userPrompt);
    }

    /**
     * AI를 활용한 컨텍스트 관련성 판단
     * @param {string} userPrompt - 사용자 질문
     * @returns {boolean} - 관련성 점수 기반 판단
     */
    aiBasedRelevanceCheck(userPrompt) {
        // 기본적으로 기술적 키워드가 있으면 컨텍스트 사용
        const techKeywords = [
            'API', 'REST', 'DB', '서버', '클라이언트', '백엔드', '프론트엔드',
            '성능', '최적화', '보안', '배포', '테스트', '구조', '아키텍처'
        ];

        return techKeywords.some(keyword =>
            userPrompt.toLowerCase().includes(keyword.toLowerCase())
        );
    }

    /**
     * 관련 컨텍스트만 선택적으로 추출
     * @param {string} userPrompt - 사용자 질문
     * @param {object} allContext - 전체 컨텍스트 정보
     * @returns {object} - 필터링된 컨텍스트
     */
    selectRelevantContext(userPrompt, allContext) {
        const relevantContext = {};

        // 백엔드 관련 질문인 경우
        if (/백엔드|서버|API|데이터베이스/i.test(userPrompt)) {
            relevantContext.preferences = this.filterPreferences(
                allContext.preferences,
                ['언어', '프레임워크', '데이터베이스']
            );
            relevantContext.entities = this.filterEntities(
                allContext.entities,
                ['programming_language', 'framework', 'database']
            );
        }

        // 프론트엔드 관련 질문인 경우
        if (/프론트엔드|클라이언트|UI|React|Vue/i.test(userPrompt)) {
            relevantContext.preferences = this.filterPreferences(
                allContext.preferences,
                ['IDE', '프론트엔드']
            );
            relevantContext.entities = this.filterEntities(
                allContext.entities,
                ['frontend_framework', 'library']
            );
        }

        // 일반적인 개발 질문인 경우
        if (Object.keys(relevantContext).length === 0) {
            relevantContext.preferences = allContext.preferences;
            relevantContext.entities = allContext.entities.slice(-3); // 최근 3개만
        }

        return relevantContext;
    }

    filterPreferences(preferences, categories) {
        const filtered = {};
        categories.forEach(category => {
            if (preferences[category]) {
                filtered[category] = preferences[category];
            }
        });
        return filtered;
    }

    filterEntities(entities, types) {
        return entities.filter(entity =>
            typeof entity === 'object' && types.includes(entity.type)
        );
    }
}

/**
 * 기존 aiFeatures.js에 통합할 개선된 컨텍스트 로직
 */
function improvedContextLogic() {
    return `
    // 기존 3428-3438번 라인을 다음과 같이 개선:

    // ✨ 지능적 컨텍스트 메모리 활용
    if (session && messagesWithHistory) {
        const contextManager = new IntelligentContextManager();
        const userPrompt = messages[messages.length - 1]?.content || '';

        // AI가 컨텍스트 필요성을 판단
        if (contextManager.shouldUseContext(userPrompt)) {
            try {
                const shortTermMemory = session.getShortTermMemory();
                const entities = session.getEntities();
                const workingContext = session.contextMemory.workingContext;
                const userPreferences = session.getUserPreference();

                const allContext = {
                    shortTermMemory,
                    entities,
                    workingContext,
                    preferences: userPreferences
                };

                // 질문과 관련된 컨텍스트만 선별
                const relevantContext = contextManager.selectRelevantContext(
                    userPrompt,
                    allContext
                );

                // 관련 컨텍스트가 있을 때만 주입
                if (this.hasRelevantContext(relevantContext)) {
                    const contextInfo = this.buildContextPrompt(relevantContext);
                    // 시스템 메시지에 추가...
                }
            } catch (error) {
                logger.warn('지능적 컨텍스트 처리 실패:', error.message);
            }
        }
    }
    `;
}

export { IntelligentContextManager };