import { requireAuth } from './auth.js';

export async function taskPage(chatMessages, translations, reqAPI, getText, showCustomDialog) {
    // 사용자 인증 체크
    const hasAccess = await requireAuth(reqAPI);
    if (!hasAccess) {
        return;
    }

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    inputContainer.style.display = 'none';

    chatMessages.innerHTML = `
        <div class="page-container">
            <div class="page-header">
                <h2 class="page-title">${getText('taskManager', 'title')}</h2>
                <p class="page-description">${getText('taskManager', 'description')}</p>
                <div class="page-actions">
                    <button id="addNewTask" class="btn-execute">${getText('taskManager', 'addNewTask')}</button>
                </div>
            </div>
            <div id="taskForm" style="display: none;">
                <div class="form-group">
                    <label for="taskName">${getText('taskManager', 'taskName')}</label>
                    <input type="text" id="taskName" placeholder="${getText('taskManager', 'taskNamePlaceholder')}">
                </div>
                <div class="form-group">
                    <label for="promptText">${getText('taskManager', 'prompt')}</label>
                    <textarea id="promptText" placeholder="${getText('taskManager', 'promptPlaceholder')}"></textarea>
                </div>
                <div class="page-actions">
                    <button id="saveTask" class="btn-execute">${getText('taskManager', 'save')}</button>
                    <button id="cancelTask" class="btn-delete">${getText('taskManager', 'cancel')}</button>
                </div>
            </div>
            <div id="taskList"></div>
        </div>
    `;

    // Add New Task 버튼 이벤트
    document.getElementById('addNewTask').addEventListener('click', () => {
        document.getElementById('taskForm').style.display = 'block';
    });

    // Cancel 버튼 이벤트
    document.getElementById('cancelTask').addEventListener('click', () => {
        document.getElementById('taskForm').style.display = 'none';
        document.getElementById('taskName').value = '';
        document.getElementById('promptText').value = '';
    });

    // Save 버튼 이벤트
    document.getElementById('saveTask').addEventListener('click', async () => {
        const taskName = document.getElementById('taskName').value;
        const prompt = document.getElementById('promptText').value;

        if (!taskName || !prompt) {
            await reqAPI('errnotify', getText('taskManager', 'emptyFieldsError'));
            return;
        }

        try {
            await reqAPI('saveTask', {
                name: taskName,
                prompt: prompt,
                createdAt: new Date().toISOString()
            });
            await reqAPI('errnotify', getText('taskManager', 'saveSuccess'));

            // 폼 초기화
            document.getElementById('taskName').value = '';
            document.getElementById('promptText').value = '';
            document.getElementById('taskForm').style.display = 'none';
            
            await refreshTaskList();
        } catch (error) {
            await reqAPI('errnotify', getText('taskManager', 'saveError'));
        }
    });

    // 작업 실행 함수
    async function executeTaskNow(id) {
        const executeButton = document.querySelector(`button[data-task-id="${id}"]`);
        try {       
            if (executeButton) {
                executeButton.disabled = true;
                executeButton.textContent = translations?.taskManager?.executing || 'Executing...';
            }
    
            const tasks = await reqAPI('getTasks') || [];
            const task = tasks.find(t => t.id === id);
            if (!task) {
                await reqAPI('errnotify', translations?.taskManager?.taskNotFound || 'Task not found.');
                return;
            }

            // 터미널 실행을 위한 프롬프트 문자열 정리
            const sanitizedPrompt = task.prompt
                .replace(/\\/g, '/') 
                .replace(/\s+/g, ' ')
                .trim();

            // Electron 환경 체크를 위한 설정값 가져오기
            const isElectron = await reqAPI('getconfig', { key: 'IS_ELECTRON' });
            
            if (isElectron === 'true') {
                const response = await reqAPI('executeTask', {
                    id: task.id,
                    name: task.name,
                    prompt: sanitizedPrompt,
                    code: task.code,
                    executionCount: task.executionCount,
                    lastUsed: task.lastUsed,
                    metadata: task.metadata
                });
                
                if (response && response.success) {
                    await reqAPI('errnotify', translations?.taskManager?.processingInChat || 'Task is being processed in chat window.');
                } else {
                    throw new Error(response.error || translations?.taskManager?.electronExecuteError || 'Failed to execute task in Electron environment');
                }
                return;
            }             

            const response = await reqAPI('executeTask', {
                id: task.id,
                name: task.name,
                prompt: sanitizedPrompt,
                code: task.code,
                executionCount: task.executionCount,
                lastUsed: task.lastUsed,
                metadata: task.metadata
            });
    
            if (response && response.success) {
                await reqAPI('errnotify', translations?.taskManager?.executeSuccess || 'Task has been executed successfully.');
            } else if (response && response.error) {
                await showCustomDialog({
                    title: getText('dialog', 'errorTitle'),
                    message: response.error,
                    type: 'alert'
                });
            } else {
                throw new Error(getText('taskManager', 'executeFailed') || 'Task execution failed');
            }
        } catch (error) {
            console.error('Task execution error:', error);
            await reqAPI('errnotify', `${translations?.taskManager?.executeError || 'An error occurred while executing the task'}: ${error.message}`);
            await showCustomDialog({
                title: getText('dialog', 'errorTitle'),
                message: error.message || translations?.taskManager?.unexpectedError || 'An unexpected error occurred.',
                type: 'alert'
            });
        } finally {
            if (executeButton) {
                executeButton.disabled = false;
                executeButton.textContent = getText('taskManager', 'executeNow');
            }
        }
    }

    // 작업 삭제 함수
    async function deleteTask(id) {
        const confirmed = await showCustomDialog({
            title: getText('dialog', 'deleteTaskTitle'),
            message: getText('dialog', 'deleteTaskMessage'),
            type: 'confirm'
        });
        
        if (confirmed) {
            try {
                await reqAPI('deleteTask', { id });
                await refreshTaskList();
            } catch (error) {
                await reqAPI('errnotify', getText('taskManager', 'deleteError'));
            }
        }
    }

    // 작업 목록 새로고침 함수
    async function refreshTaskList() {
        const taskList = document.getElementById('taskList');
        let tasks = await reqAPI('getTasks') || [];
        
        // tasks가 배열이 아닌 경우 빈 배열로 초기화
        if (!Array.isArray(tasks)) {
            console.warn('getTasks did not return an array:', tasks);
            tasks = [];
        }
        
        // 작업 목록을 마지막 사용 시간 기준으로 내림차순 정렬
        tasks.sort((a, b) => {
            const dateA = new Date(a?.lastUsed || a?.createdAt || 0);
            const dateB = new Date(b?.lastUsed || b?.createdAt || 0);
            return dateB - dateA;
        });
        
        taskList.innerHTML = tasks.map(task => `
            <div class="task-item ${task.code ? 'has-code' : ''}">
                <div class="task-info">
                    <div class="task-header">
                        ${task.code ? '<span class="code-icon">⚡</span>' : ''}
                        <h3>${task.name || ''}</h3>
                    </div>
                    <p class="task-prompt">${task.prompt || ''}</p>
                    <div class="task-metadata">
                        ${task.code ? `
                            <span class="cache-info">
                                <span class="cache-status">캐시됨</span>
                                <span class="execution-count">실행 횟수: ${task.executionCount || 1}</span>
                                <span class="last-used">마지막 실행: ${new Date(task.lastUsed || task.createdAt).toLocaleString()}</span>
                            </span>
                        ` : ''}
                        <span class="created-at">생성일: ${new Date(task.createdAt).toLocaleString()}</span>
                    </div>
                </div>
                <div class="task-actions">
                    <button class="btn-execute" data-task-id="${task.id}">${getText('taskManager', 'executeNow')}</button>
                    <button class="btn-delete" data-task-id="${task.id}">${getText('taskManager', 'delete')}</button>
                </div>
            </div>
        `).join('') || `<div class="no-tasks">${getText('taskManager', 'noTasks')}</div>`;

        // 이벤트 리스너 설정
        document.querySelectorAll('.btn-execute[data-task-id]').forEach(button => {
            const id = button.getAttribute('data-task-id');
            button.addEventListener('click', () => executeTaskNow(id));
        });

        document.querySelectorAll('.btn-delete[data-task-id]').forEach(button => {
            const id = button.getAttribute('data-task-id');
            button.addEventListener('click', () => deleteTask(id));
        });
    }

    await refreshTaskList();
} 