import { requireAdmin } from './auth.js';

export async function userPage(chatMessages, translations, reqAPI, getText, showCustomDialog) {
    // 관리자 권한 체크
    const hasAccess = await requireAdmin(reqAPI);
    if (!hasAccess) {
        return;
    }

    // 전역 상태 관리
    let currentPage = 1;
    let itemsPerPage = 10;
    let currentUserData = null;

    // 전역 함수들을 먼저 등록
    window.openUserDetail = openUserDetail;
    window.deleteUser = deleteUser;
    window.copyApiKey = copyApiKey;
    window.handlePageChange = async function(page) {
        console.log('[DEBUG] handlePageChange called with page:', page);
        currentPage = page;
        
        // URL 업데이트
        const newUrl = new URL(window.location.href);
        newUrl.searchParams.set('page', page);
        window.history.pushState({ page }, '', newUrl);

        // 로딩 상태 표시
        const userList = document.getElementById('userList');
        if (userList) {
            userList.innerHTML = '<div class="loading">로딩 중...</div>';
        }

        await loadUserData();
    };

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    inputContainer.style.display = 'none';

    // 페이지 기본 구조 생성
    chatMessages.innerHTML = `
        <div class="page-container">
            <div class="page-header">
                <div class="header-content">
                    <h2 class="page-title">${getText('userManager', 'title')}</h2>
                    <p class="page-description">${getText('userManager', 'description')}</p>
                </div>
                <div class="page-actions">
                    <button id="addNewUser" class="btn-execute">${getText('userManager', 'addNewUser')}</button>
                    <button id="logoutBtn" class="btn-delete">${getText('userManager', 'logout') || '로그아웃'}</button>
                </div>
            </div>
            
            <!-- 사용자 상세 정보 모달 -->
            <div id="userDetailModal" class="modal">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3 id="modalTitle">사용자 상세 정보</h3>
                        <button class="btn-close" id="closeUserDetail">&times;</button>
                    </div>
                    <div class="modal-body">
                        <div class="form-group">
                            <label for="detailUsername">아이디</label>
                            <input type="text" id="detailUsername" readonly>
                        </div>
                        <div class="form-group">
                            <label for="detailName">이름</label>
                            <input type="text" id="detailName">
                        </div>
                        <div class="form-group">
                            <label for="detailEmail">이메일</label>
                            <input type="email" id="detailEmail">
                        </div>
                        <div class="form-group">
                            <label for="detailPassword">새 비밀번호</label>
                            <input type="password" id="detailPassword" placeholder="변경하려면 입력하세요">
                        </div>
                        <div class="form-group">
                            <label for="detailRole">권한</label>
                            <select id="detailRole">
                                <option value="user">일반 사용자</option>
                                <option value="admin">관리자</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>API 키</label>
                            <div id="detailApiKeys" class="api-keys-container"></div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button id="saveUserDetail" class="btn-execute">저장</button>
                        <button id="closeUserDetail2" class="btn-cancel">취소</button>
                    </div>
                </div>
            </div>

            <div id="userForm" style="display: none;">
                <div class="form-group">
                    <label for="username">아이디</label>
                    <input type="text" id="username" placeholder="사용자 아이디 입력">
                </div>
                <div class="form-group">
                    <label for="name">이름</label>
                    <input type="text" id="name" placeholder="사용자 이름 입력">
                </div>
                <div class="form-group">
                    <label for="email">이메일</label>
                    <input type="email" id="email" placeholder="이메일 주소 입력">
                </div>
                <div class="form-group">
                    <label for="password">비밀번호</label>
                    <input type="password" id="password" placeholder="비밀번호 입력">
                </div>
                <div class="form-group">
                    <label for="role">권한</label>
                    <select id="role">
                        <option value="user">일반 사용자</option>
                        <option value="admin">관리자</option>
                    </select>
                </div>
                <div class="page-actions">
                    <button id="saveUser" class="btn-execute">저장</button>
                    <button id="cancelUser" class="btn-delete">취소</button>
                </div>
            </div>

            <div id="userList"></div>
        </div>
    `;

    // 사용자 상세 정보 모달 열기
    async function openUserDetail(userId) {
        console.log('[DEBUG] Opening user detail modal for userId:', userId);
        currentUserData = null;
        const modal = document.getElementById('userDetailModal');
        
        try {
            // 사용자 정보 가져오기
            const response = await reqAPI(`admin/users/${userId}`, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('userToken')}`
                }
            });

            console.log('[DEBUG] User detail API response:', response);

            if (!response || !response.success) {
                throw new Error(response?.error?.message || '사용자 정보를 불러올 수 없습니다.');
            }

            currentUserData = response.data;

            // 폼에 데이터 채우기
            document.getElementById('detailUsername').value = currentUserData.username || '';
            document.getElementById('detailName').value = currentUserData.name || '';
            document.getElementById('detailEmail').value = currentUserData.email || '';
            document.getElementById('detailRole').value = currentUserData.role || 'user';
            document.getElementById('detailPassword').value = '';

            // API 키 표시
            const apiKeysContainer = document.getElementById('detailApiKeys');
            if (Array.isArray(currentUserData.apiKeys) && currentUserData.apiKeys.length > 0) {
                apiKeysContainer.innerHTML = currentUserData.apiKeys.map(key => `
                    <div class="api-key-item">
                        <div class="api-key-value">
                            <span class="masked-key">••••••••</span>
                            <button class="btn-copy btn-sm" onclick="window.copyApiKey('${key.key}')">
                                ${getText('common', 'copy')}
                            </button>
                        </div>
                    </div>
                `).join('');
            } else {
                apiKeysContainer.innerHTML = '<p class="no-api-keys">(API 키 없음)</p>';
            }

            modal.style.display = 'block';
            console.log('[DEBUG] Modal displayed with data:', currentUserData);
        } catch (error) {
            console.error('[DEBUG] Error in openUserDetail:', error);
            await reqAPI('errnotify', getText('userManager', 'loadUserError'));
        }
    }

    // 사용자 상세 정보 저장
    async function saveUserDetail() {
        if (!currentUserData) return;

        try {
            const name = document.getElementById('detailName').value;
            const email = document.getElementById('detailEmail').value;
            const role = document.getElementById('detailRole').value;
            const password = document.getElementById('detailPassword').value;

            // 업데이트할 데이터 구성
            const updateData = {};
            if (name !== currentUserData.name) updateData.name = name;
            if (email !== currentUserData.email) updateData.email = email;
            if (role !== currentUserData.role) updateData.role = role;
            if (password.trim()) updateData.password = password.trim();

            console.log('[DEBUG] Update data:', updateData);

            // 업데이트할 내용이 없는 경우
            if (Object.keys(updateData).length === 0) {
                await showCustomDialog({
                    title: getText('userManager', 'updateTitle'),
                    message: '변경된 내용이 없습니다.',
                    type: 'info'
                });
                return;
            }

            // 사용자 정보 업데이트
            const response = await reqAPI(`admin/users/${currentUserData.id}`, {
                method: 'PUT',
                data: updateData,
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('userToken')}`
                }
            });

            if (!response.success) {
                throw new Error(response.error?.message || '사용자 정보 업데이트 실패');
            }

            await showCustomDialog({
                title: getText('userManager', 'updateTitle'),
                message: getText('userManager', 'updateSuccess'),
                type: 'success'
            });

            document.getElementById('userDetailModal').style.display = 'none';
            await loadUserData();
        } catch (error) {
            console.error('[DEBUG] Error updating user:', error);
            await reqAPI('errnotify', error.message || getText('userManager', 'updateError'));
        }
    }

    // 새 사용자 저장
    async function saveNewUser() {
        const username = document.getElementById('username').value;
        const name = document.getElementById('name').value;
        const email = document.getElementById('email').value;
        const password = document.getElementById('password').value;
        const role = document.getElementById('role').value;

        if (!username || !password) {
            await reqAPI('errnotify', getText('userManager', 'requiredFieldsError') || '아이디와 비밀번호는 필수 입력 항목입니다.');
            return;
        }

        try {
            const response = await reqAPI('auth/register', {
                username,
                name,
                email,
                password,
                role
            });

            if (!response.success) {
                throw new Error(response.error || '사용자 추가 실패');
            }

            await showCustomDialog({
                title: getText('userManager', 'saveSuccess'),
                message: getText('userManager', 'saveSuccess'),
                type: 'success'
            });

            // 폼 초기화 및 숨기기
            document.getElementById('username').value = '';
            document.getElementById('name').value = '';
            document.getElementById('email').value = '';
            document.getElementById('password').value = '';
            document.getElementById('role').value = 'user';
            document.getElementById('userForm').style.display = 'none';

            // 페이지 새로고침
            location.href = location.href;
        } catch (error) {
            await reqAPI('errnotify', getText('userManager', 'saveError'));
        }
    }

    // 모달 닫기 이벤트
    document.querySelectorAll('#closeUserDetail, #closeUserDetail2').forEach(button => {
        button.addEventListener('click', () => {
            document.getElementById('userDetailModal').style.display = 'none';
            currentUserData = null;
        });
    });

    // 저장 버튼 이벤트
    document.getElementById('saveUserDetail').addEventListener('click', saveUserDetail);

    // 새 사용자 추가 버튼 이벤트
    document.getElementById('addNewUser').addEventListener('click', () => {
        document.getElementById('userForm').style.display = 'block';
    });

    // 취소 버튼 이벤트
    document.getElementById('cancelUser').addEventListener('click', () => {
        document.getElementById('userForm').style.display = 'none';
        resetForm();
    });

    // 저장 버튼 이벤트
    document.getElementById('saveUser').addEventListener('click', saveNewUser);

    // 로그아웃 버튼 이벤트
    document.getElementById('logoutBtn').addEventListener('click', async () => {
        try {
            // 서버에 로그아웃 요청
            const response = await reqAPI('auth/logout', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('userToken')}`
                }
            });
            
            // 로컬 스토리지의 모든 인증 관련 데이터 삭제
            localStorage.removeItem('userToken');
            localStorage.removeItem('userRole');
            localStorage.removeItem('userName');
            localStorage.removeItem('username');
            localStorage.removeItem('userId');

            // 페이지 새로고침
            window.location.reload();
        } catch (error) {
            await reqAPI('errnotify', getText('userManager', 'logoutError') || '로그아웃 중 오류가 발생했습니다.');
        }
    });

    // 사용자 삭제 함수
    async function deleteUser(userId) {
        const confirmed = await showCustomDialog({
            title: getText('dialog', 'deleteUserTitle') || '사용자 삭제',
            message: getText('dialog', 'deleteUserMessage') || '이 사용자를 삭제하시겠습니까?',
            type: 'confirm'
        });
        
        if (confirmed) {
            try {
                const response = await reqAPI(`admin/users/${userId}`, {
                    method: 'DELETE',
                    headers: {
                        'Authorization': `Bearer ${localStorage.getItem('userToken')}`
                    }
                });

                if (response.success) {
                    await reqAPI('errnotify', getText('userManager', 'deleteSuccess') || '사용자가 삭제되었습니다.');
                    await loadUserData();
                } else {
                    throw new Error(response.error || '사용자 삭제 실패');
                }
            } catch (error) {
                await reqAPI('errnotify', getText('userManager', 'deleteError') || '사용자 삭제 중 오류가 발생했습니다.');
            }
        }
    }

    // API 키 복사 함수
    async function copyApiKey(key) {
        try {
            await navigator.clipboard.writeText(key);
            await showCustomDialog({
                title: 'API KEY',
                message: `API 키가 클립보드에 복사되었습니다.\n\n사용 방법:\n1. API 요청 시 헤더에 다음과 같이 추가하세요:\n   X-API-Key: ${key}\n\n2. 이 키는 안전하게 보관하고 노출되지 않도록 주의하세요.`,
                type: 'info'
            });
        } catch (error) {
            await reqAPI('errnotify', getText('userManager', 'apiKeyCopyError') || 'API 키 복사 중 오류가 발생했습니다.');
        }
    }

    // 사용자 데이터 로드 함수
    async function loadUserData() {
        console.log('[DEBUG] loadUserData called with page:', currentPage);
        const userList = document.getElementById('userList');
        if (!userList) {
            console.error('[DEBUG] userList element not found');
            return;
        }

        try {
            console.log('[DEBUG] API Request params:', { page: currentPage, limit: itemsPerPage });
            const response = await reqAPI(`admin/users?page=${currentPage}&limit=${itemsPerPage}`, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('userToken')}`
                }
            });

            console.log('[DEBUG] API Response:', response);
            console.log('[DEBUG] Response pagination:', response.data?.pagination);

            if (!response || !response.success) {
                throw new Error(response?.error?.message || '사용자 목록을 불러올 수 없습니다.');
            }

            // 서버 응답 데이터를 직접 사용
            currentUserData = {
                users: response.data.users,
                pagination: response.data.pagination
            };
            
            renderUserList(userList);
        } catch (error) {
            console.error('[DEBUG] Error in loadUserData:', error);
            // error.message가 객체인 경우를 처리
            let errorMessage = error?.message || getText('userManager', 'loadError');
            if (typeof errorMessage === 'object') {
                errorMessage = JSON.stringify(errorMessage);
            }
            userList.innerHTML = `<div class="error-message">${errorMessage}</div>`;
            await reqAPI('errnotify', '사용자 목록을 불러오는 중 오류가 발생했습니다.');
        }
    }

    // 사용자 목록 렌더링 함수
    function renderUserList(container) {
        if (!currentUserData) return;

        const { users = [], pagination = {} } = currentUserData;
        const { total = 0, page = currentPage, totalPages = 1 } = pagination;

        let html = `
            <div class="table-container">
                <table class="user-table">
                    <thead>
                        <tr>
                            <th>아이디</th>
                            <th>이름</th>
                            <th>이메일</th>
                            <th>${getText('userManager', 'role')}</th>
                            <th>API 키</th>
                            <th>작업</th>
                        </tr>
                    </thead>
                    <tbody>
        `;

        users.forEach(user => {
            html += `
                <tr class="user-row">
                    <td>
                        <a href="#" class="user-name-link" onclick="window.openUserDetail('${user._id || user.id}'); return false;">
                            ${user.username}
                        </a>
                    </td>
                    <td>${user.name || '-'}</td>
                    <td>${user.email || '-'}</td>
                    <td>
                        <span class="user-role ${user.role}">
                            ${getText('userManager', user.role === 'admin' ? 'roleAdmin' : 'roleUser')}
                        </span>
                    </td>
                    <td>
                        ${Array.isArray(user.apiKeys) && user.apiKeys.length > 0 ? `
                            <div class="api-key-item">
                                <div class="api-key-value">
                                    <span class="masked-key">••••••••</span>
                                    <button class="btn-copy btn-sm" onclick="window.copyApiKey('${user.apiKeys[0].key}'); return false;">
                                        ${getText('common', 'copy')}
                                    </button>
                                </div>
                            </div>
                        ` : '(API 키 없음)'}
                    </td>
                    <td>
                        <div class="action-buttons">
                            <button class="btn-delete btn-sm" onclick="window.deleteUser('${user._id || user.id}'); return false;">
                                ${getText('userManager', 'delete')}
                            </button>
                        </div>
                    </td>
                </tr>
            `;
        });

        html += `
                    </tbody>
                </table>
            </div>
        `;

        // 페이지네이션 추가
        if (totalPages > 1) {
            html += `
                <div class="pagination-container">
                    <div class="pagination-info">
                        전체 ${total}명의 사용자 (${page} / ${totalPages} 페이지)
                    </div>
                    <div class="pagination-controls">
            `;

            // 이전 페이지 버튼
            if (page > 1) {
                html += `
                    <button class="btn-page" onclick="window.handlePageChange(${page - 1}); return false;">
                        ${getText('common', 'previous')}
                    </button>
                `;
            }

            // 페이지 번호 버튼
            const maxVisible = 5;
            let startPage = Math.max(1, page - Math.floor(maxVisible / 2));
            let endPage = Math.min(totalPages, startPage + maxVisible - 1);

            if (endPage - startPage + 1 < maxVisible) {
                startPage = Math.max(1, endPage - maxVisible + 1);
            }

            for (let i = startPage; i <= endPage; i++) {
                html += `
                    <button class="btn-page ${i === page ? 'active' : ''}" 
                            onclick="window.handlePageChange(${i}); return false;"
                            ${i === page ? 'disabled' : ''}>
                        ${i}
                    </button>
                `;
            }

            // 다음 페이지 버튼
            if (page < totalPages) {
                html += `
                    <button class="btn-page" onclick="window.handlePageChange(${page + 1}); return false;">
                        ${getText('common', 'next')}
                    </button>
                `;
            }

            html += `
                    </div>
                </div>
            `;
        }

        // HTML 업데이트
        console.log('[DEBUG] Updating HTML');
        container.innerHTML = html;
        console.log('[DEBUG] HTML updated');
    }

    // 브라우저 뒤로가기/앞으로가기 처리
    window.onpopstate = async function(event) {
        console.log('[DEBUG] popstate event triggered');
        const urlParams = new URLSearchParams(window.location.search);
        const page = parseInt(urlParams.get('page')) || 1;
        await window.handlePageChange(page);
    };

    // 초기 페이지 로드
    const urlParams = new URLSearchParams(window.location.search);
    currentPage = parseInt(urlParams.get('page')) || 1;
    console.log('[DEBUG] Initial page load:', currentPage);
    await loadUserData();
} 