# Color definitions
$RED = [System.ConsoleColor]::Red
$GREEN = [System.ConsoleColor]::Green
$YELLOW = [System.ConsoleColor]::Yellow

# Check actual script location
$SCRIPT_DIR = Split-Path -Parent $MyInvocation.MyCommand.Path
# Project root directory (parent directory of script)
$PROJECT_ROOT = (Get-Item $SCRIPT_DIR).Parent.FullName

# Move to project root directory
Set-Location $PROJECT_ROOT

Write-Host "Project directory: " -ForegroundColor $YELLOW -NoNewline
Write-Host $PROJECT_ROOT

# Check current branch
$CURRENT_BRANCH = git rev-parse --abbrev-ref HEAD
if ($CURRENT_BRANCH -ne "dev") {
    Write-Host "Error: Can only be executed from dev branch." -ForegroundColor $RED
    Write-Host "Current branch: " -ForegroundColor $YELLOW -NoNewline
    Write-Host $CURRENT_BRANCH
    exit 1
}

# Check for uncommitted changes
if ($(git status --porcelain)) {
    Write-Host "Error: There are uncommitted changes. Please commit all changes first." -ForegroundColor $RED
    exit 1
}

# Check differences between master and dev branches
git fetch origin master dev
$MASTER_HEAD = git rev-parse origin/master
$DEV_HEAD = git rev-parse origin/dev

if ($MASTER_HEAD -eq $DEV_HEAD) {
    Write-Host "Error: master branch and dev branch are identical." -ForegroundColor $RED
    Write-Host "No new changes to release." -ForegroundColor $YELLOW
    exit 1
}

# Get current version
$CURRENT_VERSION = $(try { git describe --tags $(git rev-list --tags --max-count=1) } catch { "v0.0.0" })

# Version increment function
function Increment-Version {
    param (
        [string]$version,
        [string]$increment_type
    )
    
    # Remove v prefix
    $version = $version.TrimStart('v')
    
    # Split version by dots
    $VERSION_PARTS = $version.Split('.')
    
    $major = [int]$(if ($VERSION_PARTS.Length -gt 0) { $VERSION_PARTS[0] } else { "0" })
    $minor = [int]$(if ($VERSION_PARTS.Length -gt 1) { $VERSION_PARTS[1] } else { "0" })
    $patch = [int]$(if ($VERSION_PARTS.Length -gt 2) { $VERSION_PARTS[2] } else { "0" })
    
    switch ($increment_type) {
        "major" {
            $major++
            $minor = 0
            $patch = 0
        }
        "minor" {
            $minor++
            $patch = 0
        }
        default {
            $patch++
        }
    }
    
    return "v$major.$minor.$patch"
}

# Check version type (default: patch)
$VERSION_TYPE = if ($args[0]) { $args[0] } else { "patch" }
if ($VERSION_TYPE -notmatch '^(major|minor|patch)$') {
    Write-Host "Error: Version type must be one of: major, minor, patch" -ForegroundColor $RED
    exit 1
}

# Generate new version
$NEW_VERSION = Increment-Version $CURRENT_VERSION $VERSION_TYPE

# Update package.json version
Write-Host "`nUpdating package.json version..." -ForegroundColor $YELLOW
Write-Host "package.json version before update:" -ForegroundColor $YELLOW
Select-String -Path "package.json" -Pattern '"version":'

$content = Get-Content package.json -Raw
$content = $content -replace '"version": ".*"', "`"version`": `"$($NEW_VERSION.TrimStart('v'))`""
$content | Set-Content package.json -NoNewline

Write-Host "package.json version after update:" -ForegroundColor $YELLOW
Select-String -Path "package.json" -Pattern '"version":'

# Display changes summary
Write-Host "`nChanges summary:" -ForegroundColor $YELLOW
git --no-pager log --oneline origin/master..origin/dev

Write-Host "`nCurrent version: " -ForegroundColor $YELLOW -NoNewline
Write-Host $CURRENT_VERSION
Write-Host "New version: " -ForegroundColor $YELLOW -NoNewline
Write-Host $NEW_VERSION
Write-Host "`nStarting release process..." -ForegroundColor $GREEN

# User confirmation
$confirmation = Read-Host "`nDo you want to continue? (y/N)"
if ($confirmation -notmatch '^[Yy]$') {
    Write-Host "Release cancelled." -ForegroundColor $RED
    exit 1
}

# Update package-lock.json
Write-Host "`nRunning npm install on dev branch" -ForegroundColor $YELLOW
npm install

# Commit package.json and package-lock.json changes
Write-Host "`nCommitting package.json and package-lock.json" -ForegroundColor $YELLOW
git add package.json package-lock.json
git commit -m "chore: update dependencies and bump version to ${NEW_VERSION}"

# Execute release process
Write-Host "`n1. Switching to master branch" -ForegroundColor $YELLOW
git checkout master

Write-Host "`n2. Updating master branch" -ForegroundColor $YELLOW
git pull origin master

Write-Host "`n3. Merging dev branch" -ForegroundColor $YELLOW
git merge dev

Write-Host "`n5. Creating new tag" -ForegroundColor $YELLOW
git tag -a $NEW_VERSION -m "Release $NEW_VERSION"

Write-Host "`n6. Pushing changes" -ForegroundColor $YELLOW
git push origin master

Write-Host "`n7. Pushing tag" -ForegroundColor $YELLOW
git push origin $NEW_VERSION

Write-Host "`n8. Returning to dev branch" -ForegroundColor $YELLOW
git checkout dev

Write-Host "`n9. Syncing master changes to dev branch" -ForegroundColor $YELLOW
git merge master
git push origin dev

Write-Host "`n✨ Release completed! ($NEW_VERSION)" -ForegroundColor $GREEN 