import dbManager from '../database/index.js';

class PermissionService {
    constructor() {
        // Redis 의존성 제거
    }

    static Permissions = {
        READ_REPORT: 'read:report',
        WRITE_REPORT: 'write:report',
        DELETE_REPORT: 'delete:report',
        SHARE_REPORT: 'share:report',
        MANAGE_USERS: 'manage:users',
        MANAGE_API_KEYS: 'manage:api_keys',
        ADMIN: 'admin'
    };

    static Roles = {
        ADMIN: {
            name: 'admin',
            permissions: Object.values(PermissionService.Permissions)
        },
        MANAGER: {
            name: 'manager',
            permissions: [
                PermissionService.Permissions.READ_REPORT,
                PermissionService.Permissions.WRITE_REPORT,
                PermissionService.Permissions.SHARE_REPORT,
                PermissionService.Permissions.MANAGE_API_KEYS
            ]
        },
        USER: {
            name: 'user',
            permissions: [
                PermissionService.Permissions.READ_REPORT,
                PermissionService.Permissions.WRITE_REPORT
            ]
        }
    };

    async checkPermission(userId, permission) {
        try {
            const result = await dbManager.query(
                'SELECT role FROM users WHERE id = $1',
                [userId]
            );

            if (result.rows.length === 0) {
                return false;
            }

            const userRole = result.rows[0].role.toUpperCase();
            const rolePermissions = PermissionService.Roles[userRole]?.permissions || [];
            
            return rolePermissions.includes(permission);
        } catch (error) {
            console.error('권한 확인 중 오류:', error);
            return false;
        }
    }

    async assignRole(userId, role) {
        if (!PermissionService.Roles[role.toUpperCase()]) {
            throw new Error('유효하지 않은 역할입니다.');
        }

        const result = await dbManager.query(
            'UPDATE users SET role = $1 WHERE id = $2 RETURNING *',
            [role, userId]
        );

        if (result.rows.length === 0) {
            throw new Error('사용자를 찾을 수 없습니다.');
        }

        return { userId, role };
    }

    async assignRoleByUsername(username, role) {
        try {
            // 사용자 존재 여부 확인
            const userResult = await dbManager.query(
                'SELECT * FROM users WHERE username = $1',
                [username]
            );

            if (userResult.rows.length === 0) {
                throw new Error('사용자를 찾을 수 없습니다.');
            }

            const user = userResult.rows[0];

            // 관리자 계정이 1명뿐인데 그 계정의 역할을 변경하려는 경우 방지
            if (user.role === 'admin' && role !== 'admin') {
                const adminCount = await dbManager.query(
                    'SELECT COUNT(*) FROM users WHERE role = $1',
                    ['admin']
                );
                
                if (parseInt(adminCount.rows[0].count) <= 1) {
                    throw new Error('마지막 관리자 계정의 역할은 변경할 수 없습니다.');
                }
            }

            // 역할 업데이트
            const result = await dbManager.query(
                `UPDATE users 
                 SET role = $1
                 WHERE username = $2
                 RETURNING id, username, email, name, role, status, created_at, last_login`,
                [role, username]
            );

            return result.rows[0];
        } catch (error) {
            console.error('역할 할당 중 오류:', error);
            throw error;
        }
    }
}

export default PermissionService;
