"""
FLOP (Floating Point Operations) 추적 및 라이선스 관리 모듈

AI 모델의 연산량을 FLOP 단위로 측정하여
라이선스 비용을 산정하는 시스템입니다.

우선순위:
1. 로컬 LLM: 실제 하드웨어 FLOP 측정
2. 외부 서비스: 가상 FLOP 추정
"""

import os
import json
import time
import logging
import threading
import subprocess
from typing import Dict, Any, Optional, List, Tuple
from dataclasses import dataclass, asdict
from datetime import datetime, timedelta
import psutil
from pathlib import Path

# GPU 모니터링을 위한 선택적 import
try:
    import torch
    import GPUtil
    TORCH_AVAILABLE = True
except ImportError:
    TORCH_AVAILABLE = False
    
try:
    # nvidia-ml-py3 또는 nvidia-ml-py 사용
    try:
        import nvidia_ml_py3 as nvml
    except ImportError:
        import nvidia_ml_py as nvml
    
    nvml.nvmlInit()
    NVML_AVAILABLE = True
except ImportError:
    NVML_AVAILABLE = False

# FLOP 계산을 위한 모델별 파라미터 정보
MODEL_FLOP_CONFIGS = {
    # === 로컬 모델들 (실제 측정 가능) ===
    'llama3': {
        'parameters': 8e9,  # 8B
        'flops_per_token': 16e9,
        'type': 'transformer',
        'local': True,
        'measurement': 'hardware'  # 실제 하드웨어 측정
    },
    'llama3:13b': {
        'parameters': 13e9,  # 13B
        'flops_per_token': 26e9,
        'type': 'transformer',
        'local': True,
        'measurement': 'hardware'
    },
    'llama3:70b': {
        'parameters': 70e9,  # 70B
        'flops_per_token': 140e9,
        'type': 'transformer',
        'local': True,
        'measurement': 'hardware'
    },
    'mixtral': {
        'parameters': 46.7e9,  # 46.7B (8x7B MoE)
        'flops_per_token': 93.4e9,
        'type': 'moe',
        'local': True,
        'measurement': 'hardware'
    },
    'gemma-3-12b-it': {
        'parameters': 12e9,
        'flops_per_token': 24e9,
        'type': 'transformer',
        'local': True,
        'measurement': 'hardware'
    },
    'HyperCLOVAX-SEED': {
        'parameters': 1.5e9,
        'flops_per_token': 3e9,
        'type': 'transformer',
        'local': True,
        'measurement': 'hardware'
    },
    
    # === 파인튜닝 모델들 (실제 측정) ===
    'local_finetuning_sbert': {
        'parameters': 110e6,  # SBERT 기반 모델 (약 110M)
        'flops_per_token': 220e6,  # 토큰당 220M FLOPS
        'type': 'transformer',
        'local': True,
        'measurement': 'hardware'
    },
    'local_finetuning_generic': {
        'parameters': 300e6,  # 일반적인 파인튜닝 모델 (약 300M)
        'flops_per_token': 600e6,  # 토큰당 600M FLOPS
        'type': 'transformer',
        'local': True,
        'measurement': 'hardware'
    },
    
    # === 외부 모델들 (가상 FLOP 추정) ===
    'gpt-3.5-turbo': {
        'parameters': 175e9,  # 1750억
        'flops_per_token': 350e9,  # 토큰당 350 GFLOPS
        'type': 'transformer',
        'local': False,
        'measurement': 'virtual'  # 가상 FLOP 추정
    },
    'gpt-4': {
        'parameters': 1.76e12,  # 1.76조
        'flops_per_token': 3.5e12,  # 토큰당 3.5 TFLOPS
        'type': 'transformer',
        'local': False,
        'measurement': 'virtual'
    },
    'gpt-4-turbo': {
        'parameters': 1.76e12,
        'flops_per_token': 3.5e12,
        'type': 'transformer',
        'local': False,
        'measurement': 'virtual'
    },
    
    # Anthropic 모델들
    'claude-3-opus': {
        'parameters': 175e9,
        'flops_per_token': 350e9,
        'type': 'transformer',
        'local': False,
        'measurement': 'virtual'
    },
    'claude-3-sonnet': {
        'parameters': 65e9,
        'flops_per_token': 130e9,
        'type': 'transformer',
        'local': False,
        'measurement': 'virtual'
    },
    'claude-3-haiku': {
        'parameters': 8e9,
        'flops_per_token': 16e9,
        'type': 'transformer',
        'local': False,
        'measurement': 'virtual'
    },
    
    # Gemini 모델들
    'gemini-pro': {
        'parameters': 137e9,
        'flops_per_token': 274e9,
        'type': 'transformer',
        'local': False,
        'measurement': 'virtual'
    },
    'gemini-2.0-flash': {
        'parameters': 27e9,
        'flops_per_token': 54e9,
        'type': 'transformer',
        'local': False,
        'measurement': 'virtual'
    }
}

@dataclass
class FLOPUsage:
    """FLOP 사용량 데이터 클래스"""
    session_id: str
    provider: str
    model: str
    prompt_tokens: int
    completion_tokens: int
    total_tokens: int
    estimated_flops: float  # 총 FLOP 수
    flops_per_token: float  # 토큰당 FLOP 수
    timestamp: str
    duration_ms: float
    measurement_type: str  # 'hardware' or 'virtual'
    gpu_utilization: Optional[float] = None  # GPU 사용률 (%)
    gpu_memory_used: Optional[float] = None  # GPU 메모리 사용량 (MB)
    actual_gpu_flops: Optional[float] = None  # 실제 측정된 GPU FLOP
    user_id: Optional[str] = None
    username: Optional[str] = None
    feature: Optional[str] = None
    
    def to_dict(self) -> Dict[str, Any]:
        return asdict(self)

class HardwareFLOPMonitor:
    """실제 하드웨어 FLOP 측정 클래스"""
    
    def __init__(self):
        # logger 속성 충돌 방지를 위해 직접 logging.getLogger() 사용
        self.gpu_available = TORCH_AVAILABLE and torch.cuda.is_available()
        self.nvml_available = NVML_AVAILABLE
        
        if self.gpu_available:
            self.device_count = torch.cuda.device_count()
            logging.getLogger(__name__).info(f"GPU 감지됨: {self.device_count}개 디바이스")
        else:
            logging.getLogger(__name__).warning("GPU를 사용할 수 없습니다. CPU 모니터링만 가능합니다.")
    
    def start_monitoring(self) -> Dict[str, Any]:
        """모니터링 시작 - 시작 시점의 하드웨어 상태 기록"""
        start_state = {
            'timestamp': time.time(),
            'cpu_percent': psutil.cpu_percent(),
            'memory_percent': psutil.virtual_memory().percent,
            'gpu_states': []
        }
        
        if self.gpu_available:
            try:
                # GPUtil을 사용하여 GPU 정보 수집
                import GPUtil
                gpus = GPUtil.getGPUs()
                
                for i, gpu in enumerate(gpus):
                    gpu_state = {
                        'device_id': i,
                        'utilization': gpu.load * 100,  # GPUtil은 0-1 범위, 백분율로 변환
                        'memory_used': gpu.memoryUsed,  # MB
                        'memory_total': gpu.memoryTotal,  # MB
                        'temperature': gpu.temperature
                    }
                    start_state['gpu_states'].append(gpu_state)
                    
            except Exception as e:
                logging.getLogger(__name__).warning(f"GPUtil GPU 상태 조회 실패: {e}")
                
                # GPUtil 실패 시 PyTorch 기본 정보 사용
                for i in range(self.device_count):
                    gpu_state = {
                        'device_id': i,
                        'utilization': 0,  # PyTorch로는 사용률 측정 불가
                        'memory_used': torch.cuda.memory_allocated(i) / 1024 / 1024,  # MB
                        'memory_total': torch.cuda.get_device_properties(i).total_memory / 1024 / 1024,  # MB
                        'temperature': 0
                    }
                    start_state['gpu_states'].append(gpu_state)
        
        return start_state
    
    def end_monitoring(self, start_state: Dict[str, Any]) -> Dict[str, Any]:
        """모니터링 종료 - 실제 사용량 계산"""
        end_time = time.time()
        duration = end_time - start_state['timestamp']
        
        end_state = {
            'timestamp': end_time,
            'duration': duration,
            'cpu_percent': psutil.cpu_percent(),
            'memory_percent': psutil.virtual_memory().percent,
            'gpu_states': [],
            'estimated_flops': 0
        }
        
        total_gpu_utilization = 0
        active_gpus = 0
        
        if self.gpu_available:
            try:
                # GPUtil을 사용하여 현재 GPU 상태 수집
                import GPUtil
                gpus = GPUtil.getGPUs()
                
                for i, gpu in enumerate(gpus):
                    if i < len(start_state['gpu_states']):
                        start_util = start_state['gpu_states'][i]['utilization']
                        current_util = gpu.load * 100
                        avg_util = (start_util + current_util) / 2
                        
                        gpu_state = {
                            'device_id': i,
                            'utilization': current_util,
                            'memory_used': gpu.memoryUsed,
                            'avg_utilization': avg_util,
                            'flops_estimate': 0
                        }
                        
                        # GPU별 FLOP 추정 (GPU 성능 기반)
                        gpu_flops_estimate = self._estimate_gpu_flops(i, avg_util, duration)
                        gpu_state['flops_estimate'] = gpu_flops_estimate
                        
                        if avg_util > 5:  # 5% 이상 사용된 GPU만 계산
                            total_gpu_utilization += avg_util
                            active_gpus += 1
                        
                        end_state['gpu_states'].append(gpu_state)
                    
            except Exception as e:
                logging.getLogger(__name__).warning(f"GPUtil GPU 종료 상태 조회 실패: {e}")
                
                # GPUtil 실패 시 기본값 사용
                for i in range(min(self.device_count, len(start_state.get('gpu_states', [])))):
                    gpu_state = {
                        'device_id': i,
                        'utilization': 0,
                        'memory_used': torch.cuda.memory_allocated(i) / 1024 / 1024 if self.gpu_available else 0,
                        'avg_utilization': 0,
                        'flops_estimate': 0
                    }
                    end_state['gpu_states'].append(gpu_state)
        
        # 총 FLOP 추정
        end_state['estimated_flops'] = sum(gpu['flops_estimate'] for gpu in end_state['gpu_states'])
        end_state['avg_gpu_utilization'] = total_gpu_utilization / max(active_gpus, 1)
        
        return end_state
    
    def _estimate_gpu_flops(self, device_id: int, utilization: float, duration: float) -> float:
        """GPU별 실제 FLOP 추정"""
        try:
            if not self.gpu_available:
                return 0
            
            # GPU 모델별 이론적 성능 (TFLOPS)
            gpu_name = torch.cuda.get_device_name(device_id).lower()
            
            # 주요 GPU별 FP16 성능 (TFLOPS)
            gpu_performance = {
                'rtx 4090': 165,      # RTX 4090
                'rtx 4080': 121,      # RTX 4080
                'rtx 3090': 71,       # RTX 3090
                'rtx 3080': 58,       # RTX 3080
                'rtx 3070': 40,       # RTX 3070
                'rtx 3060': 25,       # RTX 3060
                'rtx 3050': 20,       # RTX 3050
                'a100': 312,          # A100 (FP16)
                'v100': 125,          # V100 (FP16)
                'h100': 756,          # H100 (FP16)
                't4': 65,             # Tesla T4
                'l4': 121,            # L4
            }
            
            # GPU 이름에서 성능 찾기
            peak_tflops = 50  # 기본값
            for gpu_key, tflops in gpu_performance.items():
                if gpu_key in gpu_name:
                    peak_tflops = tflops
                    break
            
            # 실제 FLOP 계산: 이론적 성능 × 사용률 × 시간
            actual_flops = peak_tflops * 1e12 * (utilization / 100) * duration
            
            logging.getLogger(__name__).debug(f"GPU {device_id} ({gpu_name}): {peak_tflops} TFLOPS × {utilization:.1f}% × {duration:.2f}s = {actual_flops:.2e} FLOPS")
            
            return actual_flops
            
        except Exception as e:
            logging.getLogger(__name__).error(f"GPU FLOP 추정 실패: {e}")
            return 0

class FLOPUsageTracker:
    """FLOP 사용량 추적 및 관리 클래스"""
    
    # __slots__를 사용하여 허용된 속성만 정의 (logger 제외)
    __slots__ = [
        'hardware_monitor', 'storage_path', 'usage_file', 'stats_file', 'lock_file',
        'usage_cache', 'stats_cache', '_lock'
    ]
    
    def __init__(self, storage_path: Optional[str] = None):
        # logger 속성 충돌 방지를 위해 직접 logging.getLogger() 사용
        
        # 하드웨어 모니터 초기화
        self.hardware_monitor = HardwareFLOPMonitor()
        
        # 저장 경로 설정
        if storage_path is None:
            storage_path = os.path.join(os.path.expanduser('~'), '.airun')
        
        self.storage_path = Path(storage_path)
        self.storage_path.mkdir(parents=True, exist_ok=True)
        
        self.usage_file = self.storage_path / 'flop_usage.json'
        self.stats_file = self.storage_path / 'flop_stats.json'
        self.lock_file = self.storage_path / '.flop_usage.lock'
        
        # 메모리 캐시
        self.usage_cache: List[FLOPUsage] = []
        self.stats_cache: Dict[str, Any] = {}
        
        # 스레드 락
        self._lock = threading.Lock()
        
        # 초기화
        self._load_usage_data()
        self._load_stats_data()
        
        # 초기화 완료 로그
        logging.getLogger(__name__).info(f"FLOP Tracker 초기화 완료: {self.storage_path}")
        logging.getLogger(__name__).info(f"하드웨어 모니터링: {'가능' if self.hardware_monitor.gpu_available else '불가능'}")
    
    def _normalize_model_key(self, provider: str, model: str) -> str:
        """모델 키를 정규화하여 MODEL_FLOP_CONFIGS에서 찾을 수 있도록 함"""
        # 파인튜닝의 경우 특별 처리
        if provider == 'local_finetuning':
            # 파인튜닝 모델은 일반적으로 transformer 기반
            if 'kure' in model.lower() or 'sbert' in model.lower():
                return 'local_finetuning_sbert'
            else:
                return 'local_finetuning_generic'
        
        # 기존 로직
        model_key = f"{provider}_{model}".lower().replace('-', '_').replace('/', '_')
        
        # 직접 매칭 시도
        if model_key in MODEL_FLOP_CONFIGS:
            return model_key
        
        # 모델명만으로 매칭 시도
        if model in MODEL_FLOP_CONFIGS:
            return model
        
        # 부분 매칭 시도
        for config_key in MODEL_FLOP_CONFIGS.keys():
            if model.lower() in config_key or config_key in model.lower():
                return config_key
        
        # 기본값 반환
        return model
    
    def _load_usage_data(self):
        """사용량 데이터 로드"""
        try:
            if self.usage_file.exists():
                with open(self.usage_file, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    self.usage_cache = [
                        FLOPUsage(**item) for item in data.get('usage', [])
                    ]
                logging.getLogger(__name__).debug(f"FLOP 사용량 데이터 로드: {len(self.usage_cache)}개 항목")
            else:
                self.usage_cache = []
        except Exception as e:
            logging.getLogger(__name__).error(f"FLOP 사용량 데이터 로드 실패: {e}")
            self.usage_cache = []
    
    def _load_stats_data(self):
        """통계 데이터 로드"""
        try:
            if self.stats_file.exists():
                with open(self.stats_file, 'r', encoding='utf-8') as f:
                    self.stats_cache = json.load(f)
            else:
                self.stats_cache = self._create_empty_stats()
        except Exception as e:
            logging.getLogger(__name__).error(f"FLOP 통계 데이터 로드 실패: {e}")
            self.stats_cache = self._create_empty_stats()
    
    def _create_empty_stats(self) -> Dict[str, Any]:
        """빈 통계 데이터 생성"""
        return {
            'total_flops': 0.0,
            'total_tokens': 0,
            'total_requests': 0,
            'by_provider': {},
            'by_model': {},
            'by_user': {},
            'daily_stats': {},
            'last_updated': datetime.now().isoformat()
        }
    
    def _save_usage_data(self):
        """사용량 데이터 저장"""
        try:
            data = {
                'usage': [usage.to_dict() for usage in self.usage_cache],
                'last_updated': datetime.now().isoformat()
            }
            
            with open(self.usage_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
                
        except Exception as e:
            logging.getLogger(__name__).error(f"FLOP 사용량 데이터 저장 실패: {e}")
    
    def _save_stats_data(self):
        """통계 데이터 저장"""
        try:
            self.stats_cache['last_updated'] = datetime.now().isoformat()
            
            with open(self.stats_file, 'w', encoding='utf-8') as f:
                json.dump(self.stats_cache, f, ensure_ascii=False, indent=2)
                
        except Exception as e:
            logging.getLogger(__name__).error(f"FLOP 통계 데이터 저장 실패: {e}")
    
    def estimate_flops(self, provider: str, model: str, prompt_tokens: int, 
                      completion_tokens: int) -> Tuple[float, float]:
        """
        모델과 토큰 수를 기반으로 FLOP 수 추정
        
        Returns:
            Tuple[총 FLOP 수, 토큰당 FLOP 수]
        """
        # 모델 키 정규화
        model_key = self._normalize_model_key(provider, model)
        
        # 모델 설정 조회
        model_config = MODEL_FLOP_CONFIGS.get(model_key)
        if not model_config:
            # 기본값 사용 (소형 모델 기준)
            flops_per_token = 8e9  # 8 GFLOPS per token
            logging.getLogger(__name__).warning(f"모델 '{model_key}' FLOP 설정 없음, 기본값 사용: {flops_per_token}")
        else:
            flops_per_token = model_config['flops_per_token']
        
        total_tokens = prompt_tokens + completion_tokens
        total_flops = total_tokens * flops_per_token
        
        return total_flops, flops_per_token
    
    def _normalize_model_key(self, provider: str, model: str) -> str:
        """모델 키 정규화"""
        # 프로바이더별 모델명 정규화
        if provider.lower() == 'ollama':
            # ollama 모델명에서 태그 제거
            return model.split(':')[0] if ':' in model else model
        elif provider.lower() == 'vllm':
            # vLLM 모델명 단순화
            if 'gemma' in model.lower():
                return 'gemma-3-12b-it'
            elif 'hyperclovax' in model.lower():
                return 'HyperCLOVAX-SEED'
        
        return model
    
    def track_usage(self, provider: str, model: str, prompt_tokens: int, 
                   completion_tokens: int, duration_ms: float = 0.0,
                   session_id: Optional[str] = None, user_id: Optional[str] = None,
                   username: Optional[str] = None, feature: Optional[str] = None,
                   hardware_monitoring_data: Optional[Dict[str, Any]] = None):
        """FLOP 사용량 추적 (하드웨어 실측 또는 가상 추정)"""
        
        with self._lock:
            try:
                # 모델 설정 조회
                model_key = self._normalize_model_key(provider, model)
                model_config = MODEL_FLOP_CONFIGS.get(model_key, {})
                is_local_model = model_config.get('local', False)
                measurement_type = model_config.get('measurement', 'virtual')
                
                # FLOP 계산 방식 결정
                if is_local_model and hardware_monitoring_data:
                    # 로컬 모델: 실제 하드웨어 측정 사용
                    total_flops = hardware_monitoring_data.get('estimated_flops', 0)
                    flops_per_token = total_flops / max(prompt_tokens + completion_tokens, 1)
                    gpu_utilization = hardware_monitoring_data.get('avg_gpu_utilization', 0)
                    gpu_memory_used = sum(gpu.get('memory_used', 0) for gpu in hardware_monitoring_data.get('gpu_states', []))
                    actual_gpu_flops = total_flops
                    
                    logging.getLogger(__name__).info(f"🔥 실제 하드웨어 FLOP 측정: {provider}/{model} - {total_flops:.2e} FLOPS (GPU 사용률: {gpu_utilization:.1f}%)")
                    
                else:
                    # 외부 모델: 가상 FLOP 추정
                    total_flops, flops_per_token = self.estimate_flops(
                        provider, model, prompt_tokens, completion_tokens
                    )
                    gpu_utilization = None
                    gpu_memory_used = None
                    actual_gpu_flops = None
                    
                    logging.getLogger(__name__).debug(f"📊 가상 FLOP 추정: {provider}/{model} - {total_flops:.2e} FLOPS")
                
                # 사용량 기록 생성
                usage = FLOPUsage(
                    session_id=session_id or f"session_{int(time.time())}",
                    provider=provider,
                    model=model,
                    prompt_tokens=prompt_tokens,
                    completion_tokens=completion_tokens,
                    total_tokens=prompt_tokens + completion_tokens,
                    estimated_flops=total_flops,
                    flops_per_token=flops_per_token,
                    timestamp=datetime.now().isoformat(),
                    duration_ms=duration_ms,
                    measurement_type=measurement_type,
                    gpu_utilization=gpu_utilization,
                    gpu_memory_used=gpu_memory_used,
                    actual_gpu_flops=actual_gpu_flops,
                    user_id=user_id,
                    username=username,
                    feature=feature
                )
                
                # 캐시에 추가
                self.usage_cache.append(usage)
                
                # 통계 업데이트
                self._update_stats(usage)
                
                # 주기적 저장 (1000개마다)
                if len(self.usage_cache) % 1000 == 0:
                    self._save_usage_data()
                    self._save_stats_data()
                
                return usage
                
            except Exception as e:
                logging.getLogger(__name__).error(f"FLOP 사용량 추적 실패: {e}")
                return None
    
    def track_with_hardware_monitoring(self, provider: str, model: str, prompt_tokens: int,
                                     completion_tokens: int, duration_ms: float = 0.0,
                                     session_id: Optional[str] = None, user_id: Optional[str] = None,
                                     username: Optional[str] = None, feature: Optional[str] = None):
        """하드웨어 모니터링과 함께 FLOP 사용량 추적"""
        
        # 모델이 로컬인지 확인
        model_key = self._normalize_model_key(provider, model)
        model_config = MODEL_FLOP_CONFIGS.get(model_key, {})
        is_local_model = model_config.get('local', False)
        
        if is_local_model and self.hardware_monitor.gpu_available:
            # 로컬 모델: 하드웨어 모니터링 시작
            start_state = self.hardware_monitor.start_monitoring()
            
            # 실제 AI 처리는 외부에서 수행
            # 여기서는 모니터링 시작만 하고 종료는 별도로 호출
            return {
                'monitoring_started': True,
                'start_state': start_state,
                'is_local_model': True
            }
        else:
            # 외부 모델: 바로 가상 FLOP 추적
            usage = self.track_usage(
                provider, model, prompt_tokens, completion_tokens, duration_ms,
                session_id, user_id, username, feature
            )
            return {
                'monitoring_started': False,
                'usage': usage,
                'is_local_model': False
            }
    
    def finish_hardware_monitoring(self, start_state: Dict[str, Any], provider: str, model: str,
                                 prompt_tokens: int, completion_tokens: int, duration_ms: float = 0.0,
                                 session_id: Optional[str] = None, user_id: Optional[str] = None,
                                 username: Optional[str] = None, feature: Optional[str] = None):
        """하드웨어 모니터링 종료 및 FLOP 사용량 기록"""
        
        try:
            # 하드웨어 모니터링 종료
            end_state = self.hardware_monitor.end_monitoring(start_state)
            
            # FLOP 사용량 추적
            usage = self.track_usage(
                provider, model, prompt_tokens, completion_tokens, duration_ms,
                session_id, user_id, username, feature, end_state
            )
            
            return {
                'usage': usage,
                'hardware_data': end_state,
                'success': True
            }
            
        except Exception as e:
            logging.getLogger(__name__).error(f"하드웨어 모니터링 종료 실패: {e}")
            return {
                'usage': None,
                'hardware_data': None,
                'success': False,
                'error': str(e)
            }
    
    def _update_stats(self, usage: FLOPUsage):
        """통계 데이터 업데이트"""
        # 전체 통계
        self.stats_cache['total_flops'] += usage.estimated_flops
        self.stats_cache['total_tokens'] += usage.total_tokens
        self.stats_cache['total_requests'] += 1
        
        # 프로바이더별 통계
        provider_stats = self.stats_cache['by_provider'].setdefault(usage.provider, {
            'flops': 0.0, 'tokens': 0, 'requests': 0
        })
        provider_stats['flops'] += usage.estimated_flops
        provider_stats['tokens'] += usage.total_tokens
        provider_stats['requests'] += 1
        
        # 모델별 통계
        model_key = f"{usage.provider}/{usage.model}"
        model_stats = self.stats_cache['by_model'].setdefault(model_key, {
            'flops': 0.0, 'tokens': 0, 'requests': 0
        })
        model_stats['flops'] += usage.estimated_flops
        model_stats['tokens'] += usage.total_tokens
        model_stats['requests'] += 1
        
        # 사용자별 통계
        if usage.user_id:
            user_stats = self.stats_cache['by_user'].setdefault(usage.user_id, {
                'flops': 0.0, 'tokens': 0, 'requests': 0, 'username': usage.username
            })
            user_stats['flops'] += usage.estimated_flops
            user_stats['tokens'] += usage.total_tokens
            user_stats['requests'] += 1
        
        # 일별 통계
        date_key = usage.timestamp[:10]  # YYYY-MM-DD
        daily_stats = self.stats_cache['daily_stats'].setdefault(date_key, {
            'flops': 0.0, 'tokens': 0, 'requests': 0
        })
        daily_stats['flops'] += usage.estimated_flops
        daily_stats['tokens'] += usage.total_tokens
        daily_stats['requests'] += 1
    
    def get_usage_stats(self, start_date: Optional[str] = None, 
                       end_date: Optional[str] = None) -> Dict[str, Any]:
        """사용량 통계 조회"""
        with self._lock:
            if start_date or end_date:
                # 기간별 통계 계산
                filtered_usage = self._filter_usage_by_date(start_date, end_date)
                return self._calculate_stats_from_usage(filtered_usage)
            else:
                # 전체 통계 반환
                return self.stats_cache.copy()
    
    def _filter_usage_by_date(self, start_date: Optional[str], 
                            end_date: Optional[str]) -> List[FLOPUsage]:
        """날짜별 사용량 필터링"""
        filtered = []
        for usage in self.usage_cache:
            usage_date = usage.timestamp[:10]
            
            if start_date and usage_date < start_date:
                continue
            if end_date and usage_date > end_date:
                continue
                
            filtered.append(usage)
        
        return filtered
    
    def _calculate_stats_from_usage(self, usage_list: List[FLOPUsage]) -> Dict[str, Any]:
        """사용량 리스트에서 통계 계산"""
        stats = self._create_empty_stats()
        
        for usage in usage_list:
            self._update_stats_dict(stats, usage)
        
        return stats
    
    def _update_stats_dict(self, stats: Dict[str, Any], usage: FLOPUsage):
        """통계 딕셔너리 업데이트"""
        # 전체 통계
        stats['total_flops'] += usage.estimated_flops
        stats['total_tokens'] += usage.total_tokens
        stats['total_requests'] += 1
        
        # 프로바이더별 통계
        provider_stats = stats['by_provider'].setdefault(usage.provider, {
            'flops': 0.0, 'tokens': 0, 'requests': 0
        })
        provider_stats['flops'] += usage.estimated_flops
        provider_stats['tokens'] += usage.total_tokens
        provider_stats['requests'] += 1
        
        # 모델별 통계
        model_key = f"{usage.provider}/{usage.model}"
        model_stats = stats['by_model'].setdefault(model_key, {
            'flops': 0.0, 'tokens': 0, 'requests': 0
        })
        model_stats['flops'] += usage.estimated_flops
        model_stats['tokens'] += usage.total_tokens
        model_stats['requests'] += 1
    
    def calculate_flop_cost(self, total_flops: float, cost_per_gflop: float = 0.001) -> float:
        """
        FLOP 수를 기반으로 비용 계산
        
        Args:
            total_flops: 총 FLOP 수
            cost_per_gflop: GFLOP당 비용 (기본값: $0.001)
        
        Returns:
            계산된 비용 (USD)
        """
        gflops = total_flops / 1e9  # GFLOP 단위로 변환
        return gflops * cost_per_gflop
    
    def get_flop_pricing_tiers(self) -> Dict[str, Dict[str, Any]]:
        """FLOP 기반 가격 정책 반환"""
        return {
            'basic': {
                'name': '베이직',
                'monthly_gflops_limit': 1e6,  # 1M GFLOPS/월
                'cost_per_gflop': 0.001,      # $0.001/GFLOP
                'overage_cost': 0.002         # 초과 시 $0.002/GFLOP
            },
            'standard': {
                'name': '스탠다드',
                'monthly_gflops_limit': 10e6,  # 10M GFLOPS/월
                'cost_per_gflop': 0.0008,      # $0.0008/GFLOP
                'overage_cost': 0.0015         # 초과 시 $0.0015/GFLOP
            },
            'premium': {
                'name': '프리미엄',
                'monthly_gflops_limit': 100e6,  # 100M GFLOPS/월
                'cost_per_gflop': 0.0005,       # $0.0005/GFLOP
                'overage_cost': 0.001           # 초과 시 $0.001/GFLOP
            },
            'enterprise': {
                'name': '엔터프라이즈',
                'monthly_gflops_limit': 0,      # 무제한
                'cost_per_gflop': 0.0003,       # $0.0003/GFLOP
                'overage_cost': 0               # 초과 비용 없음
            }
        }
    
    def save_data(self):
        """데이터 강제 저장"""
        with self._lock:
            self._save_usage_data()
            self._save_stats_data()
            logging.getLogger(__name__).info("FLOP 데이터 저장 완료")
    
    def cleanup_old_data(self, days_to_keep: int = 90):
        """오래된 데이터 정리"""
        cutoff_date = (datetime.now() - timedelta(days=days_to_keep)).isoformat()
        
        with self._lock:
            original_count = len(self.usage_cache)
            self.usage_cache = [
                usage for usage in self.usage_cache 
                if usage.timestamp >= cutoff_date
            ]
            
            removed_count = original_count - len(self.usage_cache)
            if removed_count > 0:
                logging.getLogger(__name__).info(f"오래된 FLOP 데이터 {removed_count}개 정리 완료")
                self._save_usage_data()

# 싱글톤 인스턴스
flop_tracker = FLOPUsageTracker()

def format_flops(flops: float) -> str:
    """FLOP 수를 읽기 쉬운 형태로 포맷"""
    if flops >= 1e15:
        return f"{flops/1e15:.2f} PFLOPS"
    elif flops >= 1e12:
        return f"{flops/1e12:.2f} TFLOPS"
    elif flops >= 1e9:
        return f"{flops/1e9:.2f} GFLOPS"
    elif flops >= 1e6:
        return f"{flops/1e6:.2f} MFLOPS"
    elif flops >= 1e3:
        return f"{flops/1e3:.2f} KFLOPS"
    else:
        return f"{flops:.2f} FLOPS"

if __name__ == "__main__":
    # 테스트 코드
    logging.basicConfig(level=logging.INFO)
    
    tracker = FLOPUsageTracker()
    
    # 테스트 사용량 추가
    usage = tracker.track_usage(
        provider="openai",
        model="gpt-4",
        prompt_tokens=1000,
        completion_tokens=500,
        duration_ms=2500.0,
        session_id="test_session",
        user_id="test_user",
        username="테스트사용자"
    )
    
    if usage:
        print(f"추적된 사용량: {format_flops(usage.estimated_flops)}")
        print(f"토큰당 FLOP: {format_flops(usage.flops_per_token)}")
    
    # 통계 조회
    stats = tracker.get_usage_stats()
    print(f"총 FLOP 사용량: {format_flops(stats['total_flops'])}")
    print(f"총 요청 수: {stats['total_requests']}")
    
    # 데이터 저장
    tracker.save_data() 