import { aiChat } from '../../../aiFeatures.js';
import { shell_exec } from '../../../codeExecution.js';
import ProgressDisplay from '../utils/ProgressDisplay.js';
import { getVarVal } from '../../../configuration.js';
import fs from 'fs';
import path from 'path';
import { generateModuleInstallCode } from '../../../codeExecution.js';

class ExecutionManager {
    constructor() {
        this.maxRetries = 5;  // 최대 재시도 횟수
        this.directories = {
            inputs: [],      // 여러 입력 경로 저장
            output: null,    // 출력 경로
            temp: process.cwd()  // 임시 작업 경로
        };
        this.progress = new ProgressDisplay();
        this.executionResults = new Map();
    }

    parsePathInformation(analysis) {
        // TaskAnalyzer의 응답에서 경로 정보 파싱
        const inputPathsMatch = analysis.match(/(?:입력 경로|Input Paths?):\s*([^\n]+)/i);
        const outputPathMatch = analysis.match(/(?:출력 경로|Output Path?):\s*([^\n]+)/i);

        if (inputPathsMatch) {
            this.directories.inputs = inputPathsMatch[1]
                .split(/[,，]/)  // 한글/영어 쉼표 모두 처리
                .map(path => path.trim())
                .filter(path => path && !path.includes('[') && !path.includes(']'));
        }

        if (outputPathMatch) {
            const outputPath = outputPathMatch[1].trim();
            if (!outputPath.includes('[') && !outputPath.includes(']')) {
                this.directories.output = outputPath;
            }
        }

        // 임시 디렉토리는 현재 작업 디렉토리 내에 생성
        this.directories.temp = path.join(process.cwd(), 'temp');

        this.progress.showInfo('=== 분석된 경로 ===');
        this.progress.showInfo('입력 경로: ' + this.directories.inputs.join(', '));
        this.progress.showInfo('출력 경로: ' + this.directories.output);
        this.progress.showInfo('임시 경로: ' + this.directories.temp);
    }

    async executeStep(step) {
        this.progress.showStepHeader(step);
        
        try {
            // 실행 시작 시 상태 업데이트
            if (step.context && step.context.taskSteps) {
                const taskStep = step.context.taskSteps.find(s => s.id === step.id);
                if (taskStep) {
                    taskStep.status = 'running';
                    taskStep.startTime = new Date().toISOString();
                }
            }

            let code = await this.generateCode(step);
            this.progress.showCodeGeneration(true, code);
            let retryCount = 0;
            
            while (retryCount < this.maxRetries) {
                // 코드 실행
                const result = await this.executeCode(code);
                
                // AI가 실행 결과만 분석
                const analysis = await this.analyzeExecutionResult(result.stdout, result.stderr, step);
                
                // 분석 결과에 따른 조치
                if (analysis.success) {
                    const executionResult = {
                        id: step.id,
                        number: step.number,
                        description: step.description,
                        script: code,
                        result: result.stdout || '',
                        status: 'completed',  // 성공 상태
                        error: null
                    };
                    this.executionResults.set(step.id, executionResult);
                    
                    // 단계 상태 업데이트
                    if (step.context && step.context.taskSteps) {
                        const taskStep = step.context.taskSteps.find(s => s.id === step.id);
                        if (taskStep) {
                            taskStep.status = 'completed';
                            taskStep.endTime = new Date().toISOString();
                            taskStep.result = {
                                output: result.stdout || '',
                                files: [],
                                data: null
                            };
                            taskStep.error = null;
                        }
                    }
                    
                    return executionResult;
                }
                
                // 실패 시 조치
                if (analysis.needsPackageInstall) {
                    // 패키지 설치 후 같은 코드로 재시도
                    await this.installPackage(analysis.packageName);
                } else {
                    // 패키지 문제가 아닌 경우 새로운 코드 생성
                    code = await this.generateCode(step);
                }
                
                retryCount++;
            }
            
            throw new Error('최대 재시도 횟수 초과');
        } catch (error) {
            const errorResult = {
                id: step.id,
                number: step.number,
                description: step.description,
                error: error.message,
                status: 'failed',  // 실패 상태
                result: null
            };
            this.executionResults.set(step.id, errorResult);
            
            // 단계 상태 업데이트
            if (step.context && step.context.taskSteps) {
                const taskStep = step.context.taskSteps.find(s => s.id === step.id);
                if (taskStep) {
                    taskStep.status = 'failed';
                    taskStep.endTime = new Date().toISOString();
                    taskStep.error = error.message;
                    taskStep.result = null;
                }
            }
            
            throw error;
        }
    }

    async executeCode(code) {
        const venvPath = await getVarVal('PYTHON_VENV_PATH');
        if (!venvPath) {
            throw new Error('Python virtual environment path not found in configuration');
        }

        // 디버그: 저장될 코드 출력
        // this.progress.showInfo('=== 실행할 코드 ===');
        // this.progress.showCodeBox(code);

        // 임시 파일에 코드 저장
        const tempFilePath = path.join(venvPath, '._code.py');
        await fs.promises.writeFile(tempFilePath, code, 'utf-8');

        try {
            // generateModuleInstallCode를 사용하여 실행 코드 생성
            const execCode = await generateModuleInstallCode(code, true);
            const result = await shell_exec(execCode.code, false, true, true);
            return result;
        } finally {
            try {
                await fs.promises.unlink(tempFilePath);
            } catch {
                // 임시 파일 삭제 실패는 무시
            }
        }
    }

    async generateCode(step) {
        const prompt = `You are a Python code generator specialized in using the utils module. Your primary role is to generate Python code that strictly follows the provided guidelines.

# Task Overview:
## Original Request:
${step.context.originalRequest}

## Task Analysis:
- Total Steps: ${step.context.taskSteps.length}
${step.context.taskSteps.map(s => `- Step ${s.number}: ${s.description}`).join('\n')}

## Current Progress:
### Previous Steps:
${step.context.taskSteps
    .filter(s => s.number < step.number)
    .map(s => `${s.number}. [${s.status}] ${s.description}`).join('\n')}

### Current Step (${step.number}/${step.context.taskSteps.length}):
${step.description}

## Working Environment:
- Input Directories: ${step.context.paths.inputs.join(', ')}
- Output Directory: ${step.context.paths.output}
- Temp Directory: ${this.directories.temp}

## CORE PRINCIPLES (In order of priority):
1. USE utils module functions when available for their specific purposes
2. MAINTAIN consistent error handling and logging patterns
3. FOLLOW Python best practices for new functionality
4. CREATE new functions only when utils module doesn't provide the needed functionality

##Available Utils Functions:
1. File Operations:
   - safe_path_join: Safe function for path joining
     Example: path = safe_path_join(output_dir, "results.txt")
   
   - read_file: Read any file type
     Example: content = read_file("document.pdf")  # Returns str
             data = read_file("data.xlsx")  # Returns DataFrame
   
   - write_file: Write content to files
     Example: write_file("output.txt", "content")
             write_file("data.csv", dataframe)
   
   - list_directory: List directory contents
     Example: files = list_directory("/path/to/dir")  # List all files
             files = list_directory("/path/to/dir", pattern="*.pdf")  # Filter by pattern
   
   - remove_file: Safe file removal
     Example: remove_file("old_file.txt")
   
   - remove_directory_recursively: Safe recursive directory removal
     Example: remove_directory_recursively("old_dir")
   
   - create_directory: Create directory safely
     Example: create_directory("new_dir")
   
   - ensure_directory: Ensure directory exists
     Example: ensure_directory("output_dir")

2. Visualization:
   - create_matplotlib: Create matplotlib figure (returns fig, ax, font_prop)
   - save_plot: Save matplotlib plots with options
   
3. Document Processing:
   - create_pdf_document: Create PDF documents with Korean support
   - extract_from_hwp: Extract text from HWP/HWPX files
   - extract_tables_from_hwp: Extract tables from HWP to Excel
   - extract_images_from_hwp: Extract images from HWP
   
4. Search and Content:
   - search_content: Search content in files
   - search_web: Web search (supports various sites)
   - extract_web_content: Extract content from web pages
   
5. Browser Actions:
   - open_in_browser: Open URLs in browser
   - open_search_result: Open search results
   
6. Communication:
   - send_email: Send emails with attachments
   
7. Conversion:
   - convert_dot_to_svg: Convert DOT files to SVG
   
8. AI Processing:
   - summarize_content: Summarize text using AI
   - translate_content: Translate text using AI
   - review_code: Review code using AI
   - TextProcessor: Class for text processing operations

## CRITICAL RULES:
1. ONLY generate valid Python code
2. NO explanations or markdown
3. NO comments except step markers
4. Code must be immediately executable
5. Use print() for all output
6. Use % formatting, never f-strings
7. Wrap main code in try-except
8. Show progress with print statements
9. ALWAYS print meaningful execution results:
   - Print what was found/processed
   - Print where results were saved
   - Print counts and summaries
   - Print any important data or paths
10. Print clear start and end messages for each step
11. Print error details when exceptions occur
12. NEVER use built-in alternatives for utils functions:
    ❌ NO: with open("file.txt") as f:
    ✅ YES: content = read_file("file.txt")
    
    ❌ NO: os.path.join(dir, file)
    ✅ YES: safe_path_join(dir, file)
    
    ❌ NO: os.makedirs(dir)
    ✅ YES: ensure_directory(dir)
    
    ❌ NO: os.remove(file)
    ✅ YES: remove_file(file)
    
    ❌ NO: shutil.rmtree(dir)
    ✅ YES: remove_directory_recursively(dir)
    
    ❌ NO: os.listdir(dir)
    ✅ YES: list_directory(dir)

Example Code Structure:
import os
import sys
from utils import (
    read_file,
    write_file,
    safe_path_join,
    list_directory,
    ensure_directory
)

try:
    print("Starting process...")
    
    input_dirs = ${JSON.stringify(step.context.paths.inputs)}
    output_dir = "${step.context.paths.output}"
    temp_dir = "${this.directories.temp}"
    
    print("Processing input directories: %s" % ", ".join(input_dirs))
    
    # Create output directory if needed
    ensure_directory(output_dir)
    print("Ensured output directory exists: %s" % output_dir)
    
    # Process files
    found_files = []
    for input_dir in input_dirs:
        files = list_directory(input_dir, pattern="*.pdf")
        found_files.extend(files)
        print("Found %d files in: %s" % (len(files), input_dir))
    
    # Save results
    result_path = safe_path_join(output_dir, "results.txt")
    write_file(result_path, "\\n".join(found_files))
    print("Results saved to: %s" % result_path)
    
    print("Process completed successfully")
    print("Summary:")
    print("- Files processed: %d" % len(found_files))
    print("- Output saved to: %s" % output_dir)
except Exception as e:
    print("Error occurred: %s" % str(e))
    raise`;

        // AI에 전달되는 프롬프트 출력
        console.log('\n=== AI 프롬프트 ===\n');
        // 500자로 잘라서 출력
        console.log(prompt.length > 500 ? prompt.substring(0, 1000) + '...' : prompt);
        console.log('\n=== 프롬프트 끝 ===\n');

        let response;
        try {
            // aiChat 직접 호출 (API 서버 없이 독립 실행)
            response = await aiChat([{role: 'user', content: prompt}], {}, undefined, [], null, false);
            if (!response || typeof response !== 'string') {
                throw new Error('aiChat에서 유효한 응답을 받지 못했습니다');
            }
        } catch (error) {
            console.error('aiChat 호출 실패, 기본 코드로 대체:', error.message);
            response = this.getDefaultCode(step);
        }

        // AI 응답에서 Python 코드만 추출
        let code = response.replace(/```python\n/g, '')
                         .replace(/```\n/g, '')
                         .replace(/```/g, '')
                         .trim();

        // 생성된 코드 출력
        // this.progress.showCodeGeneration(true, code);

        return code;
    }

    async analyzeExecutionResult(stdout, stderr, step) {
        this.progress.showInfo('실행 결과 분석 중...');
        if (stderr) {
            this.progress.showWarning(`오류 메시지: ${stderr}`);
        }

        const prompt = `You are a Python code execution result analyzer.
Your only task is to analyze the execution result and determine:
1. If the execution was successful
2. If a package installation is needed (only if you find a ModuleNotFoundError)

Step Information:
- Step ${step.number}/${step.context.taskSteps.length}: ${step.description}

Code Execution Result:
${stderr ? 'Error Output:\n' + stderr + '\n' : ''}
Standard Output:
${stdout}

Respond in the following JSON format:
{
    "success": true | false,
    "needsPackageInstall": false,
    "packageName": null,
    "output": "stdout content",
    "error": "stderr content if any"
}

Note: Set needsPackageInstall to true ONLY if you find a ModuleNotFoundError, and set packageName to the name of the missing package.`;

        try {
            this.progress.startSpinner('AI에 실행 결과 분석 요청 중...');
            const response = await aiChat([{
                role: "system",
                content: prompt
            }]);

            const analysis = JSON.parse(response);
            this.progress.showSuccess('실행 결과 분석 완료');

            // 실행 결과 저장
            if (step.context && step.context.taskSteps) {
                const taskStep = step.context.taskSteps.find(s => s.id === step.id);
                if (taskStep) {
                    taskStep.result = {
                        output: analysis.output || '',
                        error: analysis.error || null,
                        files: [],
                        data: null
                    };
                }
            }

            return analysis;
        } catch (error) {
            return {
                success: false,
                needsPackageInstall: false,
                packageName: null,
                output: stdout,
                error: stderr
            };
        }
    }

    async installPackage(packageName) {
        this.progress.startSpinner(`${packageName} 패키지 설치 중...`);
        try {
            const importCode = await generateModuleInstallCode(`import ${packageName}`, true);
            await shell_exec(importCode.code, false, true, true);
            this.progress.showSuccess(`${packageName} 패키지 설치 완료`);
        } catch (error) {
            this.progress.showError(`${packageName} 패키지 설치 실패: ${error.message}`);
            throw error;
        }
    }

    printSummary() {
        this.progress.showExecutionResult(this.executionResults);
    }
    
    
    getDefaultCode(step) {
        // API 서버 호출 실패시 사용하는 기본 Python 코드
        return `
import os
import sys

try:
    print("Task: ${step.description}")
    print("Step execution started...")
    
    # Basic task execution
    print("Processing task: ${step.description}")
    
    # Create output directory
    output_dir = "${step.context.paths.output}"
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)
        print("Created output directory: %s" % output_dir)
    
    # Simple output
    result = "Task completed: ${step.description}"
    output_file = os.path.join(output_dir, "task_result.txt")
    
    with open(output_file, "w", encoding="utf-8") as f:
        f.write(result)
    
    print("Result saved to: %s" % output_file)
    print("Task execution completed successfully")
    
except Exception as e:
    print("Error occurred: %s" % str(e))
    raise
`;
    }
}

export default ExecutionManager; 