/**
 * @swagger
 * tags:
 *   - name: Auth
 *     description: 사용자 인증 및 관리 관련 API
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     LoginRequest:
 *       type: object
 *       required:
 *         - username
 *         - password
 *       properties:
 *         username:
 *           type: string
 *           description: 사용자 아이디
 *         password:
 *           type: string
 *           format: password
 *           description: 비밀번호
 *     LoginResponse:
 *       type: object
 *       properties:
 *         success:
 *           type: boolean
 *         data:
 *           type: object
 *           properties:
 *             user:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 username:
 *                   type: string
 *                 email:
 *                   type: string
 *                 name:
 *                   type: string
 *                 role:
 *                   type: string
 *                 created_at:
 *                   type: string
 *                   format: date-time
 *                 last_login:
 *                   type: string
 *                   format: date-time
 *                 status:
 *                   type: string
 *             token:
 *               type: string
 *               description: JWT 토큰
 *             apiKeys:
 *               type: array
 *               description: 사용자의 활성화된 API 키 목록
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   name:
 *                     type: string
 *                   key:
 *                     type: string
 *                   status:
 *                     type: string
 *                   created_at:
 *                     type: string
 *                   last_used:
 *                     type: string
 *                   permissions:
 *                     type: object
 *                     properties:
 *                       users:
 *                         type: object
 *                         properties:
 *                           read:
 *                             type: boolean
 *                           write:
 *                             type: boolean
 *                           manage:
 *                             type: boolean
 *                       reports:
 *                         type: object
 *                         properties:
 *                           read:
 *                             type: boolean
 *                           write:
 *                             type: boolean
 *                           share:
 *                             type: boolean
 *                           delete:
 *                             type: boolean
 *         error:
 *           type: object
 *           nullable: true
 *           properties:
 *             message:
 *               type: string
 *             code:
 *               type: string
 *         timestamp:
 *           type: string
 *           format: date-time
 *     RegisterRequest:
 *       type: object
 *       required:
 *         - username
 *         - password
 *         - email
 *       properties:
 *         username:
 *           type: string
 *           description: 사용자 아이디
 *         password:
 *           type: string
 *           format: password
 *           description: 비밀번호
 *         email:
 *           type: string
 *           format: email
 *           description: 이메일
 *         name:
 *           type: string
 *           description: 사용자 이름
 *         role:
 *           type: string
 *           description: 사용자 역할
 *           default: user
 *     ApiKey:
 *       type: object
 *       properties:
 *         id:
 *           type: string
 *           description: API 키 ID
 *         name:
 *           type: string
 *           description: API 키 이름
 *         key:
 *           type: string
 *           description: API 키 값
 *         status:
 *           type: string
 *           enum: [active, inactive]
 *           description: API 키 상태
 *         created_at:
 *           type: string
 *           format: date-time
 *         expires_at:
 *           type: string
 *           format: date-time
 *     CreateApiKeyRequest:
 *       type: object
 *       required:
 *         - name
 *       properties:
 *         name:
 *           type: string
 *           description: API 키 이름
 *         expires_at:
 *           type: string
 *           format: date-time
 *           description: 만료일 (선택사항)
 */

/**
 * @swagger
 * /api/v1/auth/login:
 *   post:
 *     summary: 로그인
 *     description: 사용자 인증 및 JWT 토큰 발급
 *     tags: [Auth]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/LoginRequest'
 *     responses:
 *       200:
 *         description: 로그인 성공
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/LoginResponse'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/auth/register:
 *   post:
 *     summary: 회원가입
 *     description: 새로운 사용자 등록
 *     tags: [Auth]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/RegisterRequest'
 *     responses:
 *       200:
 *         description: 회원가입 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     id:
 *                       type: string
 *                     username:
 *                       type: string
 */

/**
 * @swagger
 * /api/v1/auth/me:
 *   get:
 *     summary: 내 정보 조회
 *     description: 현재 로그인한 사용자의 정보를 조회합니다.
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: 사용자 정보 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     id:
 *                       type: string
 *                     username:
 *                       type: string
 *                     email:
 *                       type: string
 *                     role:
 *                       type: string
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/auth/logout:
 *   post:
 *     summary: 로그아웃
 *     description: 현재 세션을 종료하고 토큰을 무효화합니다.
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: 로그아웃 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 message:
 *                   type: string
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/auth/change-password:
 *   post:
 *     summary: 비밀번호 변경
 *     description: 로그인한 사용자의 비밀번호를 변경합니다.
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - currentPassword
 *               - newPassword
 *             properties:
 *               currentPassword:
 *                 type: string
 *                 format: password
 *                 description: 현재 비밀번호
 *               newPassword:
 *                 type: string
 *                 format: password
 *                 description: 새 비밀번호
 *     responses:
 *       200:
 *         description: 비밀번호 변경 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 message:
 *                   type: string
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/auth/activities:
 *   get:
 *     operationId: getUserActivities
 *     summary: 사용자 활동 내역 조회
 *     description: 현재 로그인한 사용자의 활동 내역을 조회합니다.
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: 활동 내역 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: array
 *                   items:
 *                     $ref: '#/components/schemas/Activity'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/admin/users:
 *   get:
 *     operationId: listUsers
 *     summary: 사용자 목록 조회
 *     description: 모든 사용자의 목록을 조회합니다. (관리자 전용)
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           default: 1
 *         description: 페이지 번호
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 10
 *         description: 페이지당 항목 수
 *       - in: query
 *         name: sortBy
 *         schema:
 *           type: string
 *         description: 정렬 기준 필드
 *       - in: query
 *         name: sortOrder
 *         schema:
 *           type: string
 *           enum: [ASC, DESC]
 *         description: 정렬 순서
 *       - in: query
 *         name: status
 *         schema:
 *           type: string
 *         description: 사용자 상태 필터
 *       - in: query
 *         name: role
 *         schema:
 *           type: string
 *         description: 사용자 역할 필터
 *       - in: query
 *         name: search
 *         schema:
 *           type: string
 *         description: 검색어
 *     responses:
 *       200:
 *         description: 사용자 목록 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     users:
 *                       type: array
 *                       items:
 *                         $ref: '#/components/schemas/User'
 *                     total:
 *                       type: integer
 *                     page:
 *                       type: integer
 *                     limit:
 *                       type: integer
 */

/**
 * @swagger
 * /api/v1/admin/users/{userId}:
 *   get:
 *     operationId: getUser
 *     summary: 사용자 상세 정보 조회
 *     description: 사용자 ID 또는 username으로 사용자의 상세 정보를 조회합니다. (관리자 전용)
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: userId
 *         required: true
 *         schema:
 *           oneOf:
 *             - type: integer
 *               description: 사용자 ID (숫자)
 *             - type: string
 *               description: 사용자 이름 (username)
 *         description: 사용자 ID(숫자) 또는 username
 *     responses:
 *       200:
 *         description: 사용자 정보 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     id:
 *                       type: integer
 *                       description: 사용자 ID
 *                     username:
 *                       type: string
 *                       description: 사용자 아이디
 *                     email:
 *                       type: string
 *                       format: email
 *                       description: 이메일
 *                     name:
 *                       type: string
 *                       description: 사용자 이름
 *                     role:
 *                       type: string
 *                       description: 사용자 역할
 *                     status:
 *                       type: string
 *                       description: 계정 상태
 *                     created_at:
 *                       type: string
 *                       format: date-time
 *                       description: 계정 생성일
 *                     last_login:
 *                       type: string
 *                       format: date-time
 *                       description: 마지막 로그인 시간
 *                     apiKeys:
 *                       type: array
 *                       description: 사용자의 활성화된 API 키 목록
 *                       items:
 *                         type: object
 *                         properties:
 *                           id:
 *                             type: integer
 *                             description: API 키 ID
 *                           name:
 *                             type: string
 *                             description: API 키 이름
 *                           key:
 *                             type: string
 *                             description: API 키 값
 *                           status:
 *                             type: string
 *                             enum: [active, inactive]
 *                             description: API 키 상태
 *                           created_at:
 *                             type: string
 *                             format: date-time
 *                             description: 생성일
 *                           last_used:
 *                             type: string
 *                             format: date-time
 *                             description: 마지막 사용 시간
 *                           permissions:
 *                             type: object
 *                             description: API 키 권한
 *       400:
 *         description: 잘못된 요청
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       404:
 *         description: 사용자를 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/admin/users/{userId}/status:
 *   put:
 *     operationId: updateUserStatus
 *     summary: 사용자 상태 변경
 *     description: 사용자 ID 또는 username으로 사용자의 상태를 변경합니다. (관리자 전용)
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: userId
 *         required: true
 *         schema:
 *           oneOf:
 *             - type: integer
 *               description: 사용자 ID (숫자)
 *             - type: string
 *               description: 사용자 이름 (username)
 *         description: 사용자 ID(숫자) 또는 username
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - status
 *             properties:
 *               status:
 *                 type: string
 *                 enum: [active, inactive]
 *                 description: 변경할 상태
 *     responses:
 *       200:
 *         description: 사용자 상태 변경 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     user:
 *                       $ref: '#/components/schemas/User'
 *       400:
 *         description: 잘못된 요청 (자기 자신의 상태를 변경하려는 경우 등)
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       404:
 *         description: 사용자를 찾을 수 없음
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/admin/users/{userId}/role:
 *   post:
 *     operationId: assignUserRole
 *     summary: 사용자 역할 할당
 *     description: 특정 사용자에게 역할을 할당합니다. (관리자 전용)
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: userId
 *         required: true
 *         schema:
 *           type: string
 *         description: 사용자 ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - role
 *             properties:
 *               role:
 *                 type: string
 *                 enum: [user, admin]
 *                 description: 할당할 역할
 *     responses:
 *       200:
 *         description: 역할 할당 성공
 */

/**
 * @swagger
 * /api/v1/admin/api-keys:
 *   post:
 *     summary: 관리자 - 특정 사용자의 API 키 재발급
 *     description: 관리자가 지정한 userId(인덱스넘버)의 기존 API 키를 모두 비활성화하고 새 API 키를 발급합니다.
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - userId
 *             properties:
 *               userId:
 *                 type: string
 *                 description: 재발급할 사용자 ID
 *     responses:
 *       200:
 *         description: 새로 발급된 API 키
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     key:
 *                       type: string
 *                       description: 새로 발급된 API 키 값
 *       400:
 *         description: 잘못된 요청
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

/**
 * @swagger
 * /api/v1/api-keys:
 *   post:
 *     summary: API 키 생성
 *     description: 새로운 API 키를 생성합니다.
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/CreateApiKeyRequest'
 *     responses:
 *       200:
 *         description: API 키 생성 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   $ref: '#/components/schemas/ApiKey'
 *   get:
 *     summary: API 키 목록 조회
 *     description: 현재 로그인한 사용자의 API 키 목록을 조회합니다.
 *     tags: [Auth]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: API 키 목록 조회 성공
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: array
 *                   items:
 *                     type: object
 *                     properties:
 *                       id:
 *                         type: integer
 *                         description: API 키 ID
 *                       user_id:
 *                         type: integer
 *                         description: 사용자 ID
 *                       key_value:
 *                         type: string
 *                         description: API 키 값
 *                       name:
 *                         type: string
 *                         description: API 키 이름
 *                       status:
 *                         type: string
 *                         enum: [active, inactive]
 *                         description: API 키 상태
 *                       last_used:
 *                         type: string
 *                         format: date-time
 *                         nullable: true
 *                         description: 마지막 사용 시간
 *                       created_at:
 *                         type: string
 *                         format: date-time
 *                         description: 생성 시간
 *                       expires_at:
 *                         type: string
 *                         format: date-time
 *                         nullable: true
 *                         description: 만료 시간
 *                       permissions:
 *                         type: object
 *                         nullable: true
 *                         description: API 키 권한
 *                       allowed_ips:
 *                         type: array
 *                         items:
 *                           type: string
 *                         nullable: true
 *                         description: 허용된 IP 주소 목록
 *                       rate_limit:
 *                         type: integer
 *                         description: API 호출 제한 횟수
 *       401:
 *         description: 인증 실패
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

export const authSchemas = {
    User: {
        type: 'object',
        properties: {
            id: {
                type: 'string',
                description: '사용자 ID'
            },
            username: {
                type: 'string',
                description: '사용자 아이디'
            },
            email: {
                type: 'string',
                format: 'email',
                description: '이메일 주소'
            },
            name: {
                type: 'string',
                description: '사용자 이름'
            },
            role: {
                type: 'string',
                enum: ['user', 'admin'],
                description: '사용자 역할'
            },
            status: {
                type: 'string',
                enum: ['active', 'inactive', 'suspended'],
                description: '사용자 상태'
            },
            created_at: {
                type: 'string',
                format: 'date-time',
                description: '생성 시간'
            },
            updated_at: {
                type: 'string',
                format: 'date-time',
                description: '수정 시간'
            }
        }
    },
    LoginRequest: {
        type: 'object',
        required: ['username', 'password'],
        properties: {
            username: {
                type: 'string',
                description: '사용자 아이디'
            },
            password: {
                type: 'string',
                format: 'password',
                description: '비밀번호'
            }
        }
    },
    RegisterRequest: {
        type: 'object',
        required: ['username', 'password', 'email'],
        properties: {
            username: {
                type: 'string',
                description: '사용자 아이디'
            },
            password: {
                type: 'string',
                format: 'password',
                description: '비밀번호'
            },
            email: {
                type: 'string',
                format: 'email',
                description: '이메일'
            },
            name: {
                type: 'string',
                description: '사용자 이름'
            },
            role: {
                type: 'string',
                description: '사용자 역할',
                default: 'user'
            }
        }
    },
    LoginResponse: {
        type: 'object',
        properties: {
            success: {
                type: 'boolean',
                description: '성공 여부'
            },
            data: {
                type: 'object',
                properties: {
                    user: {
                        type: 'object',
                        properties: {
                            id: {
                                type: 'integer',
                                description: '사용자 ID'
                            },
                            username: {
                                type: 'string',
                                description: '사용자 아이디'
                            },
                            email: {
                                type: 'string',
                                format: 'email',
                                description: '이메일'
                            },
                            name: {
                                type: 'string',
                                description: '사용자 이름'
                            },
                            role: {
                                type: 'string',
                                description: '사용자 역할'
                            },
                            created_at: {
                                type: 'string',
                                format: 'date-time',
                                description: '계정 생성일'
                            },
                            last_login: {
                                type: 'string',
                                format: 'date-time',
                                description: '마지막 로그인 시간'
                            },
                            status: {
                                type: 'string',
                                description: '계정 상태'
                            }
                        }
                    },
                    token: {
                        type: 'string',
                        description: 'JWT 토큰'
                    },
                    apiKeys: {
                        type: 'array',
                        description: '사용자의 활성화된 API 키 목록',
                        items: {
                            type: 'object',
                            properties: {
                                id: {
                                    type: 'integer',
                                    description: 'API 키 ID'
                                },
                                name: {
                                    type: 'string',
                                    description: 'API 키 이름'
                                },
                                key: {
                                    type: 'string',
                                    description: 'API 키 값'
                                },
                                status: {
                                    type: 'string',
                                    enum: ['active', 'inactive'],
                                    description: 'API 키 상태'
                                },
                                created_at: {
                                    type: 'string',
                                    format: 'date-time',
                                    description: '생성일'
                                },
                                last_used: {
                                    type: 'string',
                                    format: 'date-time',
                                    description: '마지막 사용 시간'
                                },
                                permissions: {
                                    type: 'object',
                                    description: 'API 키 권한',
                                    properties: {
                                        users: {
                                            type: 'object',
                                            properties: {
                                                read: {
                                                    type: 'boolean',
                                                    description: '사용자 조회 권한'
                                                },
                                                write: {
                                                    type: 'boolean',
                                                    description: '사용자 수정 권한'
                                                },
                                                manage: {
                                                    type: 'boolean',
                                                    description: '사용자 관리 권한'
                                                }
                                            }
                                        },
                                        reports: {
                                            type: 'object',
                                            properties: {
                                                read: {
                                                    type: 'boolean',
                                                    description: '리포트 조회 권한'
                                                },
                                                write: {
                                                    type: 'boolean',
                                                    description: '리포트 작성 권한'
                                                },
                                                share: {
                                                    type: 'boolean',
                                                    description: '리포트 공유 권한'
                                                },
                                                delete: {
                                                    type: 'boolean',
                                                    description: '리포트 삭제 권한'
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            },
            error: {
                type: 'object',
                nullable: true,
                properties: {
                    message: {
                        type: 'string',
                        description: '에러 메시지'
                    },
                    code: {
                        type: 'string',
                        description: '에러 코드'
                    }
                }
            },
            timestamp: {
                type: 'string',
                format: 'date-time',
                description: '응답 시간'
            }
        }
    },
    Error: {
        type: 'object',
        properties: {
            message: {
                type: 'string',
                description: '에러 메시지'
            }
        }
    },
    Activity: {
        type: 'object',
        properties: {
            id: {
                type: 'string',
                description: '활동 ID'
            },
            user_id: {
                type: 'string',
                description: '사용자 ID'
            },
            action: {
                type: 'string',
                description: '수행한 작업'
            },
            details: {
                type: 'object',
                description: '작업 상세 정보'
            },
            ip: {
                type: 'string',
                description: 'IP 주소'
            },
            user_agent: {
                type: 'string',
                description: '사용자 에이전트'
            },
            created_at: {
                type: 'string',
                format: 'date-time',
                description: '활동 시간'
            }
        }
    },
    ApiKey: {
        type: 'object',
        properties: {
            id: {
                type: 'string',
                description: 'API 키 ID'
            },
            name: {
                type: 'string',
                description: 'API 키 이름'
            },
            key: {
                type: 'string',
                description: 'API 키 값'
            },
            status: {
                type: 'string',
                enum: ['active', 'inactive'],
                description: 'API 키 상태'
            },
            created_at: {
                type: 'string',
                format: 'date-time',
                description: '생성일'
            },
            expires_at: {
                type: 'string',
                format: 'date-time',
                description: '만료일'
            }
        }
    }
}; 