import { NextRequest, NextResponse } from 'next/server';
import pg from 'pg';
import { getDatabaseConfig } from '@/config/serverConfig';

const { Pool } = pg;

// 데이터베이스 연결
function getDbConnection() {
  const config = getDatabaseConfig();
  return new Pool({
    host: config.host,
    port: config.port,
    user: config.user,
    password: config.password,
    database: config.name,
    max: 5
  });
}

// 기존 builtin-tools.js와 동일한 판정 기준
const HEALTH_EVALUATION_CRITERIA = {
  bloodPressure: {
    systolic: 140,    // mmHg 이상 시 해당(1)
    diastolic: 90     // mmHg 이상 시 해당(1)
  },
  lipid: {
    triglycerides: 150,  // mg/dl 이상 시 해당(1)
    hdl: 40,            // mg/dl 미만 시 해당(1)
    ldl: 140            // mg/dl 이상 시 해당(1)
  },
  bloodSugar: {
    general: 100,       // mg/dl 이상 시 해당(1)
    fasting: 100,       // mg/dl 이상 시 해당(1)
    hba1c: 5.6          // % 이상 시 해당(1)
  },
  obesity: {
    bmi: 25,           // kg/m² 이상 시 해당(1)
    waist: { male: 85, female: 90 }  // cm 이상 시 해당(1)
  },
  urineProtein: {
    // ー(음성)→1, ±→3, ＋→5, ＋＋→7, ＋＋＋→9
    codes: { '－': 1, '±': 3, '＋': 5, '＋＋': 7, '＋＋＋': 9 }
  }
};

// 숫자 파싱 헬퍼 함수
function parseNumber(value: any): number | null {
  if (value === null || value === undefined || value === '' || value === '-') {
    return null;
  }
  const num = parseFloat(String(value));
  return isNaN(num) ? null : num;
}

// 환자 데이터 분석 함수
function analyzePatientHealthData(rawData: any): any {
  const analysis = {
    bloodPressureScore: 3, // 기본값: 비해당
    cholesterolScore: 3,
    hba1cReceived: 3,
    bloodSugarScore: 3,
    obesityScore: 3,
    urineProteinScore: 1,
    pastHistoryTreatment: '',
    currentHistoryTreatment: '',
    finalJudgment: 3,
    hasValidData: false
  };

  try {
    // 1. 혈압 판정
    const systolicBp = parseNumber(rawData['수축기혈압평균']) || parseNumber(rawData['수축기혈압']) || parseNumber(rawData['systolic_bp']);
    const diastolicBp = parseNumber(rawData['확장기혈압평균']) || parseNumber(rawData['확장기혈압']) || parseNumber(rawData['diastolic_bp']);
    
    if (systolicBp !== null && diastolicBp !== null) {
      if (systolicBp >= HEALTH_EVALUATION_CRITERIA.bloodPressure.systolic || 
          diastolicBp >= HEALTH_EVALUATION_CRITERIA.bloodPressure.diastolic) {
        analysis.bloodPressureScore = 1; // 해당
      }
      analysis.hasValidData = true;
    }

    // 2. 혈중지질 판정
    const hdl = parseNumber(rawData['HDL']) || parseNumber(rawData['hdl']);
    const ldl = parseNumber(rawData['LDL']) || parseNumber(rawData['ldl']);
    const triglycerides = parseNumber(rawData['중성지방']) || parseNumber(rawData['TG']) || parseNumber(rawData['triglycerides']);
    
    if (hdl !== null || ldl !== null || triglycerides !== null) {
      if ((hdl !== null && hdl < HEALTH_EVALUATION_CRITERIA.lipid.hdl) ||
          (ldl !== null && ldl >= HEALTH_EVALUATION_CRITERIA.lipid.ldl) ||
          (triglycerides !== null && triglycerides >= HEALTH_EVALUATION_CRITERIA.lipid.triglycerides)) {
        analysis.cholesterolScore = 1; // 해당
      }
      analysis.hasValidData = true;
    }

    // 3. HbA1c 수진 여부 판정
    const hba1c = parseNumber(rawData['HbA1c']) || parseNumber(rawData['당화혈색소']) || parseNumber(rawData['hba1c']);
    if (hba1c !== null && hba1c > 0) {
      analysis.hba1cReceived = 1; // 수진
      analysis.hasValidData = true;
    }

    // 4. 혈당 판정
    const bloodSugar = parseNumber(rawData['혈당']) || parseNumber(rawData['공복혈당']) || parseNumber(rawData['FBS']) || parseNumber(rawData['glucose']);
    if (bloodSugar !== null) {
      if (bloodSugar >= HEALTH_EVALUATION_CRITERIA.bloodSugar.fasting) {
        analysis.bloodSugarScore = 1; // 해당
      }
      analysis.hasValidData = true;
    }

    // HbA1c로 혈당 판정 보완
    if (hba1c !== null) {
      if (hba1c >= HEALTH_EVALUATION_CRITERIA.bloodSugar.hba1c) {
        analysis.bloodSugarScore = 1; // 해당
      }
      analysis.hasValidData = true;
    }

    // 5. 비만 판정
    const bmi = parseNumber(rawData['BMI']) || parseNumber(rawData['bmi']) || parseNumber(rawData['체질량지수']);
    const waist = parseNumber(rawData['복위']) || parseNumber(rawData['허리둘레']) || parseNumber(rawData['waist']);
    const gender = parseNumber(rawData['성별']) || parseNumber(rawData['gender']);
    
    if (bmi !== null) {
      if (bmi >= HEALTH_EVALUATION_CRITERIA.obesity.bmi) {
        analysis.obesityScore = 1; // 해당
      }
      analysis.hasValidData = true;
    }
    
    if (waist !== null && gender !== null) {
      const waistThreshold = gender === 1 ? 
        HEALTH_EVALUATION_CRITERIA.obesity.waist.male : 
        HEALTH_EVALUATION_CRITERIA.obesity.waist.female;
      if (waist >= waistThreshold) {
        analysis.obesityScore = 1; // 해당
      }
      analysis.hasValidData = true;
    }

    // 6. 요단백 판정
    const urineProtein = String(rawData['요단백'] || rawData['urine_protein'] || rawData['蛋白尿'] || '－');
    if (urineProtein && urineProtein !== '') {
      analysis.urineProteinScore = HEALTH_EVALUATION_CRITERIA.urineProtein.codes[urineProtein] || 1;
      analysis.hasValidData = true;
    }

    // 7. 기왕력/현병력 처리
    analysis.pastHistoryTreatment = String(rawData['기왕력'] || rawData['과거병력'] || rawData['past_history'] || '');
    analysis.currentHistoryTreatment = String(rawData['현병력'] || rawData['현재병력'] || rawData['current_history'] || '');

    // 8. 최종 판정 계산
    const riskFactors = [
      analysis.bloodPressureScore === 1,
      analysis.cholesterolScore === 1,
      analysis.bloodSugarScore === 1,
      analysis.obesityScore === 1
    ].filter(Boolean).length;

    if (riskFactors >= 3) {
      analysis.finalJudgment = 1; // 고위험군
    } else if (riskFactors >= 1) {
      analysis.finalJudgment = 2; // 중위험군
    } else {
      analysis.finalJudgment = 3; // 저위험군
    }

    return analysis;

  } catch (error) {
    console.error('환자 데이터 분석 오류:', error);
    return {
      ...analysis,
      error: error instanceof Error ? error.message : '분석 중 오류 발생',
      hasValidData: false
    };
  }
}

// 배치 분석 백그라운드 프로세스
async function processBatchHealthAnalysis(jobId: string) {
  const dbManager = getDbConnection();

  try {
    // 작업 상태를 처리 중으로 변경 및 총 레코드 수 조회
    const jobResult = await dbManager.query(`
      UPDATE health_batch_jobs
      SET status = 'processing', updated_at = CURRENT_TIMESTAMP
      WHERE id = $1
      RETURNING total_records
    `, [jobId]);

    const totalRecords = jobResult.rows[0]?.total_records || 0;

    // 분석할 환자 목록 가져오기
    const patientsResult = await dbManager.query(`
      SELECT id, patient_name, raw_data
      FROM health_batch_patients
      WHERE job_id = $1 AND analysis_status = 'pending'
      ORDER BY row_number
    `, [jobId]);

    const patients = patientsResult.rows;
    let processed = 0;
    let successful = 0;
    let failed = 0;

    for (const patient of patients) {
      try {
        // 환자 상태를 처리 중으로 변경
        await dbManager.query(`
          UPDATE health_batch_patients 
          SET analysis_status = 'processing', updated_at = CURRENT_TIMESTAMP
          WHERE id = $1
        `, [patient.id]);

        // 작업 진행률 업데이트
        const progressPercentage = Math.round((processed / totalRecords) * 100);
        await dbManager.query(`
          UPDATE health_batch_jobs
          SET processed_records = $1, current_patient = $2,
              progress = $3,
              updated_at = CURRENT_TIMESTAMP
          WHERE id = $4
        `, [processed, patient.patient_name, progressPercentage, jobId]);

        // AI 분석 수행
        const rawData = patient.raw_data;
        const analysisResult = analyzePatientHealthData(rawData);

        if (analysisResult.hasValidData) {
          // 분석 성공 - 결과 저장
          await dbManager.query(`
            UPDATE health_batch_patients 
            SET 
              blood_pressure_score = $1,
              cholesterol_score = $2,
              hba1c_received = $3,
              blood_sugar_score = $4,
              obesity_score = $5,
              urine_protein_score = $6,
              past_history_treatment = $7,
              current_history_treatment = $8,
              final_judgment = $9,
              analysis_status = 'completed',
              analyzed_at = CURRENT_TIMESTAMP,
              updated_at = CURRENT_TIMESTAMP
            WHERE id = $10
          `, [
            analysisResult.bloodPressureScore,
            analysisResult.cholesterolScore,
            analysisResult.hba1cReceived,
            analysisResult.bloodSugarScore,
            analysisResult.obesityScore,
            analysisResult.urineProteinScore,
            analysisResult.pastHistoryTreatment,
            analysisResult.currentHistoryTreatment,
            analysisResult.finalJudgment,
            patient.id
          ]);
          successful++;
        } else {
          // 분석 실패 - 오류 저장
          await dbManager.query(`
            UPDATE health_batch_patients 
            SET 
              analysis_status = 'failed',
              analysis_error = $1,
              updated_at = CURRENT_TIMESTAMP
            WHERE id = $2
          `, [analysisResult.error || '유효한 데이터가 없습니다.', patient.id]);
          failed++;
        }

        processed++;

        // 1초 대기 (시뮬레이션)
        await new Promise(resolve => setTimeout(resolve, 1000));

      } catch (error) {
        console.error(`환자 ${patient.patient_name} 분석 오류:`, error);
        
        // 환자별 오류 저장
        await dbManager.query(`
          UPDATE health_batch_patients 
          SET 
            analysis_status = 'failed',
            analysis_error = $1,
            updated_at = CURRENT_TIMESTAMP
          WHERE id = $2
        `, [error instanceof Error ? error.message : '분석 실패', patient.id]);
        
        failed++;
        processed++;
      }
    }

    // 작업 완료 처리
    await dbManager.query(`
      UPDATE health_batch_jobs 
      SET 
        status = 'completed',
        processed_records = $1,
        successful_records = $2,
        failed_records = $3,
        progress = 100,
        completed_at = CURRENT_TIMESTAMP,
        updated_at = CURRENT_TIMESTAMP
      WHERE id = $4
    `, [processed, successful, failed, jobId]);

    console.log(`배치 분석 완료: ${jobId}, 성공: ${successful}, 실패: ${failed}`);

  } catch (error) {
    console.error('배치 분석 오류:', error);
    
    // 작업 실패 처리
    await dbManager.query(`
      UPDATE health_batch_jobs 
      SET 
        status = 'failed',
        error_message = $1,
        updated_at = CURRENT_TIMESTAMP
      WHERE id = $2
    `, [error instanceof Error ? error.message : '배치 분석 실패', jobId]);
  }
}

export async function POST(request: NextRequest) {
  try {
    const { jobId } = await request.json();
    
    if (!jobId) {
      return NextResponse.json({
        success: false,
        error: '작업 ID가 필요합니다.'
      }, { status: 400 });
    }

    const dbManager = getDbConnection();

    // 작업 존재 여부 확인
    const jobResult = await dbManager.query(`
      SELECT id, status, total_records 
      FROM health_batch_jobs 
      WHERE id = $1
    `, [jobId]);

    if (jobResult.rows.length === 0) {
      return NextResponse.json({
        success: false,
        error: '작업을 찾을 수 없습니다.'
      }, { status: 404 });
    }

    const job = jobResult.rows[0];
    
    if (job.status !== 'pending') {
      return NextResponse.json({
        success: false,
        error: '이미 처리된 작업입니다.'
      }, { status: 400 });
    }

    // 백그라운드에서 분석 시작
    processBatchHealthAnalysis(jobId).catch(error => {
      console.error('백그라운드 분석 오류:', error);
    });

    return NextResponse.json({
      success: true,
      data: {
        jobId,
        message: '배치 분석이 시작되었습니다.',
        totalRecords: job.total_records
      }
    });

  } catch (error) {
    console.error('배치 분석 시작 오류:', error);
    return NextResponse.json({
      success: false,
      error: error instanceof Error ? error.message : '배치 분석 시작에 실패했습니다.'
    }, { status: 500 });
  }
}