import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../../middlewares';
import { pg } from '../../db';

// 어시스턴트 세션 타입 정의
interface AssistantSession {
  id: string;
  provider: string;
  model: string;
  parameters: {
    temperature: number;
  };
  history: Array<{
    role: string;
    content: string;
    files?: Array<{
      name: string;
      type: string;
      base64: string;
    }>;
  }>;
  type: string;
  title: string;
  createdAt: string;
  updatedAt: string;
}

// 세션 데이터 변환 함수
const transformSessionData = (dbSession: any): AssistantSession => {
  const sessionData = dbSession.data || {};
  
  return {
    id: dbSession.id,
    provider: dbSession.provider || sessionData.provider || 'ollama',
    model: dbSession.model || sessionData.model || 'hamonize:latest',
    parameters: sessionData.parameters || { temperature: 0.7 },
    history: sessionData.history || [],
    type: dbSession.type || 'assistant',
    title: dbSession.title || '새 대화',
    createdAt: dbSession.created_at,
    updatedAt: dbSession.updated_at
  };
};

// GET: 사용자의 어시스턴트 세션 목록 조회 또는 단일 세션 조회
export async function GET(req: NextRequest) {
  return withAnyAuth(req, async (req, user) => {
    try {
      const userId = user?.id;
      if (!userId) {
        return NextResponse.json(
          { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
          { status: 401 }
        );
      }

      // URL 파라미터에서 sessionId 확인
      const { searchParams } = new URL(req.url);
      const sessionId = searchParams.get('sessionId');

      if (sessionId) {
        // 단일 세션 조회
        const result = await pg.query(`
          SELECT 
            id, user_id, username, data, provider, model, 
            created_at, updated_at, title, message_count, 
            last_message, status, type
          FROM sessions 
          WHERE id = $1 AND user_id = $2 AND type = 'assistant' AND status = 'active'
        `, [sessionId, userId]);

        if (result.rows.length === 0) {
          return NextResponse.json(
            { success: false, error: '세션을 찾을 수 없습니다.' },
            { status: 404 }
          );
        }

        const sessionData = transformSessionData(result.rows[0]);
        // console.log(`사용자 ${user.username} 어시스턴트 세션 조회 완료: ${sessionId}`);
        return NextResponse.json({ success: true, data: sessionData });
      } else {
        // 모든 세션 목록 조회
        const result = await pg.query(`
          SELECT 
            id, user_id, username, data, provider, model, 
            created_at, updated_at, title, message_count, 
            last_message, status, type
          FROM sessions 
          WHERE user_id = $1 AND type = 'assistant' AND status = 'active'
          ORDER BY updated_at DESC
        `, [userId]);

        const sessions: AssistantSession[] = [];
        
        for (const dbSession of result.rows) {
          try {
            const sessionData = dbSession.data || {};
            
            // 실제 대화가 있는 세션만 포함 (사용자 메시지와 AI 응답이 모두 있어야 함)
            if (sessionData.history && sessionData.history.length > 0) {
              const userMessages = sessionData.history.filter((msg: any) => msg.role === 'user');
              const assistantMessages = sessionData.history.filter((msg: any) => msg.role === 'assistant');
              
              if (userMessages.length > 0 && assistantMessages.length > 0) {
                sessions.push(transformSessionData(dbSession));
              }
            }
          } catch (error) {
            console.error(`세션 데이터 변환 오류: ${dbSession.id}`, error);
            // 손상된 세션은 건너뛰기
            continue;
          }
        }

        // console.log(`사용자 ${user.username} 어시스턴트 세션 로드 완료: ${sessions.length}개 세션`);
        return NextResponse.json({ success: true, data: sessions });
      }
      
    } catch (error) {
      console.error('세션 조회 오류:', error);
      return NextResponse.json(
        { success: false, error: '세션 조회 중 오류가 발생했습니다.' },
        { status: 500 }
      );
    }
  });
}

// POST: 새 어시스턴트 세션 생성
export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user) => {
    try {
      const userId = user?.id;
      const username = user?.username;
      if (!userId || !username) {
        return NextResponse.json(
          { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
          { status: 401 }
        );
      }

      const { provider, model, temperature } = await req.json();
      
      if (!provider || !model) {
        return NextResponse.json(
          { success: false, error: '프로바이더와 모델이 필요합니다.' },
          { status: 400 }
        );
      }

      const sessionId = `assistant-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      
      const sessionData = {
        parameters: {
          temperature: temperature || 0.7,
        },
        history: []
      };

      // 데이터베이스에 세션 저장
      await pg.query(`
        INSERT INTO sessions (
          id, user_id, username, data, provider, model, 
          title, type, status, message_count
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
      `, [
        sessionId,
        userId,
        username,
        JSON.stringify(sessionData),
        provider,
        model,
        '새 대화',
        'assistant',
        'active',
        0
      ]);
      
    //   console.log(`새 어시스턴트 세션 생성 완료: ${sessionId} (사용자: ${username})`);
      return NextResponse.json({ success: true, data: { sessionId } });
      
    } catch (error) {
      console.error('세션 생성 오류:', error);
      return NextResponse.json(
        { success: false, error: '세션 생성 중 오류가 발생했습니다.' },
        { status: 500 }
      );
    }
  });
}

// PUT: 어시스턴트 세션 업데이트 (히스토리 추가 등)
export async function PUT(req: NextRequest) {
  return withAnyAuth(req, async (req, user) => {
    try {
      const userId = user?.id;
      const username = user?.username;
      if (!userId || !username) {
        return NextResponse.json(
          { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
          { status: 401 }
        );
      }

      const { sessionId, newMessage, title } = await req.json();
      
      if (!sessionId) {
        return NextResponse.json(
          { success: false, error: '세션 ID가 필요합니다.' },
          { status: 400 }
        );
      }

      // 세션 존재 확인 및 권한 검증
      const sessionResult = await pg.query(`
        SELECT id, data, title, message_count 
        FROM sessions 
        WHERE id = $1 AND user_id = $2 AND type = 'assistant'
      `, [sessionId, userId]);

      if (sessionResult.rows.length === 0) {
        return NextResponse.json(
          { success: false, error: '세션을 찾을 수 없습니다.' },
          { status: 404 }
        );
      }

      const session = sessionResult.rows[0];
      const sessionData = session.data || {};

      // 세션 업데이트
      if (newMessage) {
        if (!sessionData.history) {
          sessionData.history = [];
        }
        sessionData.history.push(newMessage);
      }
      
      let newTitle = session.title;
      if (title) {
        newTitle = title;
      }
      
      const messageCount = sessionData.history ? sessionData.history.length : 0;
      const lastMessage = sessionData.history && sessionData.history.length > 0 
        ? sessionData.history[sessionData.history.length - 1].content.substring(0, 100)
        : null;

      // 데이터베이스 업데이트
      await pg.query(`
        UPDATE sessions 
        SET data = $1, title = $2, message_count = $3, last_message = $4, updated_at = CURRENT_TIMESTAMP
        WHERE id = $5 AND user_id = $6
      `, [
        JSON.stringify(sessionData),
        newTitle,
        messageCount,
        lastMessage,
        sessionId,
        userId
      ]);
      
      // console.log(`어시스턴트 세션 업데이트 완료: ${sessionId} (사용자: ${username})`);
      return NextResponse.json({ success: true });
      
    } catch (error) {
      console.error('세션 업데이트 오류:', error);
      return NextResponse.json(
        { success: false, error: '세션 업데이트 중 오류가 발생했습니다.' },
        { status: 500 }
      );
    }
  });
}

// DELETE: 어시스턴트 세션 삭제
export async function DELETE(req: NextRequest) {
  return withAnyAuth(req, async (req, user) => {
    try {
      const userId = user?.id;
      const username = user?.username;
      if (!userId || !username) {
        return NextResponse.json(
          { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
          { status: 401 }
        );
      }

      const { searchParams } = new URL(req.url);
      const sessionId = searchParams.get('sessionId');
      
      if (!sessionId) {
        return NextResponse.json(
          { success: false, error: '세션 ID가 필요합니다.' },
          { status: 400 }
        );
      }

      // 세션 삭제 (소프트 삭제 - status를 'deleted'로 변경)
      const result = await pg.query(`
        UPDATE sessions 
        SET status = 'deleted', updated_at = CURRENT_TIMESTAMP
        WHERE id = $1 AND user_id = $2 AND type = 'assistant'
        RETURNING id
      `, [sessionId, userId]);
      
      if (result.rows.length === 0) {
        return NextResponse.json(
          { success: false, error: '세션을 찾을 수 없습니다.' },
          { status: 404 }
        );
      }

      // console.log(`어시스턴트 세션 삭제 완료: ${sessionId} (사용자: ${username})`);
      return NextResponse.json({ success: true });
      
    } catch (error) {
      console.error('세션 삭제 오류:', error);
      return NextResponse.json(
        { success: false, error: '세션 삭제 중 오류가 발생했습니다.' },
        { status: 500 }
      );
    }
  });
} 