import { NextRequest, NextResponse } from 'next/server';

// 기본 도구들 정의
const tools = [
  {
    type: "function",
    function: {
      name: "get_current_time",
      description: "현재 시간을 가져옵니다",
      parameters: {
        type: "object",
        properties: {
          timezone: {
            type: "string", 
            description: "시간대 (기본값: Asia/Seoul)"
          }
        }
      }
    }
  },
  {
    type: "function",
    function: {
      name: "calculate",
      description: "수학 계산을 수행합니다",
      parameters: {
        type: "object",
        properties: {
          expression: {
            type: "string",
            description: "계산할 수식 (예: 2+3*4)"
          }
        },
        required: ["expression"]
      }
    }
  },
  {
    type: "function", 
    function: {
      name: "get_random_number",
      description: "랜덤 숫자를 생성합니다",
      parameters: {
        type: "object",
        properties: {
          min: {
            type: "number",
            description: "최소값 (기본값: 1)"
          },
          max: {
            type: "number", 
            description: "최대값 (기본값: 100)"
          }
        }
      }
    }
  }
];

// 도구 관련 오류인지 확인하는 함수
function isToolRelatedError(errorText: string): boolean {
  return errorText.includes('tools') || 
         errorText.includes('tool_choice') || 
         errorText.includes('function calling') || 
         errorText.includes('does not support tools');
}

// 도구 실행 함수
async function executeTool(toolCall: any) {
  const { name, arguments: args } = toolCall.function;
  let parsedArgs: any = {};
  
  try {
    parsedArgs = typeof args === 'string' ? JSON.parse(args) : args;
  } catch (error) {
    return { error: `인수 파싱 실패: ${error}` };
  }

  switch (name) {
    case 'get_current_time':
      const timezone = parsedArgs.timezone || 'Asia/Seoul';
      const now = new Date().toLocaleString('ko-KR', { 
        timeZone: timezone,
        year: 'numeric',
        month: '2-digit', 
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
      });
      return { result: `현재 시간 (${timezone}): ${now}` };

    case 'calculate':
      try {
        const expression = parsedArgs.expression;
        // 안전한 계산을 위해 eval 대신 간단한 계산기 구현
        const result = Function(`"use strict"; return (${expression})`)();
        return { result: `${expression} = ${result}` };
      } catch (error) {
        return { error: `계산 오류: ${error}` };
      }

    case 'get_random_number':
      const min = parsedArgs.min || 1;
      const max = parsedArgs.max || 100;
      const randomNum = Math.floor(Math.random() * (max - min + 1)) + min;
      return { result: `${min}과 ${max} 사이의 랜덤 숫자: ${randomNum}` };

    default:
      return { error: `알 수 없는 도구: ${name}` };
  }
}

export async function POST(request: NextRequest) {
  try {
    const { model, message, messages, files, enableTools } = await request.json();

    if (!model || !message) {
      return NextResponse.json(
        { error: '모델과 메시지가 필요합니다.' },
        { status: 400 }
      );
    }

    // 메시지 히스토리 구성
    const chatHistory = messages.map((msg: any) => ({
      role: msg.role,
      content: msg.content
    }));

    // 파일이 있는 경우 처리
    if (files && files.length > 0) {
      const imageFiles = files.filter((file: any) => file.type.startsWith('image/'));
      const textFiles = files.filter((file: any) => file.type.startsWith('text/'));
      const pdfFiles = files.filter((file: any) => file.type === 'application/pdf');
      
      // 텍스트 파일 내용 추출
      let additionalText = '';
      if (textFiles.length > 0) {
        for (const file of textFiles) {
          try {
            // base64 디코딩하여 텍스트 내용 추출
            const base64Content = file.data.split(',')[1];
            const textContent = Buffer.from(base64Content, 'base64').toString('utf-8');
            additionalText += `\n\n[파일: ${file.name}]\n${textContent}`;
          } catch (error) {
            console.error(`텍스트 파일 처리 오류: ${file.name}`, error);
            additionalText += `\n\n[파일: ${file.name}] - 텍스트 파일 처리 중 오류가 발생했습니다.`;
          }
        }
      }
      
      // PDF 파일 처리 (텍스트 내용 포함)
      if (pdfFiles.length > 0) {
        for (const file of pdfFiles) {
          if (file.extractedText) {
            additionalText += `\n\n[PDF 파일: ${file.name}]\n${file.extractedText}`;
          } else {
            additionalText += `\n\n[PDF 파일: ${file.name}] - PDF 텍스트 추출에 실패했습니다.`;
          }
        }
      }
      
      // 최종 메시지에 텍스트 내용 포함
      const finalMessage = message + additionalText;
      
      if (imageFiles.length > 0) {
        // 이미지와 텍스트 모두 포함
        const content: any[] = [
          {
            type: "text",
            text: finalMessage
          }
        ];
        
        // 이미지 추가
        imageFiles.forEach((file: any) => {
          content.push({
            type: "image_url",
            image_url: {
              url: file.data
            }
          });
        });
        
        chatHistory.push({
          role: 'user',
          content: content
        });
      } else {
        // 텍스트/PDF 파일만 있는 경우
        chatHistory.push({
          role: 'user',
          content: finalMessage
        });
      }
    } else {
      // 파일이 없는 경우
      chatHistory.push({
        role: 'user',
        content: message
      });
    }

    // 환경변수에서 서버 주소 가져오기
    const ollamaUrl = process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || process.env.OLLAMA_URL || 'http://localhost:11434';
    const vllmUrl = process.env.NEXT_PUBLIC_VLLM_SERVER || 'http://localhost:8000';

    // Ollama용 메시지 히스토리 (images 필드 사용)
    const ollamaChatHistory = chatHistory.map((msg: any) => {
      if (Array.isArray(msg.content)) {
        // OpenAI 형식을 Ollama 형식으로 변환
        const textContent = msg.content.find((c: any) => c.type === 'text')?.text || '';
        const imageUrls = msg.content.filter((c: any) => c.type === 'image_url').map((c: any) => c.image_url.url);
        
        const ollamaMsg: any = {
          role: msg.role,
          content: textContent
        };
        
        if (imageUrls.length > 0) {
          // Ollama API는 순수한 base64 문자열을 요구하므로 data: 접두사 제거
          ollamaMsg.images = imageUrls.map((url: string) => {
            if (url.startsWith('data:image/')) {
              return url.split(',')[1]; // data:image/jpeg;base64, 부분 제거
            }
            return url;
          });
        }
        
        return ollamaMsg;
      }
      return msg;
    });

    // 도구 사용이 활성화된 경우 요청에 도구 정보 추가
    const ollamaRequestBody: any = {
      model: model,
      messages: ollamaChatHistory,
      stream: true,
      options: {
        temperature: 0.7,
        top_p: 0.9,
        top_k: 40,
      }
    };

    if (enableTools) {
      ollamaRequestBody.tools = tools;
    }

    // 스트리밍 응답 생성
    const encoder = new TextEncoder();
    const stream = new ReadableStream({
      async start(controller) {
        try {
          console.log('Ollama 요청 시도:', { 
            url: `${ollamaUrl}/api/chat`, 
            model, 
            hasImages: ollamaRequestBody.messages.some((m: any) => m.images),
            imageCount: ollamaRequestBody.messages.filter((m: any) => m.images).length,
            messageStructure: ollamaRequestBody.messages.map((m: any) => ({
              role: m.role,
              hasImages: !!m.images,
              imageCount: m.images ? m.images.length : 0,
              contentLength: m.content ? m.content.length : 0
            }))
          });
          
          // Ollama 스트리밍 시도
          const response = await fetch(`${ollamaUrl}/api/chat`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(ollamaRequestBody),
          });

          console.log('Ollama 응답 상태:', response.status, response.statusText);

          if (!response.ok) {
            const ollamaErrorText = await response.text();
            console.log('Ollama 오류 응답:', ollamaErrorText);
            
            // 도구 지원하지 않는 모델인 경우 도구 없이 재시도
            if (enableTools && isToolRelatedError(ollamaErrorText)) {
              console.log('Ollama: 도구를 지원하지 않는 모델로 판단, 도구 없이 재시도');
              
              const withoutToolsRequest: any = {
                model: model,
                messages: ollamaChatHistory,
                stream: true,
                options: {
                  temperature: 0.7,
                  top_p: 0.9,
                  top_k: 40,
                }
              };
              // tools 필드 제거
              
              const withoutToolsResponse = await fetch(`${ollamaUrl}/api/chat`, {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify(withoutToolsRequest),
              });
              
              if (withoutToolsResponse.ok) {
                console.log('Ollama 도구 없는 요청 성공');
                const reader = withoutToolsResponse.body?.getReader();
                const decoder = new TextDecoder();
                
                // 도구 관련 오류가 발생한 경우에만 안내 메시지 전송
                controller.enqueue(encoder.encode(`data: ${JSON.stringify({ 
                  content: '⚠️ 이 모델은 도구 기능을 지원하지 않습니다. 일반 채팅으로 처리합니다.\n\n' 
                })}\n\n`));
                
                while (true) {
                  const { done, value } = await reader!.read();
                  if (done) {
                    controller.enqueue(encoder.encode(`data: [DONE]\n\n`));
                    break;
                  }
                  
                  const chunk = decoder.decode(value);
                  const lines = chunk.split('\n');
                  
                  for (const line of lines) {
                    if (line.trim()) {
                      try {
                        const parsed = JSON.parse(line);
                        if (parsed.message && parsed.message.content) {
                          controller.enqueue(encoder.encode(`data: ${JSON.stringify({
                            content: parsed.message.content
                          })}\n\n`));
                        }
                        
                        if (parsed.done) {
                          controller.enqueue(encoder.encode(`data: [DONE]\n\n`));
                          return;
                        }
                      } catch (e) {
                        console.error('Ollama 스트리밍 파싱 오류:', e);
                      }
                    }
                  }
                }
                return;
              }
            }
            
            // 이미지가 포함된 경우 특정 오류에서만 fallback 처리
            if (files && files.length > 0 && (
              ollamaErrorText.includes('does not support images') ||
              ollamaErrorText.includes('vision') ||
              ollamaErrorText.includes('multimodal') ||
              response.status === 400
            )) {
              console.log('Ollama: 이미지 지원하지 않는 모델로 판단, 텍스트만으로 재시도');
              
              const textOnlyOllamaRequest: any = {
                model: model,
                messages: [{ role: 'user', content: message }],
                stream: true,
                options: {
                  temperature: 0.7,
                  num_predict: 2048,
                }
              };
              
              if (enableTools) {
                textOnlyOllamaRequest.tools = tools;
              }
              
              const textOnlyOllamaResponse = await fetch(`${ollamaUrl}/api/chat`, {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify(textOnlyOllamaRequest),
              });
              
              if (textOnlyOllamaResponse.ok) {
                console.log('Ollama 텍스트 전용 요청 성공');
                const reader = textOnlyOllamaResponse.body?.getReader();
                const decoder = new TextDecoder();
                const encoder = new TextEncoder();
                
                // 이미지 처리 불가 안내 메시지 전송
                controller.enqueue(encoder.encode(`data: ${JSON.stringify({ 
                  content: '⚠️ 이 모델은 이미지 처리를 지원하지 않습니다. 텍스트만 처리합니다.\n\n' 
                })}\n\n`));
                
                while (true) {
                  const { done, value } = await reader!.read();
                  if (done) {
                    controller.enqueue(encoder.encode(`data: [DONE]\n\n`));
                    break;
                  }
                  
                  const chunk = decoder.decode(value);
                  const lines = chunk.split('\n');
                  
                  for (const line of lines) {
                    if (line.trim()) {
                      try {
                        const parsed = JSON.parse(line);
                        if (parsed.message && parsed.message.content) {
                          controller.enqueue(encoder.encode(`data: ${JSON.stringify({
                            content: parsed.message.content
                          })}\n\n`));
                        }
                        
                        if (parsed.done) {
                          controller.enqueue(encoder.encode(`data: [DONE]\n\n`));
                          return;
                        }
                      } catch (e) {
                        console.error('Ollama 스트리밍 파싱 오류:', e);
                      }
                    }
                  }
                }
                return;
              }
            }
            
            // VLLM 스트리밍 시도
            const vllmRequestBody: any = {
              model: model,
              messages: chatHistory,
              max_tokens: 2048,
              temperature: 0.7,
              stream: true,
            };

            if (enableTools) {
              vllmRequestBody.tools = tools;
              vllmRequestBody.tool_choice = "auto";
            }

            console.log('VLLM 요청 시도:', { url: `${vllmUrl}/v1/chat/completions`, model, hasImages: chatHistory.some((m: any) => Array.isArray(m.content)) });

            const vllmResponse = await fetch(`${vllmUrl}/v1/chat/completions`, {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify(vllmRequestBody),
            });

            console.log('VLLM 응답 상태:', vllmResponse.status, vllmResponse.statusText);

            if (!vllmResponse.ok) {
              const vllmErrorText = await vllmResponse.text();
              console.log('VLLM 오류 응답:', vllmErrorText);
              
              // 도구 지원하지 않는 모델인 경우 도구 없이 재시도
              if (enableTools && isToolRelatedError(vllmErrorText)) {
                console.log('VLLM: 도구를 지원하지 않는 모델로 판단, 도구 없이 재시도');
                
                const withoutToolsRequest = {
                  model: model,
                  messages: chatHistory,
                  max_tokens: 2048,
                  temperature: 0.7,
                  stream: true,
                };
                // tools와 tool_choice 필드 제거
                
                const withoutToolsResponse = await fetch(`${vllmUrl}/v1/chat/completions`, {
                  method: 'POST',
                  headers: {
                    'Content-Type': 'application/json',
                  },
                  body: JSON.stringify(withoutToolsRequest),
                });
                
                if (withoutToolsResponse.ok) {
                  console.log('VLLM 도구 없는 요청 성공');
                  const reader = withoutToolsResponse.body?.getReader();
                  const decoder = new TextDecoder();
                  
                  // 도구 관련 오류가 발생한 경우에만 안내 메시지 전송
                  controller.enqueue(encoder.encode(`data: ${JSON.stringify({ 
                    content: '⚠️ 이 모델은 도구 기능을 지원하지 않습니다. 일반 채팅으로 처리합니다.\n\n' 
                  })}\n\n`));
                  
                  while (true) {
                    const { done, value } = await reader!.read();
                    if (done) break;
                    
                    const chunk = decoder.decode(value);
                    const lines = chunk.split('\n');
                    
                    for (const line of lines) {
                      if (line.startsWith('data: ')) {
                        const data = line.slice(6);
                        if (data === '[DONE]') {
                          controller.enqueue(encoder.encode(`data: [DONE]\n\n`));
                          return;
                        }
                        
                        try {
                          const parsed = JSON.parse(data);
                          if (parsed.choices && parsed.choices[0] && parsed.choices[0].delta && parsed.choices[0].delta.content) {
                            controller.enqueue(encoder.encode(`data: ${JSON.stringify({
                              content: parsed.choices[0].delta.content
                            })}\n\n`));
                          }
                        } catch (e) {
                          console.error('VLLM 스트리밍 파싱 오류:', e);
                        }
                      }
                    }
                  }
                  return;
                }
              }
              
              // 이미지가 포함된 경우 특정 오류에서만 fallback 처리
              if (files && files.length > 0 && (
                vllmErrorText.includes('does not support images') ||
                vllmErrorText.includes('vision') ||
                vllmErrorText.includes('multimodal') ||
                vllmErrorText.includes('content_type') ||
                vllmResponse.status === 400 || vllmResponse.status === 422
              )) {
                console.log('VLLM: 이미지 지원하지 않는 모델로 판단, 텍스트만으로 재시도');
                
                // 텍스트만 포함된 간단한 요청
                const textOnlyRequest = {
                  model: model,
                  messages: [{ role: 'user', content: message }],
                  max_tokens: 2048,
                  temperature: 0.7,
                  stream: true,
                };
                
                const textOnlyResponse = await fetch(`${vllmUrl}/v1/chat/completions`, {
                  method: 'POST',
                  headers: {
                    'Content-Type': 'application/json',
                  },
                  body: JSON.stringify(textOnlyRequest),
                });
                
                if (textOnlyResponse.ok) {
                  console.log('VLLM 텍스트 전용 요청 성공');
                  const reader = textOnlyResponse.body?.getReader();
                  const decoder = new TextDecoder();
                  
                  // 이미지 처리 불가 안내 메시지 전송
                  const encoder = new TextEncoder();
                  controller.enqueue(encoder.encode(`data: ${JSON.stringify({ 
                    content: '⚠️ 이 모델은 이미지 처리를 지원하지 않습니다. 텍스트만 처리합니다.\n\n' 
                  })}\n\n`));
                  
                  while (true) {
                    const { done, value } = await reader!.read();
                    if (done) break;
                    
                    const chunk = decoder.decode(value);
                    const lines = chunk.split('\n');
                    
                    for (const line of lines) {
                      if (line.startsWith('data: ')) {
                        const data = line.slice(6);
                        if (data === '[DONE]') {
                          controller.enqueue(encoder.encode(`data: [DONE]\n\n`));
                          return;
                        }
                        
                        try {
                          const parsed = JSON.parse(data);
                          if (parsed.choices && parsed.choices[0] && parsed.choices[0].delta && parsed.choices[0].delta.content) {
                            controller.enqueue(encoder.encode(`data: ${JSON.stringify({
                              content: parsed.choices[0].delta.content
                            })}\n\n`));
                          }
                        } catch (e) {
                          console.error('VLLM 스트리밍 파싱 오류:', e);
                        }
                      }
                    }
                  }
                  return;
                }
              }
              
              throw new Error(`스트리밍 응답을 받을 수 없습니다. Ollama: ${response.status} ${ollamaErrorText.substring(0, 100)}, VLLM: ${vllmResponse.status} ${vllmErrorText.substring(0, 100)}`);
            }

            const reader = vllmResponse.body?.getReader();
            const decoder = new TextDecoder();
            let toolCalls: any[] = [];
            let isToolCallComplete = false;

            while (true) {
              const { done, value } = await reader!.read();
              if (done) break;

              const chunk = decoder.decode(value);
              const lines = chunk.split('\n');

              for (const line of lines) {
                if (line.startsWith('data: ')) {
                  const data = line.slice(6);
                  if (data === '[DONE]') {
                    // 도구 호출이 완료된 경우 결과 추가
                    if (enableTools && toolCalls.length > 0 && isToolCallComplete) {
                      for (const toolCall of toolCalls) {
                        const toolResult = await executeTool(toolCall);
                        if (toolResult.result) {
                          controller.enqueue(encoder.encode(`data: ${JSON.stringify({ content: `\n\n🔧 도구 실행 결과: ${toolResult.result}` })}\n\n`));
                        } else if (toolResult.error) {
                          controller.enqueue(encoder.encode(`data: ${JSON.stringify({ content: `\n\n❌ 도구 실행 오류: ${toolResult.error}` })}\n\n`));
                        }
                      }
                    }
                    controller.enqueue(encoder.encode('data: [DONE]\n\n'));
                    controller.close();
                    return;
                  }
                  try {
                    const parsed = JSON.parse(data);
                    // 도구 호출 수집
                    if (parsed.choices?.[0]?.delta?.tool_calls) {
                      toolCalls = parsed.choices[0].delta.tool_calls;
                      isToolCallComplete = true;
                    }
                    if (parsed.choices?.[0]?.delta?.content) {
                      controller.enqueue(encoder.encode(`data: ${JSON.stringify({ content: parsed.choices[0].delta.content })}\n\n`));
                    }
                  } catch (e) {
                    console.error('VLLM 파싱 오류:', e);
                  }
                }
              }
            }
          } else {
            // Ollama 스트리밍 처리
            const reader = response.body?.getReader();
            const decoder = new TextDecoder();
            let toolCalls: any[] = [];
            let isToolCallComplete = false;

            while (true) {
              const { done, value } = await reader!.read();
              if (done) break;

              const chunk = decoder.decode(value);
              const lines = chunk.split('\n');

              for (const line of lines) {
                if (line.trim()) {
                  try {
                    const parsed = JSON.parse(line);
                    // 도구 호출 수집
                    if (parsed.message?.tool_calls) {
                      toolCalls = parsed.message.tool_calls;
                      isToolCallComplete = true;
                    }
                    if (parsed.message?.content) {
                      controller.enqueue(encoder.encode(`data: ${JSON.stringify({ content: parsed.message.content })}\n\n`));
                    }
                    if (parsed.done) {
                      // 도구 호출이 완료된 경우 결과 추가
                      if (enableTools && toolCalls.length > 0 && isToolCallComplete) {
                        for (const toolCall of toolCalls) {
                          const toolResult = await executeTool(toolCall);
                          if (toolResult.result) {
                            controller.enqueue(encoder.encode(`data: ${JSON.stringify({ content: `\n\n🔧 도구 실행 결과: ${toolResult.result}` })}\n\n`));
                          } else if (toolResult.error) {
                            controller.enqueue(encoder.encode(`data: ${JSON.stringify({ content: `\n\n❌ 도구 실행 오류: ${toolResult.error}` })}\n\n`));
                          }
                        }
                      }
                      controller.enqueue(encoder.encode('data: [DONE]\n\n'));
                      controller.close();
                      return;
                    }
                  } catch (e) {
                    console.error('Ollama 파싱 오류:', e);
                  }
                }
              }
            }
          }
        } catch (error) {
          console.error('스트리밍 오류:', error);
          controller.enqueue(encoder.encode(`data: ${JSON.stringify({ error: '스트리밍 중 오류가 발생했습니다.' })}\n\n`));
          controller.close();
        }
      }
    });

    return new Response(stream, {
      headers: {
        'Content-Type': 'text/event-stream',
        'Cache-Control': 'no-cache',
        'Connection': 'keep-alive',
        'Access-Control-Allow-Origin': '*',
        'Access-Control-Allow-Methods': 'POST',
        'Access-Control-Allow-Headers': 'Content-Type'
      }
    });

  } catch (error) {
    console.error('스트리밍 API 오류:', error);
    return NextResponse.json(
      { error: '스트리밍 처리 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
} 