import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../../middlewares';

const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';

/**
 * 이미지 파일 업로드 API
 * 
 * 참고: 현재 클라이언트에서는 이미지를 base64로 직접 변환하여 채팅 API에 전송하므로
 * 이 엔드포인트가 항상 필요하지는 않을 수 있습니다. 그러나 API 서버와의 완전한
 * 호환성을 위해 유지됩니다.
 * 
 * 이 엔드포인트는 다음과 같은 경우에 유용합니다:
 * 1. 대용량 이미지를 서버 측에서 최적화해야 할 때
 * 2. 이미지를 서버에 영구 저장하고 URL로 참조해야 할 때
 * 3. 이미지 처리 로직이 클라이언트 측 JavaScript에서 처리하기 어려울 때
 */
export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      // formData 형식으로 전송된 데이터 파싱
      const formData = await req.formData();
      const file = formData.get('file') as File;
      
      if (!file) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '이미지 파일이 제공되지 않았습니다.'
            }
          }, 
          { status: 400 }
        );
      }
      
      // 이미지 파일 타입 확인
      if (!file.type.startsWith('image/')) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '올바른 이미지 파일이 아닙니다.'
            }
          }, 
          { status: 400 }
        );
      }
      
      // 토큰 또는 API 키 설정
      const headers: Record<string, string> = {};

      if (authType === 'token') {
        const token = req.cookies.get('auth_token')?.value;
        if (token) {
          headers['Authorization'] = `Bearer ${token}`;
        } else {
          return NextResponse.json(
            {
              success: false,
              error: {
                message: '인증 토큰이 필요합니다.'
              }
            }, 
            { status: 401 }
          );
        }
      } else if (authType === 'apikey') {
        const apiKey = req.headers.get('x-api-key');
        if (apiKey) {
          headers['X-API-Key'] = apiKey;
        } else {
          return NextResponse.json(
            {
              success: false,
              error: {
                message: 'API 키가 필요합니다.'
              }
            }, 
            { status: 401 }
          );
        }
      } else {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '인증이 필요합니다.'
            }
          }, 
          { status: 401 }
        );
      }
      
      // 파일 데이터를 바이너리 형태로 준비
      const bytes = await file.arrayBuffer();
      const buffer = Buffer.from(bytes);
      
      // 이미지를 Base64로 인코딩
      const base64Image = buffer.toString('base64');
      const mimeType = file.type;
      const base64String = `data:${mimeType};base64,${base64Image}`;
      
      // API 서버로 이미지 처리 요청을 위한 FormData 생성
      const apiFormData = new FormData();
      const blob = new Blob([buffer], { type: file.type });
      apiFormData.append('file', blob, file.name);
      
      if (user?.id) {
        apiFormData.append('user_id', user.id);
      }
      
      // API 서버에 이미지 처리 요청
      console.log('API 서버에 이미지 처리 요청 전송:', {
        url: `${API_SERVER}/rag/process-image`,
        method: 'POST',
        file: {
          name: file.name,
          type: file.type,
          size: file.size
        }
      });
      
      try {
        // API 서버에 이미지 전송 시도
        const apiResponse = await fetch(`${API_SERVER}/rag/process-image`, {
          method: 'POST',
          headers: headers,
          body: apiFormData
        });
        
        if (!apiResponse.ok) {
          console.warn('API 서버 이미지 처리 실패. 로컬 처리로 대체:', {
            status: apiResponse.status,
            statusText: apiResponse.statusText
          });
          
          // API 서버 처리 실패 시 로컬에서 처리된 Base64 데이터 반환
          return NextResponse.json({
            success: true,
            data: {
              id: `image-${Date.now()}`,
              name: file.name,
              type: file.type,
              size: file.size,
              base64: base64String,
              message: '이미지가 처리되었습니다. (로컬 처리)'
            }
          });
        }
        
        const apiData = await apiResponse.json();
        console.log('API 서버 이미지 처리 성공:', apiData);
        
        // API 서버에서 처리된 이미지 URL 또는 Base64 반환
        return NextResponse.json({
          success: true,
          data: {
            id: apiData.id || `image-${Date.now()}`,
            name: file.name,
            type: file.type,
            size: file.size,
            base64: apiData.base64 || base64String,
            imageUrl: apiData.imageUrl,
            message: '이미지가 성공적으로 업로드되었습니다.'
          }
        });
      } catch (error) {
        console.error('API 서버 이미지 처리 요청 실패:', error);
        
        // 오류 발생 시 로컬에서 처리된 Base64 데이터 반환
        return NextResponse.json({
          success: true,
          data: {
            id: `image-${Date.now()}`,
            name: file.name,
            type: file.type,
            size: file.size,
            base64: base64String,
            message: '이미지가 처리되었습니다. (로컬 처리)'
          }
        });
      }
    } catch (error) {
      console.error('이미지 처리 요청 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: error instanceof Error ? error.message : '이미지 처리 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
} 