import { NextResponse } from 'next/server';

const OLLAMA_URL = process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || process.env.OLLAMA_URL || "https://api.hamonize.com/ollama";
const VLLM_SERVER = process.env.NEXT_PUBLIC_VLLM_SERVER || "https://api.hamonize.com/vllm";

// 메타데이터 캐시 (메모리 기반, 서버 재시작 시 초기화)
const metadataCache = new Map<string, any>();
const CACHE_EXPIRY = 24 * 60 * 60 * 1000; // 24시간

// Ollama 모델 상세 정보 실시간 수집
async function fetchOllamaModelInfo(modelName: string) {
  const cacheKey = `ollama:${modelName}`;
  const cached = metadataCache.get(cacheKey);
  
  // 캐시된 데이터가 있고 24시간 이내라면 반환
  if (cached && (Date.now() - cached.timestamp) < CACHE_EXPIRY) {
    return cached.data;
  }

  try {
    // Ollama API로 모델 상세 정보 조회
    const response = await fetch(`${OLLAMA_URL}/api/show`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ name: modelName }),
    });

    if (response.ok) {
      const data = await response.json();
      const modelfile = data.modelfile || '';
      const parameters = data.parameters || {};
      const template = data.template || '';
      
      // 기본 메타데이터 추출 (실제 테스트는 타임아웃으로 인해 비활성화)
      const basicSupportsTools = checkToolSupport(template, modelfile, modelName);
      const basicSupportsVision = checkVisionSupport(modelfile, modelName);
      
      // 실제 테스트는 성능상 이유로 비활성화 - 기본 감지 로직 사용
      const testedToolSupport = basicSupportsTools;
      const testedVisionSupport = basicSupportsVision;
      
      // 실제 데이터에서 메타데이터 추출
      const modelInfo = {
        parameters: extractParameterCount(modelfile, modelName),
        context_length: extractContextLength(modelfile, modelName, parameters),
        supports_tools: testedToolSupport,
        supports_vision: testedVisionSupport,
        license: extractLicense(modelfile),
        family: extractModelFamily(modelName),
        size_gb: data.size ? (data.size / (1024 * 1024 * 1024)).toFixed(1) : null,
        modified_at: data.modified_at,
        source: 'ollama_api_metadata'
      };

      // 캐시에 저장
      metadataCache.set(cacheKey, {
        data: modelInfo,
        timestamp: Date.now()
      });

      return modelInfo;
    }
  } catch (error) {
    console.warn(`Ollama 모델 ${modelName} 상세 정보 조회 실패:`, error);
  }

  // API 실패 시 패턴 기반 추론으로 폴백
  return getOllamaModelInfoFallback(modelName);
}

// HuggingFace 모델 상세 정보 실시간 수집
async function fetchHuggingFaceModelInfo(modelName: string) {
  const cacheKey = `hf:${modelName}`;
  const cached = metadataCache.get(cacheKey);
  
  if (cached && (Date.now() - cached.timestamp) < CACHE_EXPIRY) {
    return cached.data;
  }

  try {
    // HuggingFace API로 모델 정보 조회
    const [modelResponse, configResponse] = await Promise.allSettled([
      fetch(`https://huggingface.co/api/models/${modelName}`, {
        headers: { 'User-Agent': 'airun-eval/1.0' }
      }),
      fetch(`https://huggingface.co/${modelName}/raw/main/config.json`, {
        headers: { 'User-Agent': 'airun-eval/1.0' }
      })
    ]);

    let modelData = null;
    let configData = null;

    if (modelResponse.status === 'fulfilled' && modelResponse.value.ok) {
      modelData = await modelResponse.value.json();
    }

    if (configResponse.status === 'fulfilled' && configResponse.value.ok) {
      configData = await configResponse.value.json();
    }

    if (modelData) {
      // 기본 메타데이터 추출 (실제 테스트는 타임아웃으로 인해 비활성화)
      const basicSupportsTools = checkHFToolSupport(modelData);
      const basicSupportsVision = checkHFVisionSupport(modelData, modelName);
      
      // 실제 테스트는 성능상 이유로 비활성화 - 기본 감지 로직 사용
      const testedToolSupport = basicSupportsTools;
      const testedVisionSupport = basicSupportsVision;
      
      const modelInfo = {
        parameters: extractHFParameterCount(modelData, modelName),
        context_length: extractHFContextLength(configData, modelName),
        supports_tools: testedToolSupport,
        supports_vision: testedVisionSupport,
        license: modelData.cardData?.license || 'Unknown',
        family: extractModelFamily(modelName),
        downloads: modelData.downloads || 0,
        likes: modelData.likes || 0,
        tags: modelData.tags || [],
        created_at: modelData.createdAt,
        source: 'huggingface_api_metadata'
      };

      metadataCache.set(cacheKey, {
        data: modelInfo,
        timestamp: Date.now()
      });

      return modelInfo;
    }
  } catch (error) {
    console.warn(`HuggingFace 모델 ${modelName} 정보 조회 실패:`, error);
  }

  // API 실패 시 패턴 기반 추론으로 폴백
  return getVLLMModelInfoFallback(modelName);
}

// 파라미터 수 추출 함수들
function extractParameterCount(modelfile: string, modelName: string): string {
  // Modelfile에서 파라미터 수 추출
  const sizeMatch = modelfile.match(/(\d+\.?\d*)[BM]/i);
  if (sizeMatch) return sizeMatch[0].toUpperCase();
  
  // 모델명에서 추출
  const nameMatch = modelName.match(/(\d+\.?\d*)[bB]/);
  if (nameMatch) return `${nameMatch[1]}B`;
  
  return 'Unknown';
}

function extractHFParameterCount(modelData: any, modelName: string): string {
  if (modelData.safetensors?.total) {
    const total = modelData.safetensors.total;
    if (total > 1e9) return `${Math.round(total / 1e9)}B`;
    if (total > 1e6) return `${Math.round(total / 1e6)}M`;
  }
  
  const nameMatch = modelName.match(/(\d+\.?\d*)[bB]/);
  if (nameMatch) return `${nameMatch[1]}B`;
  
  return 'Unknown';
}

function extractContextLength(modelfile: string, modelName: string, parameters?: any): number {
  const modelNameLower = modelName.toLowerCase();
  
  // 1. parameters에서 num_ctx 확인 (명시적 설정 우선)
  if (parameters && parameters.num_ctx) {
    return parseInt(parameters.num_ctx);
  }
  
  // 2. Modelfile에서 컨텍스트 길이 추출
  const ctxMatch = modelfile.match(/num_ctx\s+(\d+)/i);
  if (ctxMatch) {
    return parseInt(ctxMatch[1]);
  }
  
  // 3. 특별한 모델 케이스 (베이스 모델 정보 기반)
  if (modelNameLower.includes('airun-chat')) {
    // airun-chat는 devstral 기반, Modelfile에서 4096으로 설정됨 (의도적 제한)
    return 4096; // 실제 설정값 반영
  }
  if (modelNameLower.includes('airun-vision')) {
    // airun-vision은 qwen2.5vl 기반, num_ctx 미설정시 베이스 모델 기본값
    return 32768; // qwen2.5vl 기본값
  }
  
  // 4. 모델 패턴 기반 추론
  if (modelNameLower.includes('llama3.2')) return 128000;
  if (modelNameLower.includes('qwen')) return 32768;
  if (modelNameLower.includes('mistral') || modelNameLower.includes('devstral')) return 32768;
  if (modelNameLower.includes('gemma')) return 8192;
  if (modelNameLower.includes('phi4')) return 16384;
  
  return 4096;
}

function extractHFContextLength(configData: any, modelName: string): number {
  if (configData) {
    return configData.max_position_embeddings || 
           configData.n_positions || 
           configData.max_sequence_length || 
           configData.seq_length || 
           4096;
  }
  
  // 모델 패턴 기반 추론
  if (modelName.includes('llama')) return 8192;
  if (modelName.includes('qwen')) return 32768;
  if (modelName.includes('mistral')) return 32768;
  
  return 4096;
}

function checkToolSupport(template: string, modelfile: string, modelName: string): boolean {
  const templateLower = template.toLowerCase();
  const modelfileLower = modelfile.toLowerCase();
  const modelNameLower = modelName.toLowerCase();
  
  // 1. 특정 모델 케이스
  if (modelNameLower.includes('airun-chat')) return true; // devstral 기반, 도구 지원
  if (modelNameLower.includes('airun-vision')) return false; // 비전 전용
  
  // 2. FROM 구문에서 베이스 모델 확인
  if (modelfileLower.includes('from devstral')) return true;
  if (modelfileLower.includes('from qwen') && !modelfileLower.includes('vl')) return true;
  if (modelfileLower.includes('from mistral')) return true;
  
  // 3. 템플릿에서 도구 관련 키워드
  return templateLower.includes('tool') || 
         templateLower.includes('function') ||
         templateLower.includes('available_tools') ||
         templateLower.includes('tool_calls') ||
         templateLower.includes('tool_results') ||
         modelfileLower.includes('tool') ||
         modelfileLower.includes('function');
}

// 실제 도구 지원 테스트 (성능상 이유로 비활성화됨)
async function testActualToolSupport(modelName: string, modelType: 'ollama' | 'vllm'): Promise<boolean> {
  // 타임아웃 문제로 인해 비활성화 - 항상 false 반환
  return false;
  /*
  try {
    const testTool = {
      type: "function",
      function: {
        name: "get_current_time",
        description: "현재 시간을 가져옵니다",
        parameters: {
          type: "object",
          properties: {},
          required: []
        }
      }
    };

    let response;
    if (modelType === 'ollama') {
      response = await fetch(`${OLLAMA_URL}/api/generate`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: modelName,
          prompt: "현재 시간을 알려주세요.",
          tools: [testTool],
          stream: false
        }),
        signal: AbortSignal.timeout(15000) // 15초 타임아웃
      });
    } else {
      response = await fetch(`${VLLM_SERVER}/v1/chat/completions`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: modelName,
          messages: [
            { role: "user", content: "현재 시간을 알려주세요." }
          ],
          tools: [testTool],
          max_tokens: 100
        }),
        signal: AbortSignal.timeout(15000)
      });
    }

    if (response.ok) {
      const data = await response.json();
      
      if (modelType === 'ollama') {
        // Ollama에서 도구 호출이 있는지 확인
        return data.response && (
          data.response.includes('tool_call') ||
          data.response.includes('function_call') ||
          data.response.includes('get_current_time') ||
          JSON.stringify(data).includes('tool')
        );
      } else {
        // VLLM에서 도구 호출이 있는지 확인
        const choice = data.choices?.[0];
        return choice?.message?.tool_calls && choice.message.tool_calls.length > 0;
      }
    }
  } catch (error) {
    console.warn(`도구 지원 테스트 실패 (${modelName}):`, error);
  }
  return false;
  */
}

// 실제 비전 지원 테스트 (성능상 이유로 비활성화됨)
async function testActualVisionSupport(modelName: string, modelType: 'ollama' | 'vllm'): Promise<boolean> {
  // 타임아웃 문제로 인해 비활성화 - 항상 false 반환
  return false;
  /*
  try {
    // 1x1 픽셀 테스트 이미지 (base64)
    const testImage = "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==";

    let response;
    if (modelType === 'ollama') {
      response = await fetch(`${OLLAMA_URL}/api/generate`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: modelName,
          prompt: "이 이미지에 무엇이 보이나요?",
          images: [testImage.split(',')[1]], // base64 부분만
          stream: false
        }),
        signal: AbortSignal.timeout(15000)
      });
    } else {
      response = await fetch(`${VLLM_SERVER}/v1/chat/completions`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: modelName,
          messages: [
            {
              role: "user",
              content: [
                { type: "text", text: "이 이미지에 무엇이 보이나요?" },
                { type: "image_url", image_url: { url: testImage } }
              ]
            }
          ],
          max_tokens: 100
        }),
        signal: AbortSignal.timeout(15000)
      });
    }

    if (response.ok) {
      const data = await response.json();
      
      if (modelType === 'ollama') {
        // 의미있는 이미지 관련 응답이 있는지 확인
        const responseText = data.response?.toLowerCase() || '';
        return responseText.length > 10 && (
          responseText.includes('이미지') ||
          responseText.includes('그림') ||
          responseText.includes('사진') ||
          responseText.includes('보이') ||
          responseText.includes('색') ||
          responseText.includes('픽셀')
        );
      } else {
        // VLLM에서 의미있는 응답이 있는지 확인
        const responseText = data.choices?.[0]?.message?.content?.toLowerCase() || '';
        return responseText.length > 10 && !responseText.includes('cannot') && !responseText.includes('unable');
      }
    }
  } catch (error) {
    console.warn(`비전 지원 테스트 실패 (${modelName}):`, error);
  }
  return false;
  */
}

function checkHFToolSupport(modelData: any): boolean {
  const tags = modelData.tags || [];
  return tags.includes('tool-use') || 
         tags.includes('function-calling') ||
         tags.includes('agents') ||
         modelData.pipeline_tag === 'text-generation';
}

function checkVisionSupport(modelfile: string, modelName: string): boolean {
  const modelfileLower = modelfile.toLowerCase();
  const modelNameLower = modelName.toLowerCase();
  
  // 1. 모델명 기반 확실한 비전 모델 감지
  if (modelNameLower.includes('airun-vision')) return true; // qwen2.5vl 기반
  
  // 2. FROM 구문에서 비전 모델 베이스 감지
  if (modelfileLower.includes('from qwen') && modelfileLower.includes('vl')) return true;
  if (modelfileLower.includes('from llava')) return true;
  if (modelfileLower.includes('from minicpm')) return true;
  
  // 3. 일반적인 비전 관련 키워드
  return modelfileLower.includes('vision') ||
         modelfileLower.includes('multimodal') ||
         modelfileLower.includes('이미지') ||
         modelfileLower.includes('image') ||
         modelfileLower.includes('visual') ||
         modelfileLower.includes('이미지 설명') ||
         modelNameLower.includes('vision') ||
         modelNameLower.includes('vl') ||
         modelNameLower.includes('visual');
}

function checkHFVisionSupport(modelData: any, modelName: string): boolean {
  const tags = modelData.tags || [];
  return tags.includes('multimodal') ||
         tags.includes('vision') ||
         tags.includes('image-to-text') ||
         modelData.pipeline_tag === 'image-to-text' ||
         modelName.toLowerCase().includes('vision') ||
         modelName.toLowerCase().includes('vl');
}

function extractLicense(modelfile: string): string {
  const licenseMatch = modelfile.match(/license[:\s]+([^\n]+)/i);
  return licenseMatch ? licenseMatch[1].trim() : 'Unknown';
}

function extractModelFamily(modelName: string): string {
  const name = modelName.toLowerCase();
  if (name.includes('llama')) return 'llama';
  if (name.includes('gemma')) return 'gemma';
  if (name.includes('qwen')) return 'qwen';
  if (name.includes('mistral')) return 'mistral';
  if (name.includes('hyperclova')) return 'hyperclova';
  if (name.includes('airun')) return 'airun';
  return 'unknown';
}

// 폴백 함수들 (기존 패턴 기반 추론)
function getOllamaModelInfoFallback(modelName: string) {
  const name = modelName.toLowerCase();
  
  // 기본값
  let info = {
    supports_tools: false,
    supports_vision: false,
    context_length: 4096,
    parameters: 'Unknown'
  };

  // 모델명 기반 정보 추론
  if (name.includes('llama')) {
    info.supports_tools = true;
    info.context_length = name.includes('3.2') ? 128000 : 8192;
    if (name.includes('7b')) info.parameters = '7B';
    else if (name.includes('13b')) info.parameters = '13B';
    else if (name.includes('70b')) info.parameters = '70B';
  } else if (name.includes('gemma')) {
    info.supports_tools = true;
    info.context_length = 8192;
    if (name.includes('2b')) info.parameters = '2B';
    else if (name.includes('7b')) info.parameters = '7B';
    else if (name.includes('12b')) info.parameters = '12B';
  } else if (name.includes('qwen')) {
    info.supports_tools = true;
    info.supports_vision = name.includes('vl');
    info.context_length = 32768;
    if (name.includes('7b')) info.parameters = '7B';
    else if (name.includes('14b')) info.parameters = '14B';
    else if (name.includes('72b')) info.parameters = '72B';
  } else if (name.includes('mistral')) {
    info.supports_tools = true;
    info.context_length = 32768;
    if (name.includes('7b')) info.parameters = '7B';
    else if (name.includes('22b')) info.parameters = '22B';
  } else if (name.includes('hyperclova')) {
    info.supports_tools = false;
    info.context_length = 4096;
    if (name.includes('1.5b')) info.parameters = '1.5B';
  } else if (name.includes('airun-chat')) {
    info.supports_tools = true;
    info.context_length = 4096; // Modelfile에서 명시적으로 4096 설정
    info.parameters = '22B'; // devstral 기반
  } else if (name.includes('airun-vision')) {
    info.supports_tools = false; // 비전 전용
    info.supports_vision = true;
    info.context_length = 32768; // qwen2.5vl 기본값
    info.parameters = '7B'; // qwen2.5vl 추정
  } else if (name.includes('devstral')) {
    info.supports_tools = true;
    info.context_length = 32768; // devstral 기본값
    info.parameters = '22B';
  }

  return info;
}

function getVLLMModelInfoFallback(modelName: string) {
  const name = modelName.toLowerCase();
  
  // 기본값
  let info = {
    supports_tools: true, // VLLM 모델은 대부분 도구 지원
    supports_vision: false,
    context_length: 4096,
    parameters: 'Unknown',
    size_gb: null as string | null
  };

  // HuggingFace 모델명 기반 정보 추론
  if (name.includes('llama')) {
    info.context_length = 8192;
    if (name.includes('7b')) {
      info.parameters = '7B';
      info.size_gb = '4.1';
    } else if (name.includes('13b')) {
      info.parameters = '13B';
      info.size_gb = '7.4';
    } else if (name.includes('70b')) {
      info.parameters = '70B';
      info.size_gb = '39.0';
    }
  } else if (name.includes('mistral')) {
    info.context_length = 32768;
    if (name.includes('7b')) {
      info.parameters = '7B';
      info.size_gb = '4.1';
    } else if (name.includes('22b')) {
      info.parameters = '22B';
      info.size_gb = '12.9';
    }
  } else if (name.includes('qwen')) {
    info.supports_vision = name.includes('vl');
    info.context_length = 32768;
    if (name.includes('7b')) {
      info.parameters = '7B';
      info.size_gb = '4.3';
    } else if (name.includes('14b')) {
      info.parameters = '14B';
      info.size_gb = '8.2';
    } else if (name.includes('32b')) {
      info.parameters = '32B';
      info.size_gb = '18.9';
    }
  } else if (name.includes('devstral')) {
    info.context_length = 16384;
    info.parameters = '22B';
    info.size_gb = '12.9';
  }

  return info;
}

export async function GET() {
  const allModels: any[] = [];
  
  try {
    // Ollama 모델 목록 조회
    try {
      const ollamaResponse = await fetch(`${OLLAMA_URL}/api/tags`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      if (ollamaResponse.ok) {
        const ollamaData = await ollamaResponse.json();
        const ollamaModels = await Promise.all(
          (ollamaData.models || []).map(async (model: any) => {
            // Ollama 모델별 상세 정보 추론
            const modelInfo = await fetchOllamaModelInfo(model.name);
            return {
              name: model.name,
              size: model.size,
              modified_at: model.modified_at,
              digest: model.digest,
              type: 'ollama',
              provider: 'Ollama',
              ...modelInfo
            };
          })
        );
        allModels.push(...ollamaModels);
      }
    } catch (error) {
      console.warn('Ollama 모델 조회 실패:', error);
    }

    // VLLM 모델 목록 조회
    try {
      const vllmResponse = await fetch(`${VLLM_SERVER}/v1/models`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      if (vllmResponse.ok) {
        const vllmData = await vllmResponse.json();
        const vllmModels = await Promise.all(
          (vllmData.data || []).map(async (model: any) => {
            // VLLM 모델별 상세 정보 추론
            const modelInfo = await fetchHuggingFaceModelInfo(model.id);
            return {
              name: model.id,
              object: model.object,
              created: model.created,
              owned_by: model.owned_by,
              type: 'vllm',
              provider: 'VLLM',
              size: modelInfo.size_gb ? parseFloat(modelInfo.size_gb) * 1024 * 1024 * 1024 : 0, // GB를 바이트로 변환
              ...modelInfo
            };
          })
        );
        allModels.push(...vllmModels);
      }
    } catch (error) {
      console.warn('VLLM 모델 조회 실패:', error);
    }

    return NextResponse.json({ 
      models: allModels,
      total: allModels.length,
      ollama: allModels.filter(m => m.type === 'ollama').length,
      vllm: allModels.filter(m => m.type === 'vllm').length
    });
  } catch (error) {
    console.error('모델 목록 조회 오류:', error);
    return NextResponse.json({ 
      models: allModels,
      error: '일부 모델 목록을 조회할 수 없습니다.' 
    }, { status: 500 });
  }
} 