import { NextRequest, NextResponse } from 'next/server';
import { getTokenFromCookies } from '../../auth/utils';
import { getApiServerUrl } from '@/config/serverConfig';

const API_SERVER = getApiServerUrl(); //.replace('/api/v1', '');

export async function GET(request: NextRequest) {
  try {
    console.log('🔍 [피드백 데이터셋] API 라우트 호출됨');
    
    // 쿠키에서 토큰 가져오기
    const token = await getTokenFromCookies();
    console.log('🔑 [피드백 데이터셋] 토큰 상태:', token ? '토큰 있음' : '토큰 없음');
    
    if (!token) {
      console.log('❌ [피드백 데이터셋] 인증 토큰이 없음');
      return NextResponse.json(
        { 
          success: false, 
          error: '인증이 필요합니다. 로그인해주세요.' 
        },
        { status: 401 }
      );
    }
    
    const { searchParams } = new URL(request.url);
    const page = searchParams.get('page') || '1';
    const limit = searchParams.get('limit') || '100';
    
    console.log('📡 [피드백 데이터셋] API 서버 요청:', {
      url: `${API_SERVER}/feedback/dataset?page=${page}&limit=${limit}`,
      token: token.substring(0, 10) + '...'
    });
    
    // API 서버로 요청 전달 (인증 토큰 포함)
    // API_SERVER가 이미 /api/v1을 포함하고 있으므로 직접 엔드포인트만 추가
    const apiUrl = `${API_SERVER}/feedback/dataset?page=${page}&limit=${limit}`;
    
    const response = await fetch(apiUrl, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      }
    });
    
    console.log('📡 [피드백 데이터셋] API 서버 응답:', {
      status: response.status,
      ok: response.ok
    });
    
    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      console.log('❌ [피드백 데이터셋] API 서버 오류:', errorData);
      return NextResponse.json(
        { 
          success: false, 
          error: errorData.error || `API 서버 오류: ${response.status}` 
        },
        { status: response.status }
      );
    }
    
    const data = await response.json();
    console.log('✅ [피드백 데이터셋] 성공적으로 데이터 받음:', {
      success: data.success,
      datasetsCount: data.datasets?.length || 0
    });
    
    return NextResponse.json(data);
    
  } catch (error) {
    console.error('❌ [피드백 데이터셋] 프록시 오류:', error);
    return NextResponse.json(
      { 
        success: false, 
        error: '피드백 데이터셋을 불러올 수 없습니다.' 
      },
      { status: 500 }
    );
  }
}
