import { NextRequest, NextResponse } from 'next/server';
import { withTokenAuth } from '@/app/api/middlewares';
import { pg as dbManager } from '@/app/api/db';
import type { WorkflowData } from '@/types/flowai';

// 워크플로우 목록 조회
export async function GET(request: NextRequest) {
  return withTokenAuth(request, async (req, user) => {
    try {
      // 사용자의 워크플로우 목록 조회
      const query = `
        SELECT * FROM flowai_workflows 
        WHERE user_id = $1 OR is_public = true
        ORDER BY updated_at DESC
      `;
      
      const result = await dbManager.query(query, [user.id]);
      
      const workflows = result.rows.map(row => ({
        id: row.id,
        name: row.name,
        description: row.description,
        nodes: JSON.parse(row.nodes || '[]'),
        connections: JSON.parse(row.connections || '[]'),
        variables: JSON.parse(row.variables || '{}'),
        createdAt: row.created_at,
        updatedAt: row.updated_at,
        userId: row.user_id,
        isPublic: row.is_public,
        tags: JSON.parse(row.tags || '[]'),
      }));

      return NextResponse.json({
        success: true,
        data: workflows,
        timestamp: new Date().toISOString(),
      });

    } catch (error) {
      console.error('워크플로우 목록 조회 오류:', error);
      return NextResponse.json(
        { 
          success: false, 
          error: { 
            code: 'INTERNAL_SERVER_ERROR',
            message: '워크플로우 목록 조회에 실패했습니다' 
          } 
        },
        { status: 500 }
      );
    }
  });
}

// 워크플로우 생성
export async function POST(request: NextRequest) {
  return withTokenAuth(request, async (req, user) => {
    try {
      const body = await request.json();
      const workflow = body as WorkflowData;

      // 유효성 검사
      if (!workflow.name || !workflow.name.trim()) {
        return NextResponse.json(
          { success: false, error: { message: '워크플로우 이름이 필요합니다' } },
          { status: 400 }
        );
      }

      // 워크플로우 생성
      const query = `
        INSERT INTO flowai_workflows (
          id, name, description, nodes, connections, variables, 
          user_id, is_public, tags, created_at, updated_at
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, NOW(), NOW())
        RETURNING *
      `;

      const workflowId = `workflow_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      
      const values = [
        workflowId,
        workflow.name.trim(),
        workflow.description || null,
        JSON.stringify(workflow.nodes || []),
        JSON.stringify(workflow.connections || []),
        JSON.stringify(workflow.variables || {}),
        user.id,
        workflow.isPublic || false,
        JSON.stringify(workflow.tags || []),
      ];

      const result = await dbManager.query(query, values);
      const savedWorkflow = result.rows[0];

      const responseWorkflow: WorkflowData = {
        id: savedWorkflow.id,
        name: savedWorkflow.name,
        description: savedWorkflow.description,
        nodes: JSON.parse(savedWorkflow.nodes),
        connections: JSON.parse(savedWorkflow.connections),
        variables: JSON.parse(savedWorkflow.variables || '{}'),
        createdAt: savedWorkflow.created_at,
        updatedAt: savedWorkflow.updated_at,
        userId: savedWorkflow.user_id,
        isPublic: savedWorkflow.is_public,
        tags: JSON.parse(savedWorkflow.tags || '[]'),
      };

      return NextResponse.json({
        success: true,
        data: responseWorkflow,
        timestamp: new Date().toISOString(),
      });

    } catch (error) {
      console.error('워크플로우 생성 오류:', error);
      return NextResponse.json(
        { 
          success: false, 
          error: { 
            code: 'INTERNAL_SERVER_ERROR',
            message: '워크플로우 생성에 실패했습니다' 
          } 
        },
        { status: 500 }
      );
    }
  });
}

// 워크플로우 업데이트
export async function PUT(request: NextRequest) {
  return withTokenAuth(request, async (req, user) => {
    try {
      const body = await request.json();
      const workflow = body as WorkflowData;

      if (!workflow.id) {
        return NextResponse.json(
          { success: false, error: { message: '워크플로우 ID가 필요합니다' } },
          { status: 400 }
        );
      }

      // 소유권 확인
      const checkQuery = 'SELECT user_id FROM flowai_workflows WHERE id = $1';
      const checkResult = await dbManager.query(checkQuery, [workflow.id]);
      
      if (checkResult.rows.length === 0) {
        return NextResponse.json(
          { success: false, error: { message: '워크플로우를 찾을 수 없습니다' } },
          { status: 404 }
        );
      }

      if (checkResult.rows[0].user_id !== user.id && checkResult.rows[0].user_id !== String(user.id)) {
        return NextResponse.json(
          { success: false, error: { message: '워크플로우를 수정할 권한이 없습니다' } },
          { status: 403 }
        );
      }

      // 워크플로우 업데이트
      const updateQuery = `
        UPDATE flowai_workflows SET
          name = $1, description = $2, nodes = $3, connections = $4,
          variables = $5, is_public = $6, tags = $7, updated_at = NOW()
        WHERE id = $8
        RETURNING *
      `;

      const values = [
        workflow.name.trim(),
        workflow.description || null,
        JSON.stringify(workflow.nodes || []),
        JSON.stringify(workflow.connections || []),
        JSON.stringify(workflow.variables || {}),
        workflow.isPublic || false,
        JSON.stringify(workflow.tags || []),
        workflow.id,
      ];

      const result = await dbManager.query(updateQuery, values);
      const updatedWorkflow = result.rows[0];

      const responseWorkflow: WorkflowData = {
        id: updatedWorkflow.id,
        name: updatedWorkflow.name,
        description: updatedWorkflow.description,
        nodes: JSON.parse(updatedWorkflow.nodes),
        connections: JSON.parse(updatedWorkflow.connections),
        variables: JSON.parse(updatedWorkflow.variables || '{}'),
        createdAt: updatedWorkflow.created_at,
        updatedAt: updatedWorkflow.updated_at,
        userId: updatedWorkflow.user_id,
        isPublic: updatedWorkflow.is_public,
        tags: JSON.parse(updatedWorkflow.tags || '[]'),
      };

      return NextResponse.json({
        success: true,
        data: responseWorkflow,
        timestamp: new Date().toISOString(),
      });

    } catch (error) {
      console.error('워크플로우 업데이트 오류:', error);
      return NextResponse.json(
        { 
          success: false, 
          error: { 
            code: 'INTERNAL_SERVER_ERROR',
            message: '워크플로우 업데이트에 실패했습니다' 
          } 
        },
        { status: 500 }
      );
    }
  });
}