import { NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';

// app.py 패턴을 적용한 개선된 모니터링 데이터 프로세서
class ImprovedMonitoringProcessor {
  private static dataCache = new Map<string, { data: any; timestamp: number }>();
  private static CACHE_DURATION = 30000; // 30초 캐싱
  private static MAX_LOG_ENTRIES = 10000; // 메모리 보호를 위한 최대 로그 항목
  
  // app.py 패턴: 캐시된 데이터 가져오기
  private static getCachedData(cacheKey: string): any | null {
    const cached = this.dataCache.get(cacheKey);
    if (cached && (Date.now() - cached.timestamp) < this.CACHE_DURATION) {
      console.debug(`캐시에서 모니터링 데이터 반환: ${cacheKey}`);
      return cached.data;
    }
    return null;
  }

  // app.py 패턴: 캐시에 데이터 저장
  private static setCachedData(cacheKey: string, data: any): void {
    this.dataCache.set(cacheKey, {
      data,
      timestamp: Date.now()
    });
    
    // 캐시 크기 제한 (메모리 보호)
    if (this.dataCache.size > 10) {
      const oldestKey = this.dataCache.keys().next().value;
      if (oldestKey) {
        this.dataCache.delete(oldestKey);
      }
    }
  }

  // app.py 패턴: 타임아웃이 적용된 파일 읽기
  private static async readFileWithTimeout(filePath: string, timeoutMs: number = 10000): Promise<string> {
    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error(`파일 읽기 시간 초과: ${filePath}`));
      }, timeoutMs);

      fs.promises.readFile(filePath, 'utf8')
        .then(content => {
          clearTimeout(timeout);
          resolve(content);
        })
        .catch(error => {
          clearTimeout(timeout);
          reject(error);
        });
    });
  }

  // app.py 패턴: 안전한 JSON 파싱
  private static safeJsonParse(content: string): any {
    try {
      if (!content.trim()) {
        throw new Error('파일 내용이 비어 있습니다.');
      }
      return JSON.parse(content);
    } catch (error) {
      console.error('JSON 파싱 실패:', error);
      throw new Error(`JSON 파싱 오류: ${(error as Error).message}`);
    }
  }

  // app.py 패턴: 로그 데이터 청크 단위로 병렬 처리
  private static processLogsInChunks(logs: any[], chunkSize: number = 1000): Promise<any>[] {
    const chunks = [];
    for (let i = 0; i < logs.length; i += chunkSize) {
      chunks.push(logs.slice(i, i + chunkSize));
    }

    return chunks.map(chunk => this.processLogChunk(chunk));
  }

  // app.py 패턴: 로그 청크 처리
  private static async processLogChunk(chunk: any[]): Promise<{
    endpointMap: Record<string, any>;
    timeTrendMap: Record<string, { count: number; sum: number }>;
    errorTrendMap: Record<string, { count: number; error: number }>;
    allHourly: number[][];
    stats: {
      totalCalls: number;
      limitedCalls: number;
      totalResponseTime: number;
      responseCount: number;
      successCount: number;
    };
    processedLogs: any[];
  }> {
    const endpointMap: Record<string, any> = {};
    const allHourly: number[][] = Array(7).fill(0).map(() => Array(24).fill(0));
    const timeTrendMap: Record<string, { count: number; sum: number }> = {};
    const errorTrendMap: Record<string, { count: number; error: number }> = {};
    const processedLogs: any[] = [];
    
    let totalCalls = 0, limitedCalls = 0, totalResponseTime = 0, responseCount = 0, successCount = 0;

    for (const log of chunk) {
      if (!log.endpoint || !log.method) continue;
      
      const key = `${log.endpoint}|${log.method}`;
      if (!endpointMap[key]) {
        endpointMap[key] = {
          endpoint: log.endpoint,
          method: log.method,
          calls: 0,
          totalResponseTime: 0,
          successCount: 0,
          logs: [],
          hourly: Array(24).fill(0),
        };
      }

      endpointMap[key].calls++;
      totalCalls++;
      
      if (!this.isApiExcluded(log.endpoint)) limitedCalls++;
      
      if (typeof log.responseTime === 'number') {
        endpointMap[key].totalResponseTime += log.responseTime;
        totalResponseTime += log.responseTime;
        responseCount++;
      }
      
      if (log.status && String(log.status).startsWith('2')) {
        endpointMap[key].successCount++;
        successCount++;
      }
      
      endpointMap[key].logs.push({
        timestamp: log.timestamp,
        status: log.status,
        responseTime: log.responseTime,
        userId: log.userId,
        endpoint: log.endpoint,
        method: log.method
      });

      // 시간대별 호출수(요일/시간)
      if (log.timestamp) {
        try {
          const d = new Date(log.timestamp);
          const day = d.getDay();
          const hour = d.getHours();
          if (day >= 0 && day < 7 && hour >= 0 && hour < 24) {
            endpointMap[key].hourly[hour]++;
            allHourly[day][hour]++;
          }
        } catch (error) {
          console.debug(`타임스탬프 파싱 실패: ${log.timestamp}`);
        }
      }

      // 트렌드(시간별 응답/오류)
      if (log.timestamp) {
        try {
          const t = log.timestamp.slice(0, 13); // yyyy-MM-ddTHH
          if (!timeTrendMap[t]) timeTrendMap[t] = { count: 0, sum: 0 };
          if (!errorTrendMap[t]) errorTrendMap[t] = { count: 0, error: 0 };
          
          if (typeof log.responseTime === 'number') {
            timeTrendMap[t].count++;
            timeTrendMap[t].sum += log.responseTime;
          }
          
          errorTrendMap[t].count++;
          if (log.status >= 400) errorTrendMap[t].error++;
        } catch (error) {
          console.debug(`트렌드 데이터 처리 실패: ${log.timestamp}`);
        }
      }

      // 전체 로그 테이블용
      processedLogs.push({
        timestamp: log.timestamp,
        endpoint: log.endpoint,
        method: log.method,
        status: log.status,
        responseTime: log.responseTime,
        userId: log.userId
      });
    }

    return {
      endpointMap,
      timeTrendMap,
      errorTrendMap,
      allHourly,
      stats: { totalCalls, limitedCalls, totalResponseTime, responseCount, successCount },
      processedLogs
    };
  }

  // app.py 패턴: 병렬 처리 결과 병합
  private static mergeChunkResults(chunkResults: any[]): any {
    const finalEndpointMap: Record<string, any> = {};
    const finalTimeTrendMap: Record<string, { count: number; sum: number }> = {};
    const finalErrorTrendMap: Record<string, { count: number; error: number }> = {};
    const finalAllHourly: number[][] = Array(7).fill(0).map(() => Array(24).fill(0));
    const finalProcessedLogs: any[] = [];
    
    let totalCalls = 0, limitedCalls = 0, totalResponseTime = 0, responseCount = 0, successCount = 0;

    for (const result of chunkResults) {
      // 엔드포인트 맵 병합
      for (const [key, endpoint] of Object.entries(result.endpointMap)) {
        const ep = endpoint as any;
        if (!finalEndpointMap[key]) {
          finalEndpointMap[key] = { ...ep };
        } else {
          finalEndpointMap[key].calls += ep.calls;
          finalEndpointMap[key].totalResponseTime += ep.totalResponseTime;
          finalEndpointMap[key].successCount += ep.successCount;
          finalEndpointMap[key].logs.push(...ep.logs);
          for (let h = 0; h < 24; h++) {
            finalEndpointMap[key].hourly[h] += ep.hourly[h];
          }
        }
      }

      // 트렌드 맵 병합
      for (const [time, trend] of Object.entries(result.timeTrendMap)) {
        const t = trend as any;
        if (!finalTimeTrendMap[time]) {
          finalTimeTrendMap[time] = { ...t };
        } else {
          finalTimeTrendMap[time].count += t.count;
          finalTimeTrendMap[time].sum += t.sum;
        }
      }

      for (const [time, trend] of Object.entries(result.errorTrendMap)) {
        const t = trend as any;
        if (!finalErrorTrendMap[time]) {
          finalErrorTrendMap[time] = { ...t };
        } else {
          finalErrorTrendMap[time].count += t.count;
          finalErrorTrendMap[time].error += t.error;
        }
      }

      // 시간대별 데이터 병합
      for (let d = 0; d < 7; d++) {
        for (let h = 0; h < 24; h++) {
          finalAllHourly[d][h] += result.allHourly[d][h];
        }
      }

      // 통계 병합
      totalCalls += result.stats.totalCalls;
      limitedCalls += result.stats.limitedCalls;
      totalResponseTime += result.stats.totalResponseTime;
      responseCount += result.stats.responseCount;
      successCount += result.stats.successCount;

      // 로그 병합
      finalProcessedLogs.push(...result.processedLogs);
    }

    return {
      endpointMap: finalEndpointMap,
      timeTrendMap: finalTimeTrendMap,
      errorTrendMap: finalErrorTrendMap,
      allHourly: finalAllHourly,
      stats: { totalCalls, limitedCalls, totalResponseTime, responseCount, successCount },
      processedLogs: finalProcessedLogs
    };
  }

  // API 제외 검사 함수
  private static isApiExcluded(endpoint: string): boolean {
    const fullPath = endpoint.startsWith('/api/v1/') ? endpoint :
      endpoint.startsWith('/') ? `/api/v1${endpoint}` :
      `/api/v1/${endpoint}`;
    
    const patterns = [
      /^\/api\/v1\/docs/,
      /^\/api\/v1\/docs\.json$/,
      /^\/api\/v1\/health$/,
      /^\/api\/v1\/license\/validate$/,
      /^\/api\/v1\/config(\/.*)?$/,
      /^\/api\/v1\/usage$/,
      /^\/api\/v1\/sessions$/,
      /^\/api\/v1\/cache\/clear$/,
      /^\/api\/v1\/monitor\/data$/,
      /^\/api\/v1\/rag\/(search|add|path|list|init)$/,
      /^\/api\/v1\/web\/search$/
    ];
    
    return patterns.some(pattern => pattern.test(fullPath));
  }

  // app.py 패턴: 메인 처리 함수
  static async processMonitoringData(): Promise<any> {
    const startTime = Date.now();
    console.debug('모니터링 데이터 처리 시작');
    
    // 실제 api_monitor.json 파일 경로
    const monitorFile = path.join(os.homedir(), '.airun', 'logs', 'api_monitor.json');
    const cacheKey = `monitoring_${monitorFile}`;

    // app.py 패턴: 캐시에서 먼저 확인
    const cachedData = this.getCachedData(cacheKey);
    if (cachedData) {
      console.debug(`캐시된 모니터링 데이터 반환 (${Date.now() - startTime}ms)`);
      return cachedData;
    }

    // app.py 패턴: 파일 존재 확인
    if (!fs.existsSync(monitorFile)) {
      throw new Error('api_monitor.json 파일이 없습니다.');
    }

    // app.py 패턴: 타임아웃이 적용된 파일 읽기 (10초)
    const fileContent = await this.readFileWithTimeout(monitorFile, 10000);
    
    // app.py 패턴: 안전한 JSON 파싱
    const monitorData = this.safeJsonParse(fileContent);
    const logs = Array.isArray(monitorData.logs) ? monitorData.logs : [];
    
    // app.py 패턴: 메모리 보호를 위한 로그 개수 제한
    const limitedLogs = logs.slice(0, this.MAX_LOG_ENTRIES);
    if (logs.length > this.MAX_LOG_ENTRIES) {
      console.warn(`로그 개수가 제한됨: ${logs.length} -> ${this.MAX_LOG_ENTRIES}`);
    }

    console.debug(`로그 처리 시작: ${limitedLogs.length}개 항목`);

    // app.py 패턴: 로그가 많은 경우 청크 단위로 병렬 처리
    const processingPromises = this.processLogsInChunks(limitedLogs, 1000);
    const chunkResults = await Promise.all(processingPromises);
    
    console.debug(`청크 처리 완료: ${chunkResults.length}개 청크`);

    // app.py 패턴: 병렬 처리 결과 병합
    const mergedResults = this.mergeChunkResults(chunkResults);
    
    // 엔드포인트별 통계 계산
    const endpointStats: Record<string, { calls: number; avgResponseTime: number; successRate: number }> = {};
    for (const [key, ep] of Object.entries(mergedResults.endpointMap)) {
      const endpoint = ep as any;
      endpointStats[endpoint.endpoint] = {
        calls: endpoint.calls,
        avgResponseTime: endpoint.calls > 0 ? Math.round(endpoint.totalResponseTime / endpoint.calls) : 0,
        successRate: endpoint.calls > 0 ? Math.round((endpoint.successCount / endpoint.calls) * 100) / 100 : 0
      };
    }

    // 응답시간/오류율 트렌드 계산
    const timeLabels = Object.keys(mergedResults.timeTrendMap).sort();
    const responseTimes = timeLabels.map(t => {
      const trend = mergedResults.timeTrendMap[t];
      return trend.count > 0 ? Math.round(trend.sum / trend.count) : 0;
    });
    const errorRates = timeLabels.map(t => {
      const trend = mergedResults.errorTrendMap[t];
      return trend.count > 0 ? Math.round((trend.error / trend.count) * 100 * 100) / 100 : 0;
    });

    // 전체 로그 정렬 (최신순)
    mergedResults.processedLogs.sort((a: any, b: any) => {
      try {
        return new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime();
      } catch (error) {
        return 0;
      }
    });

    // 최종 통계 계산
    const { totalCalls, limitedCalls, totalResponseTime, responseCount, successCount } = mergedResults.stats;
    const avgResponseTime = responseCount > 0 ? Math.round(totalResponseTime / responseCount) : 0;
    const successRate = totalCalls > 0 ? Math.round((successCount / totalCalls) * 100) / 100 : 0;

    const result = {
      endpointStats,
      callsHeatmap: mergedResults.allHourly,
      timeLabels,
      responseTimes,
      errorRates,
      logs: mergedResults.processedLogs.slice(0, 500), // 최대 500개 로그만 반환
      totalCalls,
      limitedCalls,
      avgResponseTime,
      successRate,
      processingTimeMs: Date.now() - startTime,
      implementation: 'Improved Monitoring Processor (app.py pattern)',
      features: [
        '10초 파일 읽기 타임아웃',
        '30초 데이터 캐싱',
        '청크 단위 병렬 처리', 
        '메모리 보호 (최대 10K 로그)',
        '강화된 에러 처리',
        '성능 메트릭 로깅'
      ]
    };

    // app.py 패턴: 결과 캐싱
    this.setCachedData(cacheKey, result);
    
    console.debug(`모니터링 데이터 처리 완료: ${Date.now() - startTime}ms`);
    return result;
  }

  // app.py 패턴: 캐시 정리
  static clearCache(): void {
    this.dataCache.clear();
    console.debug('모니터링 데이터 캐시 정리됨');
  }
}

// GET: app.py 패턴을 적용한 개선된 모니터링 API
export async function GET() {
  const startTime = Date.now();
  
  try {
    console.debug('모니터링 API 요청 시작');
    
    // app.py 패턴: 전체 요청에 대한 타임아웃 적용 (30초)
    const data = await Promise.race([
      ImprovedMonitoringProcessor.processMonitoringData(),
      new Promise((_, reject) => 
        setTimeout(() => reject(new Error('모니터링 데이터 처리 시간 초과')), 30000)
      )
    ]);

    const totalTime = Date.now() - startTime;
    console.debug(`모니터링 API 완료: ${totalTime}ms`);

    return NextResponse.json({
      success: true,
      data: {
        ...data,
        apiProcessingTimeMs: totalTime
      }
    });

  } catch (error) {
    const totalTime = Date.now() - startTime;
    console.error('모니터링 API 오류:', error);
    
    // app.py 패턴: 에러 시에도 상세한 정보 제공
    const errorMessage = error instanceof Error ? error.message : String(error);
    
    return NextResponse.json({ 
      success: false, 
      error: { 
        message: errorMessage,
        processingTimeMs: totalTime,
        implementation: 'Improved Monitoring Processor (app.py pattern - Error)'
      } 
    }, { status: 500 });
  } finally {
    // app.py 패턴: 요청 완료 후 메모리 정리 (선택적)
    if (Math.random() < 0.1) { // 10% 확률로 캐시 정리
      setTimeout(() => {
        try {
          ImprovedMonitoringProcessor.clearCache();
        } catch (error) {
          console.error('캐시 정리 중 오류:', error);
        }
      }, 1000);
    }
  }
} 