import { NextResponse } from 'next/server';
import os from 'os';
import fs from 'fs';
import path from 'path';
import pg from 'pg';

const { Pool } = pg;

function getAppDataPath() {
  switch (process.platform) {
    case 'win32':
      return path.join(process.env.APPDATA || '', 'airun');
    case 'darwin':
      return path.join(os.homedir(), 'Library', 'Application Support', 'airun');
    default:
      return path.join(os.homedir(), '.airun');
  }
}

async function getAllUsageData() {
  try {
    const appDataDir = getAppDataPath();
    const archiveDir = path.join(appDataDir, 'archives');
    let allData = [];
    if (fs.existsSync(archiveDir)) {
      const archiveFiles = await fs.promises.readdir(archiveDir);
      for (const file of archiveFiles) {
        if (file.startsWith('usage_') && file.endsWith('.json')) {
          const archiveContent = await fs.promises.readFile(path.join(archiveDir, file), 'utf8');
          const archiveData = JSON.parse(archiveContent);
          allData.push(...archiveData);
        }
      }
    }
    const usageFilePath = path.join(appDataDir, 'usage.json');
    if (fs.existsSync(usageFilePath)) {
      const fileContent = await fs.promises.readFile(usageFilePath, 'utf8');
      if (fileContent.trim()) {
        const currentData = JSON.parse(fileContent);
        allData.push(...currentData);
      }
    }
    return allData;
  } catch {
    return [];
  }
}

async function getAllLogs() {
  try {
    const appDataDir = getAppDataPath();
    const archiveDir = path.join(appDataDir, 'archives');
    let allLogs = [];
    if (fs.existsSync(archiveDir)) {
      const archiveFiles = await fs.promises.readdir(archiveDir);
      for (const file of archiveFiles) {
        if (file.startsWith('logs_') && file.endsWith('.json')) {
          const archiveContent = await fs.promises.readFile(path.join(archiveDir, file), 'utf8');
          const archiveData = JSON.parse(archiveContent);
          allLogs.push(...archiveData);
        }
      }
    }
    const logsFilePath = path.join(appDataDir, 'logs.json');
    if (fs.existsSync(logsFilePath)) {
      const fileContent = await fs.promises.readFile(logsFilePath, 'utf8');
      if (fileContent.trim()) {
        const currentData = JSON.parse(fileContent);
        allLogs.push(...currentData);
      }
    }
    return allLogs;
  } catch {
    return [];
  }
}

async function getPostgreSQLInfo() {
  const pool = new Pool({
    user: (await getDatabasePool()).options.user,
    host: (await getDatabasePool()).options.host,
    database: (await getDatabasePool()).options.database,
    password: (await getDatabasePool()).options.password,
    port: (await getDatabasePool()).options.port,
    max: 1,
  });

  try {
    const client = await pool.connect();
    
    const connectionQuery = `
      SELECT 
        (SELECT count(*) FROM pg_stat_activity WHERE datname = current_database()) as current_connections,
        (SELECT setting::int FROM pg_settings WHERE name = 'max_connections') as max_connections
    `;
    
    const result = await client.query(connectionQuery);
    const { current_connections, max_connections } = result.rows[0];
    
    client.release();
    await pool.end();
    
    return {
      current_connections: parseInt(current_connections),
      max_connections: parseInt(max_connections),
      connection_usage_percent: Math.round((current_connections / max_connections) * 100)
    };
  } catch (error) {
    console.error('PostgreSQL 정보 조회 오류:', error);
    return null;
  }
}

export async function GET() {
  try {
    const cpus = os.cpus();
    const totalMem = os.totalmem();
    const freeMem = os.freemem();
    const usedMem = totalMem - freeMem;
    const uptime = os.uptime();
    const loadAvg = os.loadavg()[0];
    const cpuUsage = Math.min((loadAvg / cpus.length) * 100, 100);
    const memoryUsage = (usedMem / totalMem) * 100;
    // 디스크 사용률: Node.js 표준만으로는 불가, null 반환
    const diskUsage = null;

    // 실제 데이터 기반 집계
    const allUsage = await getAllUsageData();
    const allLogs = await getAllLogs();
    const now = new Date();
    // 활성 세션: 최근 10분 이내 요청이 있는 userId 수
    const tenMinAgo = new Date(now.getTime() - 10 * 60 * 1000);
    const activeUserSet = new Set();
    for (const u of allUsage) {
      if (u.userId && new Date(u.timestamp) >= tenMinAgo) {
        activeUserSet.add(u.userId);
      }
    }
    const activeSessions = activeUserSet.size;
    // 총 요청 수
    const totalRequests = allUsage.length;
    // 오류율: logs에서 level이 'error'인 비율 (최근 24시간)
    const oneDayAgo = new Date(now.getTime() - 24 * 60 * 60 * 1000);
    const recentLogs = allLogs.filter((l) => new Date(l.timestamp) >= oneDayAgo);
    const errorCount = recentLogs.filter((l) => l.level === 'error').length;
    const errorRate = recentLogs.length > 0 ? (errorCount / recentLogs.length) * 100 : 0;

    // PostgreSQL 연결 정보 가져오기
    const postgresqlInfo = await getPostgreSQLInfo();

    return NextResponse.json({
      success: true,
      data: {
        cpu_usage: cpuUsage,
        memory_usage: memoryUsage,
        disk_usage: diskUsage,
        active_sessions: activeSessions,
        total_requests: totalRequests,
        error_rate: errorRate,
        uptime: String(Math.floor(uptime)),
        postgresql: postgresqlInfo,
      },
    });
  } catch (error) {
    return NextResponse.json({ success: false, error: { message: error instanceof Error ? error.message : String(error) } }, { status: 500 });
  }
} 