import { NextRequest, NextResponse } from 'next/server';
import { spawn } from 'child_process';
import fs from 'fs/promises';
import path from 'path';
import os from 'os';

async function convertToWav(webmPath: string): Promise<string> {
  const wavPath = webmPath.replace(/\.[^/.]+$/, ".wav");
  
  return new Promise((resolve, reject) => {
    const ffmpeg = spawn("ffmpeg", ["-i", webmPath, "-ar", "16000", "-ac", "1", "-c:a", "pcm_s16le", wavPath]);
    ffmpeg.on("close", (code) => {
      if (code !== 0) reject(new Error(`ffmpeg 변환 실패 (코드: ${code})`));
      else resolve(wavPath);
    });
    ffmpeg.on("error", (err) => reject(new Error(`ffmpeg 실행 오류: ${err.message}`)));
  });
}

export async function POST(req: NextRequest) {
  let tempPath: string | null = null;
  let wavPath: string | null = null;

  try {
    const audioBuffer = Buffer.from(await req.arrayBuffer());
    if (!audioBuffer || audioBuffer.length === 0) {
      return NextResponse.json({ error: "오디오 데이터가 없습니다." }, { status: 400 });
    }

    const tempDir = os.tmpdir();
    tempPath = path.join(tempDir, `recording-${Date.now()}.webm`);
    await fs.writeFile(tempPath, audioBuffer);

    wavPath = await convertToWav(tempPath);
    const wavFileBuffer = await fs.readFile(wavPath);

    // Use the standard FormData and Blob classes
    const formData = new FormData();
    const audioBlob = new Blob([wavFileBuffer], { type: 'audio/wav' });
    formData.append('file', audioBlob, 'recording.wav');

    // Whisper 서버로 요청을 보냅니다.
    const whisperServerUrl = 'https://api.hamonize.com/whisper/inference';
    const response = await fetch(whisperServerUrl, {
      method: 'POST',
      body: formData,
      // NOTE: Do NOT set Content-Type header manually when using FormData with fetch.
      // The browser or fetch implementation will set it correctly with the boundary.
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Whisper 서버 오류: ${response.status} - ${errorText}`);
    }

    const result = await response.json();
    return NextResponse.json(result);

  } catch (error) {
    console.error("STT 프록시 처리 중 오류:", error);
    const errorMessage = error instanceof Error ? error.message : "알 수 없는 오류";
    return NextResponse.json({ error: "STT 프록시 실패", details: errorMessage }, { status: 500 });
  } finally {
    // 임시 파일 정리
    if (tempPath) await fs.unlink(tempPath).catch(console.error);
    if (wavPath) await fs.unlink(wavPath).catch(console.error);
  }
}
