'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { getBusinessServerUrl } from '@/config/serverConfig';
import { 
  BarChart3, 
  TrendingUp, 
  Users, 
  FileText, 
  Target,
  Calendar,
  DollarSign,
  Activity,
  Award,
  Clock,
  ArrowUp,
  ArrowDown,
  RefreshCw,
  Download,
  Filter
} from 'lucide-react';

interface DashboardStats {
  announcements: {
    total: number;
    active: number;
    this_month: number;
    change_percentage: number;
  };
  matching: {
    total_matches: number;
    high_score_matches: number;
    avg_score: number;
    score_trend: number;
  };
  proposals: {
    total: number;
    generated_this_month: number;
    success_rate: number;
    avg_generation_time: number;
  };
  budget: {
    total_available: number;
    avg_budget: number;
    largest_opportunity: number;
  };
}

interface RecentActivity {
  id: string;
  type: 'announcement' | 'match' | 'proposal';
  title: string;
  description: string;
  timestamp: string;
  status: 'success' | 'pending' | 'warning';
}

export default function BusinessDashboard() {
  const router = useRouter();
  const [stats, setStats] = useState<DashboardStats | null>(null);
  const [activities, setActivities] = useState<RecentActivity[]>([]);
  const [loading, setLoading] = useState(true);
  const [timeRange, setTimeRange] = useState('7d');

  useEffect(() => {
    fetchDashboardData();
  }, [timeRange]);

  const fetchDashboardData = async () => {
    try {
      setLoading(true);
      const baseUrl = getBusinessServerUrl();
      
      // Fetch dashboard statistics
      const statsResponse = await fetch(`${baseUrl}/dashboard/stats?range=${timeRange}`);
      if (statsResponse.ok) {
        const statsData = await statsResponse.json();
        setStats(statsData);
      }

      // Fetch recent activities
      const activitiesResponse = await fetch(`${baseUrl}/dashboard/activities?limit=10`);
      if (activitiesResponse.ok) {
        const activitiesData = await activitiesResponse.json();
        setActivities(activitiesData);
      }
    } catch (error) {
      console.error('Error fetching dashboard data:', error);
      // Set mock data for development
      setMockData();
    } finally {
      setLoading(false);
    }
  };

  const setMockData = () => {
    setStats({
      announcements: {
        total: 1247,
        active: 892,
        this_month: 156,
        change_percentage: 12.5
      },
      matching: {
        total_matches: 3421,
        high_score_matches: 287,
        avg_score: 78.3,
        score_trend: 5.2
      },
      proposals: {
        total: 145,
        generated_this_month: 23,
        success_rate: 24.8,
        avg_generation_time: 4.2
      },
      budget: {
        total_available: 127500000000,
        avg_budget: 450000000,
        largest_opportunity: 5000000000
      }
    });

    setActivities([
      {
        id: '1',
        type: 'announcement',
        title: '신규 AI 기술개발 공고 등록',
        description: '과학기술정보통신부에서 발표한 신규 AI R&D 과제',
        timestamp: '2025-09-11T10:30:00Z',
        status: 'success'
      },
      {
        id: '2',
        type: 'match',
        title: '높은 적합성 매칭 발견',
        description: '(주)테크이노베이션과 스마트팩토리 과제 - 매칭점수 94점',
        timestamp: '2025-09-11T09:45:00Z',
        status: 'success'
      },
      {
        id: '3',
        type: 'proposal',
        title: '제안서 생성 완료',
        description: '인공지능 기반 제조업 혁신 프로젝트 제안서',
        timestamp: '2025-09-11T08:20:00Z',
        status: 'success'
      },
      {
        id: '4',
        type: 'match',
        title: '매칭 점수 낮음 알림',
        description: '일부 기업 프로필 업데이트 필요',
        timestamp: '2025-09-11T07:15:00Z',
        status: 'warning'
      }
    ]);
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 1000000000) {
      return `${(amount / 1000000000).toFixed(1)}조원`;
    } else if (amount >= 100000000) {
      return `${(amount / 100000000).toFixed(1)}억원`;
    } else if (amount >= 10000) {
      return `${(amount / 10000).toFixed(1)}만원`;
    }
    return `${amount.toLocaleString()}원`;
  };

  const formatPercentage = (value: number) => {
    const isPositive = value >= 0;
    return (
      <span className={`flex items-center text-sm ${isPositive ? 'text-green-600' : 'text-red-600'}`}>
        {isPositive ? <ArrowUp className="w-4 h-4 mr-1" /> : <ArrowDown className="w-4 h-4 mr-1" />}
        {Math.abs(value)}%
      </span>
    );
  };

  const getActivityIcon = (type: string) => {
    switch (type) {
      case 'announcement': return <FileText className="w-5 h-5 text-blue-600" />;
      case 'match': return <Target className="w-5 h-5 text-green-600" />;
      case 'proposal': return <Award className="w-5 h-5 text-purple-600" />;
      default: return <Activity className="w-5 h-5 text-gray-600" />;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'success': return 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200';
      case 'pending': return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200';
      case 'warning': return 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200';
      default: return 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-200';
    }
  };

  if (loading) {
    return (
      <div className="container mx-auto px-6 py-8">
        <div className="flex items-center justify-center min-h-64">
          <RefreshCw className="w-8 h-8 animate-spin text-gray-400" />
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Controls */}
      <div className="flex items-center justify-end mb-8">
        <div className="flex items-center space-x-4">
          <select 
            value={timeRange} 
            onChange={(e) => setTimeRange(e.target.value)}
            className="px-4 py-2 border border-gray-200 rounded-lg dark:border-gray-700 dark:bg-gray-800"
          >
            <option value="7d">최근 7일</option>
            <option value="30d">최근 30일</option>
            <option value="90d">최근 90일</option>
            <option value="1y">최근 1년</option>
          </select>
          
          <button 
            onClick={fetchDashboardData}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            <RefreshCw className="w-4 h-4 mr-2" />
            새로고침
          </button>
          
          <button className="flex items-center px-4 py-2 border border-gray-200 rounded-lg hover:bg-gray-50 dark:border-gray-700 dark:hover:bg-gray-800 transition-colors">
            <Download className="w-4 h-4 mr-2" />
            내보내기
          </button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        {/* Announcements Stats */}
        <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between mb-4">
            <div className="p-2 bg-blue-100 dark:bg-blue-900 rounded-lg">
              <FileText className="w-6 h-6 text-blue-600" />
            </div>
            {formatPercentage(stats?.announcements.change_percentage || 0)}
          </div>
          <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
            {stats?.announcements.total.toLocaleString() || 0}
          </h3>
          <p className="text-sm text-gray-600 dark:text-gray-400">총 사업공고</p>
          <p className="text-xs text-gray-500 mt-1">활성: {stats?.announcements.active.toLocaleString() || 0}건</p>
        </div>

        {/* Matching Stats */}
        <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between mb-4">
            <div className="p-2 bg-green-100 dark:bg-green-900 rounded-lg">
              <Target className="w-6 h-6 text-green-600" />
            </div>
            {formatPercentage(stats?.matching.score_trend || 0)}
          </div>
          <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
            {stats?.matching.avg_score.toFixed(1) || 0}점
          </h3>
          <p className="text-sm text-gray-600 dark:text-gray-400">평균 매칭점수</p>
          <p className="text-xs text-gray-500 mt-1">고득점: {stats?.matching.high_score_matches.toLocaleString() || 0}건</p>
        </div>

        {/* Proposals Stats */}
        <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between mb-4">
            <div className="p-2 bg-purple-100 dark:bg-purple-900 rounded-lg">
              <Award className="w-6 h-6 text-purple-600" />
            </div>
            <span className="text-sm text-gray-600">{stats?.proposals.success_rate.toFixed(1) || 0}% 성공률</span>
          </div>
          <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
            {stats?.proposals.total.toLocaleString() || 0}
          </h3>
          <p className="text-sm text-gray-600 dark:text-gray-400">생성된 제안서</p>
          <p className="text-xs text-gray-500 mt-1">이번 달: {stats?.proposals.generated_this_month || 0}건</p>
        </div>

        {/* Budget Stats */}
        <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
          <div className="flex items-center justify-between mb-4">
            <div className="p-2 bg-orange-100 dark:bg-orange-900 rounded-lg">
              <DollarSign className="w-6 h-6 text-orange-600" />
            </div>
            <TrendingUp className="w-4 h-4 text-green-600" />
          </div>
          <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
            {formatCurrency(stats?.budget.total_available || 0)}
          </h3>
          <p className="text-sm text-gray-600 dark:text-gray-400">총 지원가능 예산</p>
          <p className="text-xs text-gray-500 mt-1">평균: {formatCurrency(stats?.budget.avg_budget || 0)}</p>
        </div>
      </div>

      {/* Main Content Grid */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Recent Activities */}
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
          <div className="p-6 border-b border-gray-200 dark:border-gray-700">
            <h2 className="text-xl font-semibold text-gray-900 dark:text-white">최근 활동</h2>
          </div>
          <div className="p-6">
            <div className="space-y-4">
              {activities.map((activity) => (
                <div key={activity.id} className="flex items-start space-x-3">
                  <div className="flex-shrink-0 mt-1">
                    {getActivityIcon(activity.type)}
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center justify-between">
                      <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                        {activity.title}
                      </h4>
                      <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(activity.status)}`}>
                        {activity.status === 'success' ? '완료' : activity.status === 'pending' ? '진행중' : '주의'}
                      </span>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                      {activity.description}
                    </p>
                    <p className="text-xs text-gray-500 mt-1">
                      {new Date(activity.timestamp).toLocaleString('ko-KR')}
                    </p>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Quick Actions */}
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
          <div className="p-6 border-b border-gray-200 dark:border-gray-700">
            <h2 className="text-xl font-semibold text-gray-900 dark:text-white">빠른 작업</h2>
          </div>
          <div className="p-6">
            <div className="space-y-4">
              <button 
                onClick={() => router.push('/business/announcements')}
                className="w-full flex items-center justify-between p-4 border border-gray-200 dark:border-gray-700 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
              >
                <div className="flex items-center space-x-3">
                  <FileText className="w-5 h-5 text-blue-600" />
                  <span className="text-sm font-medium">사업공고 관리</span>
                </div>
                <span className="text-xs text-gray-500">최신 공고 확인</span>
              </button>

              <button 
                onClick={() => router.push('/business/matching')}
                className="w-full flex items-center justify-between p-4 border border-gray-200 dark:border-gray-700 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
              >
                <div className="flex items-center space-x-3">
                  <Target className="w-5 h-5 text-green-600" />
                  <span className="text-sm font-medium">기업 매칭</span>
                </div>
                <span className="text-xs text-gray-500">적합성 분석</span>
              </button>

              <button 
                onClick={() => router.push('/business/proposals')}
                className="w-full flex items-center justify-between p-4 border border-gray-200 dark:border-gray-700 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
              >
                <div className="flex items-center space-x-3">
                  <Award className="w-5 h-5 text-purple-600" />
                  <span className="text-sm font-medium">제안서 생성</span>
                </div>
                <span className="text-xs text-gray-500">AI 기반 작성</span>
              </button>

              <button className="w-full flex items-center justify-between p-4 border border-gray-200 dark:border-gray-700 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                <div className="flex items-center space-x-3">
                  <BarChart3 className="w-5 h-5 text-orange-600" />
                  <span className="text-sm font-medium">상세 분석 보고서</span>
                </div>
                <span className="text-xs text-gray-500">PDF 다운로드</span>
              </button>
            </div>
          </div>
        </div>
      </div>

      {/* Additional Info */}
      <div className="mt-8 bg-gradient-to-r from-blue-50 to-purple-50 dark:from-blue-900/20 dark:to-purple-900/20 rounded-xl p-6">
        <div className="flex items-center justify-between">
          <div>
            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
              AI 기반 사업 분석 플랫폼
            </h3>
            <p className="text-gray-600 dark:text-gray-400">
              실시간 공고 모니터링, 지능형 매칭, 자동 제안서 생성으로 사업 기회를 놓치지 마세요.
            </p>
          </div>
          <div className="hidden lg:block">
            <div className="flex space-x-2">
              <div className="w-3 h-3 bg-blue-400 rounded-full animate-pulse"></div>
              <div className="w-3 h-3 bg-purple-400 rounded-full animate-pulse" style={{animationDelay: '0.5s'}}></div>
              <div className="w-3 h-3 bg-green-400 rounded-full animate-pulse" style={{animationDelay: '1s'}}></div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}