'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { getBusinessServerUrl } from '@/config/serverConfig';
import { 
  Target, 
  Search, 
  Filter, 
  Star,
  TrendingUp,
  Building2,
  MapPin,
  Calendar,
  DollarSign,
  Users,
  Zap,
  Award,
  RefreshCw,
  ChevronDown,
  ChevronUp,
  Eye,
  FileText,
  ArrowRight,
  Settings,
  Plus,
  X,
  Download
} from 'lucide-react';

interface CompanyProfile {
  id: string;
  name: string;
  business_number: string;
  industry: string[];
  location: string;
  founded_year: number;
  employees: number;
  revenue: number;
  capabilities: {
    rd_capability: number;
    manufacturing_capacity: number;
    marketing_power: number;
    financial_stability: number;
  };
  certifications: string[];
  projects: number;
  created_at: string;
}

interface MatchResult {
  company: CompanyProfile;
  announcement_id: string;
  announcement_title: string;
  score: number;
  breakdown: {
    technical: number;
    business: number;
    financial: number;
    strategic: number;
  };
  reasons: string[];
  compatibility_factors: string[];
  recommendations: string[];
}

export default function BusinessMatching() {
  const router = useRouter();
  const [companies, setCompanies] = useState<CompanyProfile[]>([]);
  const [matches, setMatches] = useState<MatchResult[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedIndustry, setSelectedIndustry] = useState('');
  const [selectedCompany, setSelectedCompany] = useState<string | null>(null);
  const [matchingInProgress, setMatchingInProgress] = useState(false);
  const [expandedMatch, setExpandedMatch] = useState<string | null>(null);

  const industries = [
    '제조업', 'IT/소프트웨어', '바이오/의료', '에너지', '환경',
    '건설', '교통/물류', '금융', '교육', '농업', '기타'
  ];

  useEffect(() => {
    fetchCompanies();
  }, []);

  const fetchCompanies = async () => {
    try {
      setLoading(true);
      const baseUrl = getBusinessServerUrl();
      const response = await fetch(`${baseUrl}/companies`);
      
      if (response.ok) {
        const data = await response.json();
        setCompanies(data);
      } else {
        // Mock data for development
        setMockCompanies();
      }
    } catch (error) {
      console.error('Error fetching companies:', error);
      setMockCompanies();
    } finally {
      setLoading(false);
    }
  };

  const setMockCompanies = () => {
    setCompanies([
      {
        id: 'comp1',
        name: '(주)테크이노베이션',
        business_number: '123-45-67890',
        industry: ['IT/소프트웨어', '제조업'],
        location: '서울특별시',
        founded_year: 2018,
        employees: 45,
        revenue: 8500000000,
        capabilities: {
          rd_capability: 85,
          manufacturing_capacity: 70,
          marketing_power: 65,
          financial_stability: 80
        },
        certifications: ['ISO 9001', 'ISO 14001', 'CMMI Level 3'],
        projects: 12,
        created_at: '2025-09-11T10:00:00Z'
      },
      {
        id: 'comp2',
        name: '그린에너지솔루션',
        business_number: '234-56-78901',
        industry: ['에너지', '환경'],
        location: '경기도',
        founded_year: 2015,
        employees: 78,
        revenue: 15200000000,
        capabilities: {
          rd_capability: 90,
          manufacturing_capacity: 85,
          marketing_power: 75,
          financial_stability: 85
        },
        certifications: ['ISO 50001', '녹색기술인증', 'KC 인증'],
        projects: 24,
        created_at: '2025-09-11T09:30:00Z'
      },
      {
        id: 'comp3',
        name: '바이오메드텍',
        business_number: '345-67-89012',
        industry: ['바이오/의료'],
        location: '대전광역시',
        founded_year: 2020,
        employees: 32,
        revenue: 4200000000,
        capabilities: {
          rd_capability: 95,
          manufacturing_capacity: 60,
          marketing_power: 55,
          financial_stability: 70
        },
        certifications: ['GMP', 'ISO 13485', 'FDA 510(k)'],
        projects: 8,
        created_at: '2025-09-11T08:45:00Z'
      }
    ]);
  };

  const runMatching = async (companyId: string) => {
    setMatchingInProgress(true);
    setSelectedCompany(companyId);
    
    try {
      const baseUrl = getBusinessServerUrl();
      const response = await fetch(`${baseUrl}/matching/run`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ company_id: companyId })
      });
      
      if (response.ok) {
        const data = await response.json();
        setMatches(data.matches);
      } else {
        // Mock matching results
        setMockMatches(companyId);
      }
    } catch (error) {
      console.error('Error running matching:', error);
      setMockMatches(companyId);
    } finally {
      setMatchingInProgress(false);
    }
  };

  const setMockMatches = (companyId: string) => {
    const mockResults: MatchResult[] = [
      {
        company: companies.find(c => c.id === companyId)!,
        announcement_id: 'bizinfo_real_20250911154638_2073',
        announcement_title: '[부산] 2025년 인도네시아 산업기계 전시회 부산관 참가 지원',
        score: 94,
        breakdown: {
          technical: 90,
          business: 95,
          financial: 92,
          strategic: 98
        },
        reasons: [
          '수출 전문 기업으로 해외 전시회 참가 경험 보유',
          '산업기계 분야 기술력 우수',
          '부산 지역 소재로 지역 특화 지원 대상',
          '중소기업 지원 조건 완벽 부합'
        ],
        compatibility_factors: [
          '지역적 적합성 (부산 소재)',
          '업종 일치 (제조업)',
          '기업 규모 적합 (중소기업)',
          '수출 경험 보유'
        ],
        recommendations: [
          '기존 수출 실적을 강조하여 지원서 작성',
          '인도네시아 시장 진출 계획 구체화 필요',
          '현지 파트너십 구축 방안 제시'
        ]
      },
      {
        company: companies.find(c => c.id === companyId)!,
        announcement_id: 'bizinfo_real_20250911154638_2074',
        announcement_title: 'AI 기반 스마트팩토리 구축 지원사업',
        score: 87,
        breakdown: {
          technical: 92,
          business: 85,
          financial: 88,
          strategic: 83
        },
        reasons: [
          'IT 기술력과 제조업 경험의 융합 가능',
          '중소기업 스마트화 지원 대상 적합',
          '충분한 기술개발 역량 보유',
          'Industry 4.0 관련 프로젝트 수행 경험'
        ],
        compatibility_factors: [
          '기술적 역량 우수',
          '제조업 도메인 지식',
          'AI/IT 전문성',
          '규모별 지원 조건 충족'
        ],
        recommendations: [
          'AI 기술 적용 제조업 사례 준비',
          '스마트팩토리 구축 로드맵 제시',
          '기술 파트너십 체계 구성'
        ]
      },
      {
        company: companies.find(c => c.id === companyId)!,
        announcement_id: 'bizinfo_real_20250911154638_2075',
        announcement_title: '중소기업 기술혁신 R&D 지원',
        score: 78,
        breakdown: {
          technical: 85,
          business: 75,
          financial: 80,
          strategic: 72
        },
        reasons: [
          'R&D 전담 조직 운영',
          '기술혁신 역량 보유',
          '과거 R&D 과제 수행 실적',
          '중소기업 지원 대상 해당'
        ],
        compatibility_factors: [
          'R&D 역량',
          '혁신 의지',
          '기술개발 경험',
          '성장 잠재력'
        ],
        recommendations: [
          '핵심 기술 차별화 요소 부각',
          '시장 진출 전략 구체화',
          '기술사업화 계획 수립'
        ]
      }
    ];
    
    setMatches(mockResults);
  };

  const getScoreColor = (score: number) => {
    if (score >= 90) return 'text-green-600 bg-green-100 dark:bg-green-900 dark:text-green-200';
    if (score >= 80) return 'text-blue-600 bg-blue-100 dark:bg-blue-900 dark:text-blue-200';
    if (score >= 70) return 'text-yellow-600 bg-yellow-100 dark:bg-yellow-900 dark:text-yellow-200';
    return 'text-red-600 bg-red-100 dark:bg-red-900 dark:text-red-200';
  };

  const getScoreIcon = (score: number) => {
    if (score >= 90) return <Star className="w-5 h-5 text-yellow-500 fill-current" />;
    if (score >= 80) return <TrendingUp className="w-5 h-5 text-green-600" />;
    if (score >= 70) return <Target className="w-5 h-5 text-blue-600" />;
    return <Zap className="w-5 h-5 text-gray-600" />;
  };

  const filteredCompanies = companies.filter(company => {
    const matchesSearch = company.name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         company.industry?.some(ind => ind?.toLowerCase().includes(searchTerm.toLowerCase()));
    const matchesIndustry = !selectedIndustry || company.industry?.includes(selectedIndustry);
    return matchesSearch && matchesIndustry;
  });

  if (loading) {
    return (
      <div className="container mx-auto px-6 py-8">
        <div className="flex items-center justify-center min-h-64">
          <RefreshCw className="w-8 h-8 animate-spin text-gray-400" />
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* Company List */}
        <div className="lg:col-span-1">
          <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
            <div className="p-6 border-b border-gray-200 dark:border-gray-700">
              <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-4">기업 목록</h2>
              
              {/* Search and Filter */}
              <div className="space-y-3">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                  <input
                    type="text"
                    placeholder="기업명 또는 업종 검색..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full pl-10 pr-4 py-2 border border-gray-200 rounded-lg dark:border-gray-700 dark:bg-gray-900"
                  />
                </div>
                
                <select
                  value={selectedIndustry}
                  onChange={(e) => setSelectedIndustry(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-200 rounded-lg dark:border-gray-700 dark:bg-gray-900"
                >
                  <option value="">전체 업종</option>
                  {industries.map(industry => (
                    <option key={industry} value={industry}>{industry}</option>
                  ))}
                </select>
              </div>
            </div>
            
            <div className="max-h-96 overflow-y-auto">
              {filteredCompanies.map((company) => (
                <div 
                  key={company.id}
                  className={`p-4 border-b border-gray-100 dark:border-gray-700 cursor-pointer transition-colors ${
                    selectedCompany === company.id 
                      ? 'bg-blue-50 dark:bg-blue-900/20' 
                      : 'hover:bg-gray-50 dark:hover:bg-gray-700'
                  }`}
                  onClick={() => runMatching(company.id)}
                >
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <h3 className="font-medium text-gray-900 dark:text-white">{company.name}</h3>
                      <div className="flex items-center text-sm text-gray-600 dark:text-gray-400 mt-1">
                        <Building2 className="w-4 h-4 mr-1" />
                        {company.industry.join(', ')}
                      </div>
                      <div className="flex items-center text-sm text-gray-500 mt-1">
                        <MapPin className="w-4 h-4 mr-1" />
                        {company.location}
                      </div>
                      <div className="flex items-center text-sm text-gray-500 mt-1">
                        <Users className="w-4 h-4 mr-1" />
                        {company.employees}명
                      </div>
                    </div>
                    {matchingInProgress && selectedCompany === company.id && (
                      <RefreshCw className="w-4 h-4 animate-spin text-blue-600" />
                    )}
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Matching Results */}
        <div className="lg:col-span-2">
          {matches.length === 0 ? (
            <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-8">
              <div className="text-center">
                <Target className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">매칭 결과가 없습니다</h3>
                <p className="text-gray-600 dark:text-gray-400">
                  왼쪽에서 기업을 선택하여 적합한 사업공고를 찾아보세요.
                </p>
              </div>
            </div>
          ) : (
            <div className="space-y-4">
              <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 p-6">
                <h2 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">매칭 결과</h2>
                <p className="text-gray-600 dark:text-gray-400">
                  {matches[0]?.company.name}에 적합한 사업공고 {matches.length}건을 찾았습니다.
                </p>
              </div>

              {matches.map((match, index) => (
                <div key={index} className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
                  <div className="p-6">
                    <div className="flex items-start justify-between mb-4">
                      <div className="flex-1">
                        <div className="flex items-center space-x-3 mb-2">
                          {getScoreIcon(match.score)}
                          <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                            {match.announcement_title}
                          </h3>
                        </div>
                        <div className="flex items-center space-x-4">
                          <span className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium ${getScoreColor(match.score)}`}>
                            매칭점수: {match.score}점
                          </span>
                          <button
                            onClick={() => router.push(`/business/announcements/${match.announcement_id}`)}
                            className="text-blue-600 hover:text-blue-800 text-sm font-medium flex items-center"
                          >
                            <Eye className="w-4 h-4 mr-1" />
                            공고 상세보기
                          </button>
                        </div>
                      </div>
                    </div>

                    {/* Score Breakdown */}
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
                      <div className="text-center">
                        <div className="text-2xl font-bold text-blue-600">{match.breakdown.technical}</div>
                        <div className="text-sm text-gray-600 dark:text-gray-400">기술적 적합성</div>
                      </div>
                      <div className="text-center">
                        <div className="text-2xl font-bold text-green-600">{match.breakdown.business}</div>
                        <div className="text-sm text-gray-600 dark:text-gray-400">사업적 적합성</div>
                      </div>
                      <div className="text-center">
                        <div className="text-2xl font-bold text-purple-600">{match.breakdown.financial}</div>
                        <div className="text-sm text-gray-600 dark:text-gray-400">재무적 적합성</div>
                      </div>
                      <div className="text-center">
                        <div className="text-2xl font-bold text-orange-600">{match.breakdown.strategic}</div>
                        <div className="text-sm text-gray-600 dark:text-gray-400">전략적 적합성</div>
                      </div>
                    </div>

                    {/* Toggle Details */}
                    <button
                      onClick={() => setExpandedMatch(expandedMatch === `${index}` ? null : `${index}`)}
                      className="flex items-center justify-center w-full py-2 text-sm text-gray-600 hover:text-gray-900 dark:text-gray-400 dark:hover:text-white transition-colors border-t border-gray-200 dark:border-gray-700"
                    >
                      {expandedMatch === `${index}` ? '간략히 보기' : '상세 분석 보기'}
                      {expandedMatch === `${index}` ? 
                        <ChevronUp className="w-4 h-4 ml-1" /> : 
                        <ChevronDown className="w-4 h-4 ml-1" />
                      }
                    </button>

                    {/* Expanded Details */}
                    {expandedMatch === `${index}` && (
                      <div className="pt-4 border-t border-gray-200 dark:border-gray-700 mt-4">
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                          {/* Reasons */}
                          <div>
                            <h4 className="font-medium text-gray-900 dark:text-white mb-3">매칭 근거</h4>
                            <ul className="space-y-2">
                              {match.reasons.map((reason, i) => (
                                <li key={i} className="flex items-start space-x-2">
                                  <div className="w-1.5 h-1.5 bg-green-500 rounded-full mt-2"></div>
                                  <span className="text-sm text-gray-600 dark:text-gray-400">{reason}</span>
                                </li>
                              ))}
                            </ul>
                          </div>

                          {/* Compatibility Factors */}
                          <div>
                            <h4 className="font-medium text-gray-900 dark:text-white mb-3">적합성 요소</h4>
                            <div className="space-y-2">
                              {match.compatibility_factors.map((factor, i) => (
                                <span key={i} className="inline-block bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200 text-xs px-2 py-1 rounded-full mr-2 mb-2">
                                  {factor}
                                </span>
                              ))}
                            </div>
                          </div>

                          {/* Recommendations */}
                          <div>
                            <h4 className="font-medium text-gray-900 dark:text-white mb-3">개선 제안</h4>
                            <ul className="space-y-2">
                              {match.recommendations.map((rec, i) => (
                                <li key={i} className="flex items-start space-x-2">
                                  <div className="w-1.5 h-1.5 bg-orange-500 rounded-full mt-2"></div>
                                  <span className="text-sm text-gray-600 dark:text-gray-400">{rec}</span>
                                </li>
                              ))}
                            </ul>
                          </div>
                        </div>

                        {/* Action Buttons */}
                        <div className="flex space-x-3 mt-6 pt-4 border-t border-gray-200 dark:border-gray-700">
                          <button 
                            onClick={() => router.push(`/business/proposals?announcement=${match.announcement_id}&company=${match.company.id}`)}
                            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                          >
                            <FileText className="w-4 h-4 mr-2" />
                            제안서 생성
                          </button>
                          <button className="flex items-center px-4 py-2 border border-gray-200 rounded-lg hover:bg-gray-50 dark:border-gray-700 dark:hover:bg-gray-700 transition-colors">
                            <Download className="w-4 h-4 mr-2" />
                            분석 보고서
                          </button>
                        </div>
                      </div>
                    )}
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
}