'use client';

import React, { useState, useEffect } from 'react';
import { useLanguage } from '@/contexts/LanguageContext';

interface DocumentPreviewModalProps {
  isOpen: boolean;
  onClose: () => void;
  fileName: string;
  userId: string;
}

export const DocumentPreviewModal: React.FC<DocumentPreviewModalProps> = ({
  isOpen,
  onClose,
  fileName,
  userId
}) => {
  const { t } = useLanguage();
  const [content, setContent] = useState<string>('');
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [metadata, setMetadata] = useState<any>(null);

  useEffect(() => {
    if (isOpen && fileName && userId) {
      fetchDocumentContent();
    }
  }, [isOpen, fileName, userId]);

  // 파일 삭제 이벤트 리스너 추가
  useEffect(() => {
    const handleFileDeleted = (event: CustomEvent) => {
      const { fileName: deletedFileName } = event.detail;
      // 현재 미리보기 중인 파일이 삭제된 경우 모달 닫기
      if (isOpen && fileName === deletedFileName) {
        onClose();
      }
    };

    window.addEventListener('fileDeleted', handleFileDeleted as EventListener);
    
    return () => {
      window.removeEventListener('fileDeleted', handleFileDeleted as EventListener);
    };
  }, [isOpen, fileName, onClose]);

  const fetchDocumentContent = async () => {
    setIsLoading(true);
    setError(null);
    try {
      const response = await fetch(`/api/rag/preview?filename=${encodeURIComponent(fileName)}&userId=${encodeURIComponent(userId)}`, {
        method: 'GET',
        credentials: 'include'
      });

      if (!response.ok) {
        throw new Error('문서를 불러올 수 없습니다.');
      }

      const data = await response.json();
      if (data.success) {
        setContent(data.data.content || '');
        setMetadata(data.data.metadata || null);
      } else {
        throw new Error(data.error?.message || '문서를 불러올 수 없습니다.');
      }
    } catch (err) {
      console.error('문서 미리보기 로드 실패:', err);
      setError(err instanceof Error ? err.message : '문서를 불러올 수 없습니다.');
    } finally {
      setIsLoading(false);
    }
  };

  if (!isOpen) return null;

  const getFileIcon = (fileName: string) => {
    const ext = fileName.split('.').pop()?.toLowerCase();
    switch (ext) {
      case 'pdf':
        return '📄';
      case 'doc':
      case 'docx':
        return '📝';
      case 'txt':
        return '📃';
      case 'md':
        return '📑';
      case 'csv':
      case 'xlsx':
      case 'xls':
        return '📊';
      case 'ppt':
      case 'pptx':
        return '📽️';
      case 'jpg':
      case 'jpeg':
      case 'png':
      case 'gif':
        return '🖼️';
      default:
        return '📎';
    }
  };

  const formatFileSize = (bytes?: number) => {
    if (!bytes) return 'N/A';
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    if (bytes === 0) return '0 Bytes';
    const i = Math.floor(Math.log(bytes) / Math.log(1024));
    return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
  };

  const formatDate = (dateString?: string) => {
    if (!dateString) return 'N/A';
    try {
      return new Date(dateString).toLocaleDateString('ko-KR', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
      });
    } catch {
      return 'N/A';
    }
  };

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50" onClick={onClose}>
      <div 
        className="relative rounded-lg shadow-xl w-11/12 max-w-4xl h-2/5 flex flex-col"
        style={{ backgroundColor: 'var(--card-bg)' }}
        onClick={(e) => e.stopPropagation()}
      >
        {/* Header */}
        <div className="flex items-center justify-between p-4 border-b dark:border-gray-700">
          <div className="flex items-center space-x-3">
            <span className="text-2xl">{getFileIcon(fileName)}</span>
            <div>
              <h2 className="text-lg font-semibold text-gray-900 dark:text-white truncate max-w-md">
                {fileName}
              </h2>
              {metadata && (
                <div className="flex items-center space-x-4 text-sm text-gray-500 dark:text-gray-400">
                  {metadata.size && (
                    <>
                      <span>크기: {formatFileSize(metadata.size)}</span>
                      {(metadata.modified || metadata.chunks) && <span>•</span>}
                    </>
                  )}
                  {metadata.modified && (
                    <>
                      <span>수정일: {formatDate(metadata.modified)}</span>
                      {metadata.chunks && <span>•</span>}
                    </>
                  )}
                  {metadata.chunks && (
                    <span>청크: {metadata.chunks}개</span>
                  )}
                </div>
              )}
            </div>
          </div>
          <button
            onClick={onClose}
            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors"
            aria-label="닫기"
          >
            <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
            </svg>
          </button>
        </div>

        {/* Content */}
        <div className="flex-1 overflow-y-auto p-6 min-h-0">
          {isLoading ? (
            <div className="flex items-center justify-center h-full">
              <div className="text-center">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
                <p className="text-gray-600 dark:text-gray-400">문서를 불러오는 중...</p>
              </div>
            </div>
          ) : error ? (
            <div className="flex items-center justify-center h-full">
              <div className="text-center">
                <svg className="w-16 h-16 text-red-500 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <p className="text-red-600 dark:text-red-400 mb-2">문서 로드 실패</p>
                <p className="text-sm text-gray-600 dark:text-gray-400">{error}</p>
                <button
                  onClick={fetchDocumentContent}
                  className="mt-4 px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600 transition-colors"
                >
                  다시 시도
                </button>
              </div>
            </div>
          ) : content ? (
            <div className="h-full">
              <div className="rounded-lg p-4 h-full overflow-y-auto" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
                <pre className="whitespace-pre-wrap text-sm font-mono" style={{ color: 'var(--text-primary)' }}>
                  {content.length > 5000 ? `${content.substring(0, 5000)}...

[문서가 너무 길어 처음 5000자만 표시됩니다]` : content}
                </pre>
              </div>
            </div>
          ) : (
            <div className="flex items-center justify-center h-full">
              <p className="text-gray-500 dark:text-gray-400">문서 내용이 없습니다.</p>
            </div>
          )}
        </div>


      </div>
    </div>
  );
};