'use client';

import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Slider } from '@/components/ui/slider';
import { Switch } from '@/components/ui/switch';
import { Progress } from '@/components/ui/progress';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Play, Square, Settings, Brain, Zap } from 'lucide-react';

interface TrainingJob {
  id: string;
  modelName: string;
  dataset: string;
  status: 'running' | 'completed' | 'failed' | 'paused';
  progress: number;
  startTime: string;
  estimatedTime: string;
  method: 'full' | 'lora';
}

export default function LLMTrainingTab() {
  const [trainingJobs, setTrainingJobs] = useState<TrainingJob[]>([]);
  const [availableDatasets, setAvailableDatasets] = useState<string[]>([]);
  const [showTrainingModal, setShowTrainingModal] = useState(false);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    loadTrainingJobs();
    loadAvailableDatasets();
  }, []);

  const loadTrainingJobs = async () => {
    try {
      const response = await fetch('/api/llm/training');
      const data = await response.json();
      if (data.success) {
        setTrainingJobs(data.jobs || []);
      }
    } catch (error) {
      console.error('학습 작업 로드 실패:', error);
    }
  };

  const loadAvailableDatasets = async () => {
    try {
      const response = await fetch('/api/llm/datasets');
      const data = await response.json();
      if (data.success) {
        setAvailableDatasets(data.datasets?.map((d: any) => d.name) || []);
      }
    } catch (error) {
      console.error('데이터셋 로드 실패:', error);
    }
  };

  const handleStartTraining = () => {
    setShowTrainingModal(true);
  };

  const handleStopTraining = async (jobId: string) => {
    try {
      await fetch(`/api/llm/training/${jobId}/stop`, { method: 'POST' });
      loadTrainingJobs();
    } catch (error) {
      console.error('학습 중단 실패:', error);
    }
  };

  return (
    <div className="space-y-6">
      {/* 학습 설정 섹션 */}
      <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
        <div className="flex justify-between items-center mb-4">
          <h4 className="text-base font-semibold text-[var(--text-primary)]">
            ⚙️ 모델 학습 설정
          </h4>
          <Button onClick={handleStartTraining} className="flex items-center gap-2">
            <Play className="h-4 w-4" />
            새 학습 시작
          </Button>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4 border border-blue-200 dark:border-blue-700">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-blue-900 dark:text-blue-200">활성 학습</p>
                <p className="text-2xl font-bold text-blue-700 dark:text-blue-300">
                  {trainingJobs.filter(job => job.status === 'running').length}
                </p>
              </div>
              <span className="text-3xl">🏃</span>
            </div>
            <p className="text-xs text-blue-600 dark:text-blue-400 mt-2">진행 중인 작업</p>
          </div>

          <div className="bg-green-50 dark:bg-green-900/20 rounded-lg p-4 border border-green-200 dark:border-green-700">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-green-900 dark:text-green-200">완료된 학습</p>
                <p className="text-2xl font-bold text-green-700 dark:text-green-300">
                  {trainingJobs.filter(job => job.status === 'completed').length}
                </p>
              </div>
              <span className="text-3xl">✅</span>
            </div>
            <p className="text-xs text-green-600 dark:text-green-400 mt-2">성공적으로 완료</p>
          </div>

          <div className="bg-red-50 dark:bg-red-900/20 rounded-lg p-4 border border-red-200 dark:border-red-700">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-red-900 dark:text-red-200">실패한 학습</p>
                <p className="text-2xl font-bold text-red-700 dark:text-red-300">
                  {trainingJobs.filter(job => job.status === 'failed').length}
                </p>
              </div>
              <span className="text-3xl">❌</span>
            </div>
            <p className="text-xs text-red-600 dark:text-red-400 mt-2">오류로 인한 실패</p>
          </div>
        </div>
      </div>

      {/* 학습 작업 목록 섹션 */}
      <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
        <h4 className="text-base font-semibold text-[var(--text-primary)] mb-4">
          📊 학습 작업 목록
        </h4>
        <div>
          {trainingJobs.length === 0 ? (
            <div className="text-center py-8 text-[var(--text-muted)]">
              <Brain className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>진행 중인 학습 작업이 없습니다.</p>
              <p className="text-sm">새 학습을 시작해보세요.</p>
            </div>
          ) : (
            <div className="space-y-4">
              {trainingJobs.map((job) => (
                <div key={job.id} className="border rounded-lg p-4">
                  <div className="flex justify-between items-start mb-3">
                    <div>
                      <h3 className="font-medium">{job.modelName}</h3>
                      <p className="text-sm text-[var(--text-secondary)]">
                        데이터셋: {job.dataset} • 방법: {job.method === 'lora' ? 'LoRA' : '풀 파인튜닝'}
                      </p>
                    </div>
                    <div className="flex items-center gap-2">
                      <span className={`px-2 py-1 text-xs rounded-full ${
                        job.status === 'running' ? 'bg-blue-100 text-blue-800' :
                        job.status === 'completed' ? 'bg-green-100 text-green-800' :
                        job.status === 'failed' ? 'bg-red-100 text-red-800' :
                        'bg-gray-100 text-gray-800'
                      }`}>
                        {job.status === 'running' ? '진행 중' :
                         job.status === 'completed' ? '완료' :
                         job.status === 'failed' ? '실패' : '일시정지'}
                      </span>
                      {job.status === 'running' && (
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleStopTraining(job.id)}
                          className="text-red-600 hover:text-red-800"
                        >
                          <Square className="h-4 w-4" />
                        </Button>
                      )}
                    </div>
                  </div>
                  
                  {job.status === 'running' && (
                    <div className="space-y-2">
                      <div className="flex justify-between text-sm">
                        <span>진행률</span>
                        <span>{job.progress}%</span>
                      </div>
                      <Progress value={job.progress} className="w-full" />
                      <div className="text-xs text-[var(--text-muted)]">
                        시작 시간: {job.startTime} • 예상 완료: {job.estimatedTime}
                      </div>
                    </div>
                  )}
                </div>
              ))}
            </div>
          )}
        </div>
      </div>

      <Dialog open={showTrainingModal} onOpenChange={setShowTrainingModal}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>새 LLM 학습 시작</DialogTitle>
          </DialogHeader>
          <TrainingConfigForm 
            datasets={availableDatasets}
            onClose={() => setShowTrainingModal(false)} 
            onSuccess={() => {
              loadTrainingJobs();
              setShowTrainingModal(false);
            }} 
          />
        </DialogContent>
      </Dialog>
    </div>
  );
}

interface TrainingConfigFormProps {
  datasets: string[];
  onClose: () => void;
  onSuccess: () => void;
}

function TrainingConfigForm({ datasets, onClose, onSuccess }: TrainingConfigFormProps) {
  const [config, setConfig] = useState({
    modelName: '',
    baseModel: 'microsoft/DialoGPT-small',
    dataset: '',
    method: 'lora' as 'full' | 'lora',
    epochs: 3,
    batchSize: 4,
    learningRate: 0.0002,
    useGradientCheckpointing: true,
    useLoRA: true,
    loraR: 16,
    loraAlpha: 32,
    loraDropout: 0.1
  });
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const response = await fetch('/api/llm/training', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(config)
      });

      const data = await response.json();
      if (data.success) {
        onSuccess();
      }
    } catch (error) {
      console.error('학습 시작 실패:', error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div>
          <Label htmlFor="modelName">모델 이름</Label>
          <Input
            id="modelName"
            value={config.modelName}
            onChange={(e) => setConfig(prev => ({ ...prev, modelName: e.target.value }))}
            placeholder="my-custom-model"
            required
          />
        </div>

        <div>
          <Label htmlFor="baseModel">베이스 모델</Label>
          <Select value={config.baseModel} onValueChange={(value) => setConfig(prev => ({ ...prev, baseModel: value }))}>
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="microsoft/DialoGPT-small">DialoGPT-small</SelectItem>
              <SelectItem value="microsoft/DialoGPT-medium">DialoGPT-medium</SelectItem>
              <SelectItem value="gpt2">GPT-2</SelectItem>
              <SelectItem value="distilgpt2">DistilGPT-2</SelectItem>
            </SelectContent>
          </Select>
        </div>

        <div>
          <Label htmlFor="dataset">데이터셋</Label>
          <Select value={config.dataset} onValueChange={(value) => setConfig(prev => ({ ...prev, dataset: value }))}>
            <SelectTrigger>
              <SelectValue placeholder="데이터셋 선택" />
            </SelectTrigger>
            <SelectContent>
              {datasets.map(dataset => (
                <SelectItem key={dataset} value={dataset}>{dataset}</SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div>
          <Label htmlFor="method">학습 방법</Label>
          <Select value={config.method} onValueChange={(value: 'full' | 'lora') => setConfig(prev => ({ ...prev, method: value }))}>
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="lora">
                <div className="flex items-center gap-2">
                  <Zap className="h-4 w-4" />
                  LoRA (효율적)
                </div>
              </SelectItem>
              <SelectItem value="full">
                <div className="flex items-center gap-2">
                  <Brain className="h-4 w-4" />
                  풀 파인튜닝 (완전)
                </div>
              </SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div>
          <Label>에포크 수: {config.epochs}</Label>
          <Slider
            value={[config.epochs]}
            onValueChange={([value]) => setConfig(prev => ({ ...prev, epochs: value }))}
            max={10}
            min={1}
            step={1}
            className="mt-2"
          />
        </div>

        <div>
          <Label>배치 크기: {config.batchSize}</Label>
          <Slider
            value={[config.batchSize]}
            onValueChange={([value]) => setConfig(prev => ({ ...prev, batchSize: value }))}
            max={16}
            min={1}
            step={1}
            className="mt-2"
          />
        </div>

        <div>
          <Label>학습률: {config.learningRate}</Label>
          <Slider
            value={[config.learningRate * 10000]}
            onValueChange={([value]) => setConfig(prev => ({ ...prev, learningRate: value / 10000 }))}
            max={10}
            min={1}
            step={1}
            className="mt-2"
          />
        </div>
      </div>

      {config.method === 'lora' && (
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">LoRA 설정</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div>
                <Label>LoRA R: {config.loraR}</Label>
                <Slider
                  value={[config.loraR]}
                  onValueChange={([value]) => setConfig(prev => ({ ...prev, loraR: value }))}
                  max={64}
                  min={4}
                  step={4}
                  className="mt-2"
                />
              </div>

              <div>
                <Label>LoRA Alpha: {config.loraAlpha}</Label>
                <Slider
                  value={[config.loraAlpha]}
                  onValueChange={([value]) => setConfig(prev => ({ ...prev, loraAlpha: value }))}
                  max={128}
                  min={8}
                  step={8}
                  className="mt-2"
                />
              </div>

              <div>
                <Label>드롭아웃: {config.loraDropout}</Label>
                <Slider
                  value={[config.loraDropout * 100]}
                  onValueChange={([value]) => setConfig(prev => ({ ...prev, loraDropout: value / 100 }))}
                  max={50}
                  min={0}
                  step={5}
                  className="mt-2"
                />
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      <div className="flex items-center space-x-2">
        <Switch
          checked={config.useGradientCheckpointing}
          onCheckedChange={(checked) => setConfig(prev => ({ ...prev, useGradientCheckpointing: checked }))}
        />
        <Label>그래디언트 체크포인팅 (메모리 절약)</Label>
      </div>

      <div className="flex justify-end gap-2 pt-4">
        <Button type="button" variant="outline" onClick={onClose}>취소</Button>
        <Button type="submit" disabled={loading || !config.dataset}>
          {loading ? '시작 중...' : '학습 시작'}
        </Button>
      </div>
    </form>
  );
}